;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER -*-

;;; ChangeLog:
;;;
;;; 12 Dec 88  Jamie Zawinski    Created.

;;;
;;; This code lets you install arbitrary bitmaps into fonts, creating the
;;; fonts if necessary.  A use of this is to replace the TI logo on bootup
;;; with a logo of your own.  Here's how:
;;;
;;;  o  Use this code to make a font which has your logo in #\T.
;;;  o  Type (MAKUNBOUND 'SYS::BITBLT-TI-LOGO)
;;;  o  Type (setq FONTS::TI-LOGO <your-new-font>)
;;;  o  Do a disk-save.
;;;
;;; Of course, you could also probably just put your bitmap in the 
;;; variable SYS::BITBLT-TI-LOGO, but I didn't realize that until after
;;; I had written this code.  But this code is a good example of how to
;;; grovel around inside of Font Descriptor structures.
;;;



(defun font-of-bitmap (font-name char-code bitmap x y w h)
  "  This is for storing arbitrary bitmaps into fonts.
  If the named font does not exist, it is created.
  The portion of BITMAP specified by XYWH is stored into the CHAR-CODE index
  of the font.  The font's Line-Spacing and Baseline are adjusted accordingly.
  
  You can save this font to a file by using the Font Editor.
  If you use the Font Editor to look at or touch up the character you added, be careful -
  The Font Editor caches some information that this function modifies.  Consequently,
  this function and the Font Editor can undo each others' work.

  It is a good idea to not look at this font in the font editor until you want to save it."
  
  (check-type font-name (and symbol (not null)))
  (check-type char-code (or fixnum string-char))
  (check-type bitmap (array bit 2))
  (setq char-code (char-int char-code))
  (let* ((fd (fed:font-get-fd font-name))	; Lookup or create a Font Descriptor.
	 (cd (aref fd char-code)))		; Get the Char Descriptor, if it exists.
    (unless cd					; Create it and store it, otherwise.
      (setq cd (fed:make-char-descriptor :make-array (:type ART-4B :dimensions '(11 7))
					 :cd-char-width 7
					 :cd-char-left-kern 0))
      (fed:fd-store-cd fd cd char-code))
    ;; Resize the Char Descriptor to fit the new char.
    (adjust-array cd (list (- w x) (- h y)))
    (setf (fed:cd-char-width cd) (- w x))
    ;; Copy the given bitmap into the Char Descriptor.
    ;; We must copy iteratively, because we can't blit a 1bit image into a 4bit image.
    (dotimes (i (- w x))
      (dotimes (j (- h y))
	(setf (aref cd i j) (aref bitmap (+ x i) (+ y j)))))
    ;; Adjust the line-spacing and baseline of the FD to be the MAX of what they are
    ;; now, and what they must be to display this char.
    (setf (fed:fd-line-spacing fd)
	  (max (fed:fd-line-spacing fd) (- h y)))
    (setf (fed:fd-baseline fd)
	  (max (fed:fd-baseline fd) (- h y)))
    ;; Turn the Font Descriptor into a real font.
    (set font-name (fed:font-descriptor-into-font fd))
    (values font-name fd cd)))




;;; Make a font from a rectangle selected from a Raster file.
;;; Requires the Andrew Raster reader.

(defun capture-logo-font-from-raster (font-name pathname)
  (user:read-and-show-raster-file pathname)
  (multiple-value-bind (x y x2 y2) (tv:mouse-specify-rectangle nil nil nil nil tv:selected-window)
    (let* ((w (- x2 x))
	   (h (- y2 y))
	   (bitmap (tv:sheet-screen-array tv:selected-window))
	   (array (make-array (list h w) :element-type 'bit)))
      (dotimes (i w)
	(dotimes (j h)
	  (let* ((val (aref bitmap (+ y j) (+ x i))))
	    (setf (aref array j i) val)
	    )))
      (let* ((font (font-of-bitmap font-name #\T array 0 0 h w)))
	(send tv:selected-window :clear-screen)
	(tv:sheet-string-out-explicit tv:selected-window "T" 200 200 nil font tv:alu-seta)
	font))))



;;; Make a font from a rectangle selected from an IFF file.
;;; Requires the IFF reader.

(eval-when (load eval compile) (unless (find-package "IFF") (make-package "IFF")))


(defun capture-logo-font-from-ilbm (font-name ilbm)
  (iff:show ilbm)
  (multiple-value-bind (x y x2 y2) (tv:mouse-specify-rectangle nil nil nil nil tv:selected-window)
    (let* ((w (- x2 x))
	   (h (- y2 y))
	   (body (iff:ilbm-body ilbm))
	   (bitmap (iff:body-body-bitmap body))
	   (array (make-array (list h w) :element-type (array-element-type bitmap))))
      (dotimes (i w)
	(dotimes (j h)
	  (let* ((val (aref bitmap (+ y j) (+ x i))))
	    (setf (aref array j i) val)
	    )))
      (let* ((font (font-of-bitmap font-name #\T array 0 0 h w)))
	(send tv:selected-window :clear-screen)
	(tv:sheet-string-out-explicit tv:selected-window "T" 200 200 nil font tv:alu-seta)
	font))))
