;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:NET; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B CPTFONTBI HL12BI) -*-

;1;; File "3FINGER*".*
;1;; Various functionality enhancements to the Explorer finger server.*
;1;;*
;1;;   Fingering 2<userid>@some-explorer* will print a bunch of information about the given user; this info*
;1;;   is gleaned from the 5:user* class of the namespace.  If 2<userid>* is not the user ID of someone,*
;1;;   then a guess is made, based on a phonetic interpretation of the given ID and the user IDs and*
;1;;   real names of the people in the namespace.  If only one user ID matches this, then info is printed*
;1;;   for that user as usual; if more than one match is found, then a list of the possible matches is*
;1;;   printed, so the fingerer can try again.  The 2<userid>* is matched against the user IDs, and the first*
;1;;   and last names; if it contains a period, it is interpreted as a firstname/lastname pair, and compared*
;1;;   against both.  For example:*
;1;;   * 	3(finger "jaime.zuhwinsky")*	1==>  2jwz (Jamie Zawinski)**
;1;;*	1  because 2"jaime"* matches 2"Jamie"* and 2"zuhwinsky"* matches 2"Zawinski"**
;1;;*	3(finger "peter")*		1==>  2norvig (Peter Norvig)* and 2gator (John Doe)**
;1;;*	1  because 2"peter"* matches both 2"Peter"* (a first name) and 2"gator"* (a user ID).*
;1;;*
;1;;   Fingering 7/W2@some-explorer** will print a bunch of information about the currently logged in user.*
;1;;*
;1;;   Mailing list names may be fingered as well; fingering one prints its documentation, and lists*
;1;;*  1its members.*
;1;;*
;1;;   If there exists a file called 2some-explorer5: *userid5; plan.text#>**, this file will be printed after the*
;1;;   automatically generated user info.  It is possible to override the location of this plan file by*
;1;;   adding a 5:plan-file* attribute to the 5:user* class of the namespace; that way, you can have your*
;1;;   plan file on one machine, but have all the Explorers know where to look for it.*
;1;;   *
;1;;   If the variable 5net:*notify-when-fingered** is set to T, then a notification will be sent whenever*
;1;;   someone fingers the logged in user; this will enable you to gauge your popularity.*
;1;;*
;1;;   If the variable 5net:*finger-hook** is non-nil, it is a user-supplied function which will be called*
;1;;   before anything is printed, so you can have, in effect, a dynamic plan file; you could, for*
;1;;   example, print a different zippy quote each time you are fingered.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   28 Jul 88*	1Eric Karlson*	1Created.*
;1;;  10 Aug 89*	1Jamie Zawinski*	1Modified this to know about plan files, and to make the being-fingered-notification switchable.*
;1;;  14 Aug 89*	1Jamie Zawinski *	1Modified ``snoop'' mode to say who is being fingered as well.*
;1;;  16 Aug 89*	1Jamie Zawinski *	1Added the User Profile stuff from Eric's version of this file.*
;1;;  17 Aug 89*	1Jamie Zawinski *	1Made the displaying of the plan file be more robust in the cases where the plan file can't be accessed.*
;1;;  23 Aug 89*	1Jamie Zawinski *	1Made the user-info also say where mail is forwarded.*
;1;; * 115 Sep 89*	1Jamie Zawinski *	1Oops, show-user-info was always thinking that "fingerd" was logged on...*
;1;; * 122 Sep 89*	1Jamie Zawinski *	1Made 2*notify-when-fingered** take 2T*, 2nil*, or 2:always* so that one can turn off notifications*
;1;;*				1 for Term-F responses, while still receiving notifications for more specific ones.*
;1;;*  13 Oct 89*	1Jamie Zawinski *	1Added a better definition of 2user-activity-string*.  The old one was just silly.*
;1;;*  16 Oct 89*	1Jamie Zawinski *	1Found another situation where the fingerd would prompt console for a password...  It 2should* be *
;1;;*				1possible to handle that with a condition-handler, but noooo....*
;1;;*  18 Nov 89*	1Jamie Zawinski *	1Added more general code for tracking down the originating host.*
;1;;* 129 Nov 89*	1Jamie Zawinski *	1Modularized things some.  Added phonetic matching.*
;1;;* 113 Dec 89*	1Jamie Zawinski *	1Changed the phonetic matcher's output to be less sure of itself, since it makes some crazy matches.*
;1;;*  18 Jan 90*	1Jamie Zawinski *	1Made mailing-lists fingerable.*
;1;;   3 Mar 90*	1Eric Karlson*	1Fixed handling of /W so that fingering "/W user" would show info about user even if they are not *
;1;;*				1 logged in - it had been showing info about the logged-in user.*
;1;;*		1Jamie Zawinski*	1Made the phonetic matcher not dive into the debugger when there are :personal-names in the *
;1;;*				1 namespace that contain no alphabetic characters.*

(export '4*notify-when-fingered**)
(defvar 4*notify-when-fingered* *nil
  "2If T, then send a notification when someone fingers you at this machine.  
If :ALWAYS, then send a notification when someone fingers this machine or anybody on it.*")

;1;; The Profile interface to plan files.*
;1;;*
(profile:define-profile-variable *notify-when-fingered* (:network)
  :cvv-type (:assoc '(("3Never *" . nil) ("3Sometimes *" . t) ("3Always *" . :always)))
  :documentation "2Never: Don't notify.  Sometimes: Notify when someone fingers you personally.
Always: Notify when this machine recieves any finger request at all.*")

(profile:define-profile-variable .plan-file.
  (:important :network)
  :cvv-type :pathname-or-nil
  :declare-special-p t
  :Documentation "2The file that the User has a Finger Plan in.*"
  :get-value (name:lookup-attribute-value USER-ID :USER :PLAN-FILE :local T)
  :set-effect (name:add-attribute USER-ID :USER :PLAN-FILE plan-file :local T)
  :form-for-init-file (lambda (var)
			`(progn (login-eval (profile-setq ,var nil))
				(profile-setq ,var ',plan-file))))


(defun 4find-host-and-medium *(net-stream-or-connection)
  "2Given just about anything, returns two values: a medium-name, and a host object (or address if host unknown).*"
  (let ((conn net-stream-or-connection)
	med addr host)
    (typecase net-stream-or-connection
      (IP:BASIC-STREAM
       (setq conn (send net-stream-or-connection :connection)))
      ((CHAOS:INPUT-STREAM-MIXIN CHAOS:OUTPUT-STREAM-MIXIN)
       (setq conn (send net-stream-or-connection :connection)))
      (IP:UDP-BASIC-STREAM
       (setq conn (send net-stream-or-connection :port)))
      (t (setq conn net-stream-or-connection)))
    (typecase conn
      (CHAOS:CONN
       (setq med :CHAOS
	     addr (chaos:conn-foreign-address conn)
	     host (net:get-host-from-address addr :CHAOS)))
      (IP:TCP-CONNECTION
       (setq med :IP
	     addr (send conn :destination-address)
	     host (net:get-host-from-address addr :IP)))
      (IP:UDP-PORT
       (setq med :UDP
	     addr (send conn :sender-address)
	     host (net:get-host-from-address addr :IP)))
      (W:SHEET (setq med :interactively host si:local-host addr t))
      )
    (when addr
      (unless host
	(setq host (or (net:print-network-address addr (if (eq med :UDP) :IP med) nil nil)
		       addr)))
      (values med host))))


(defvar 4*finger-hook* *nil
  "2If this is non-NIL, it should be a function of three arguments:
 an output stream, the name-spec being fingered, and whether /W was specified.
 It will be called before any finger information is sent to the remote system;
 If it returns NIL, then all is normal.  If it returns non-NIL, then nothing further will be printed.
 Anything this function writes on the given stream will be shown to the remote system as a finger response.*")


(defun 4show-users-info* (server-stream name-spec &optional suppress-plan-file)
  "2Write a string (perhaps long) on the server-stream describing who is logged into the machine.*"
  (declare (function show-users-info (STREAM STRING) NULL)
	   (values NIL))
  (setq name-spec (string-left-trim '(#\Space #\Tab #\.) name-spec))
  (let ((pos (search "3/W*" name-spec :test #'char-equal)))
    (when pos
      (if (zerop pos)
	(setq name-spec (subseq name-spec 2))
	(setq name-spec (subseq name-spec 0 pos)))
      (setq name-spec (string-trim '(#\Space #\Tab #\.) name-spec)))
    (maybe-notify-about-being-fingered name-spec pos server-stream)
    ;1;*
    ;1; Now decide what string to send back.*
    ;1; If there is a *finger-hook*, invoke it.  If it returns NIL, bug out now (meaning they've printed all they like).*
    ;1; If someone in particular is being fingered, or the logged in user (with /W) then be verbose.*
    ;1; Otherwise be brief.*
    (cond ((and (boundp '*finger-hook*) *finger-hook*
		(funcall *finger-hook* server-stream name-spec (not (null pos)))))
	  ((or (string/= "" name-spec) pos)
	   (show-users-info-verbose name-spec server-stream suppress-plan-file))
	  ((or (null USER-ID) (string= "" USER-ID))
	   (format server-stream "3No one logged on~%*"))
	  (t (show-users-info-brief server-stream)))))


(defun 4maybe-notify-about-being-fingered *(name-spec logged-in-user-p server-stream)
  ;1;*
  ;1; Notify the logged-in user if desired.*
  ;1; If **notify-when-fingered*1 is :ALWAYS, then they always get notified.*
  ;1; If **notify-when-fingered*1 is T, and the fingered name is "/W" or is the logged in user ID, then they get notified.*
  ;1; If nobody is logged in, or one of the servers is logged in, then they get notified regardless.*
  (when (or (eq *notify-when-fingered* :always)
	    (and *notify-when-fingered*
		 (if (equal name-spec "")
		     logged-in-user-p
		     (or (string= USER-ID "")
			 (string-equal name-spec USER-ID)
			 (search "3Server*" USER-ID :test #'char-equal)))))
    ;1;*
    ;1; Figure out what kind of medium was in use, and what host called us.*
    (multiple-value-bind (medium host) (find-host-and-medium server-stream)
      (unless host (setq host "3an unknown host*"))
      (when (eq medium :chaos) (setq medium "3Chaos*"))	;1 I like mixed case.*
      (if (equal name-spec "")
	  (if logged-in-user-p
	      (process-run-function "3Notify*" 'w:notify nil "3Logged in user being ~A fingered from ~A*" medium host)
	      (process-run-function "3Notify*" 'w:notify nil "3~A Finger from ~A*" medium host))
	  (process-run-function "3Notify*" 'w:notify nil "3~A being ~A fingered from ~A*" name-spec medium host)))))


(defun 4show-users-info-brief *(stream)
  "2Write one line describing who is logged in, what they're doing, and how long they've been idle.*"
  (let* ((idle (floor (time-difference (time) w:KBD-LAST-ACTIVITY-TIME) 3600)))
    (format stream "3~9A~18A ~A ~:[~:[~3,32D:~2,48D~;~*~6,32D~]~;~3*      ~] ~A~:[~;~:*, ~A~]~%*"
	    USER-ID
	    (or (name:lookup-attribute-value USER-ID :USER :PERSONAL-NAME) "3???*")
	    (user-activity-string)
	    (zerop idle)
	    (zerop (floor idle 60))
	    (floor idle 60)
	    (rem idle 60)
	    sys:LOCAL-FINGER-LOCATION
	    (name:lookup-attribute-value USER-ID :USER :WORK-PHONE))))


(defvar 4*finger-matches-phonetically* *t "2True if phonetic matching is enabled in the finger server.*")

(defun 4show-users-info-verbose *(name-spec stream &optional suppress-plan-file)
  "2Write everything we know about the given user.  If NAME-SPEC is NIL or \"\", talk about the logged in user (if any).*"
  (cond ((and (string= "" name-spec) (or (null USER-ID) (string= USER-ID "")))
	 (format stream "3No one logged on~%*"))
	(t
	 (when (string= "" name-spec) (setq name-spec (or USER-ID "")))
	 (let* ((user-info (or (name:list-object name-spec :USER)
			       (name:list-object name-spec :MAILING-LIST))))
	   (cond ((and (null user-info)
		       *finger-matches-phonetically*)
		  (let* ((matches (let ((*trace-output* stream))  ;1 for debugging the matcher.*
				    (match-user-names name-spec))))
		    (cond ((null matches)
			   (format stream "3No matches for \"~A\".*" name-spec))
			  ((null (cdr matches))
			   (format stream "3Unknown user: \"~A\"; best guess is ~A: \"~A\"~2%*"
				   name-spec (car matches)
				   (or (name:lookup-attribute-value (car matches) :USER :PERSONAL-NAME) ""))
			   (let* ((*finger-matches-phonetically* nil))
			     (show-users-info-verbose (car matches) stream)))
			  (t
			   (format stream "3Unknown user: \"~A\"; perhaps you mean:~2%*" name-spec)
			   (dolist (match matches)
			     (format stream "3   ~15A ~A~%*"
				     match (or (name:lookup-attribute-value match :USER :PERSONAL-NAME) "3???*")))))))
		 ((null user-info)
		  (format stream "3Login name: ~30A In real life: ???~%*" name-spec))
		 
		 ((eq (second user-info) :MAILING-LIST) (show-users-info-mailing-list (car user-info) stream))
		 
		 (t
		  (let* ((real-name (get (locf (third user-info)) :PERSONAL-NAME))
			 (home-host (name:lookup-attribute-value name-spec :USER :HOME-HOST))
			 (mail-addr (name:lookup-attribute-value name-spec :USER :MAIL-ADDRESS))
			 (office (or (name:lookup-attribute-value name-spec :USER :OFFICE)
				     (name:lookup-attribute-value home-host :HOST :LOCATION)))
			 (work-phone (name:lookup-attribute-value name-spec :USER :WORK-PHONE))
			 (home-phone (name:lookup-attribute-value name-spec :USER :HOME-PHONE))
			 (plan-file (and (not suppress-plan-file)
					 (or (name:lookup-attribute-value name-spec :USER :PLAN-FILE)
					     (catch 'PASSWORD-BUGOUT
					       ;1 user-homedir-pathname sometimes wants a password... bug out if it does.*
					       (let ((fs:*generic-login-function* 'fingerd-avoid-password-query)
						     (USER-ID "3Fingerd*"))
						 (make-pathname :defaults
								(fs:user-homedir-pathname home-host nil (first user-info))
								:name "3PLAN*" :type "3TEXT*" :version :newest)))))))
		    (show-users-info-verbose-internal (car user-info)
						      real-name mail-addr office work-phone home-phone plan-file
						      stream))))))))


(defun 4show-users-info-mailing-list *(mlist-name stream)
  (let* ((remark (name:lookup-attribute-value mlist-name :mailing-list :remark :chase-aliases t))
	 (addrs  (name:lookup-attribute-value mlist-name :mailing-list :address-list :chase-aliases t)))
    (format stream "3Mailing List: ~A~%*" mlist-name)
    (when remark (format stream "3  ~A~&*" remark))
    (cond ((null addrs) nil)
	  ((null (cdr addrs))
	   (format stream "3Mail is forwarded to \"~A\"*" (car addrs)))
	  (t (format stream "3Mail is relayed to:~%*")
	     (dolist (addr addrs) (format stream "3  ~A~%*" addr))))
    ))


(defun 4show-users-info-verbose-internal *(name-spec real-name mail-addr office
					     work-phone home-phone plan-file
					     stream)
  "2Write the given user info on the stream in a structured manner.  Also dump the plan file if it exists.*"
  (format stream "3Login name: ~30A In real life: ~A~%*" name-spec real-name)
  (let* ((idle (floor (time-difference (time) w:KBD-LAST-ACTIVITY-TIME) 3600)))
    (format stream "3User activity: ~:[Not logged on~;~27A Idle: ~:[~D:~2,48D~;~*~D~]~]~%*"
	    (string-equal USER-ID name-spec)
	    (let* ((s (user-activity-string)))
	      (if (> (length s) 26) (subseq s 0 26) s))
	    (zerop (floor idle 60))
	    (floor idle 60)
	    (rem idle 60)))
  (format stream "3~42A~:[~;~:* Home phone: ~A~]~%*"
	  (format nil "3~@[Office: ~A~:[~;~:*, ~A~]~]*" office work-phone)
	  home-phone)
  (when mail-addr
    (ignore-errors
      (let* ((addr (mail:parse-address mail-addr))
	     (domain (send addr :destination-domain))
	     (host (or (net:parse-host domain t)
		       (net:parse-host (subseq domain 0 (position #\. domain)) t))))
	(when (eq host si:local-host)
	  (setq mail-addr "3this machine.*"))))
    (when mail-addr (format stream "3Mail is forwarded to ~A~%*" mail-addr)))
  (when plan-file
    (ignore-errors
      (let* ((USER-ID "3Fingerd*"))
	(show-plan-file plan-file stream)))))


(defun 4show-plan-file *(plan-file output-stream)
  "2Print the contents of the given file on the give stream, being careful not to prompt for passwords, and trapping errors.*"
  (when (eq output-stream 't) (setq output-stream *standard-output*))
  (send output-stream :force-output)	;1 Force output so that the fingerer has something to read while we are opening *
  (condition-call (condition)		;1 file connections and so on.*
      (let* ((result
	       (catch 'PASSWORD-BUGOUT	;1 If the machine holding the plan file wants a password, bug out by throwing here.*
		 (let ((fs:*generic-login-function* 'fingerd-avoid-password-query))
		   ;1; Try to look at the file.  If an error occurs, handle it below.*
		   (let* ((user-id "3fingerd*"))
		     (when (and plan-file (probe-file plan-file))
		       (view-file plan-file output-stream)
		       t))))))
	(cond ((eq result :passwd-lossage) ;1 This will be thrown to password-bugout if a password is required and we don't have it.*
	       (format output-stream "3Error accessing plan file on ~A: password required.~%*"
		       (or (ignore-errors (pathname-host plan-file)) "3host ???*")))
	      ((null result)		     ;1 The file didn't exist.*
	       (format output-stream "3No plan.~%*"))
	      ;1; Otherwise, the file was printed.*
	      (t nil)))
    ;1; If any kind of error occurred while accessing the plan file (while probing its existence or opening it for input) report the error.*
    ((condition-typep condition 'error)		;1 If we get any error, report it to the fingerer.*
     (format output-stream "3Error accessing plan file on ~A: ~A~%*"
	     (or (ignore-errors (pathname-host plan-file)) "3host ???*")
	     condition))))


(defun 4fingerd-avoid-password-query *(host &rest ignore)
  "2Throw the symbol :PASSWD-LOSSAGE to the catch tag 'PASSWORD-BUGOUT if an attempt is made to prompt for a password.
  Used by SHOW-PLAN-FILE.*"
  ;1; Find any password to the target machine that might still be hanging around.*
  (let* ((uname (cdr (assoc host fs:user-unames)))
	 (rest (and uname (fs:lookup-password-etc uname host))))
    (cond ((and uname (car rest))	;1 If we have a password, use it.*
	   (apply #'values uname rest))
	  (t
	   (throw 'PASSWORD-BUGOUT :passwd-lossage)))))


(defun 4ascii-name-server* ()
  "2This function is called to handle a foreign FINGER request over TCP or UDP.*"
  (declare (function ascii-name-server () STREAM)
	   (values SERVER-STREAM))
  (condition-case ()
      (with-open-stream (server-stream (host:listen-for-connection-on-medium
					 :byte-stream "3ascii-name*"
					 :stream-type :ascii-translating-character-stream
					 :timeout-after-open nil))
	(send tv:who-line-file-state-sheet
	      :add-server (make-instance 'net:generic-peek-bs-server :stream server-stream)
	      "3ASCII Name*" si:current-process)
	(when server-stream (show-users-info server-stream (read-line server-stream))))
    (sys:network-error nil)))


(defun 4chaos-name-server* ()
  "2This function is called to handle a foreign FINGER request over CHAOS.*"
  (declare (function chaos-name-server () STREAM)
	   (values SERVER-STREAM))
  (condition-case ()
      (with-open-stream (server-stream (host:listen-for-connection-on-medium
					 :byte-stream "3name*"
					 :stream-type :character-stream
					 :timeout-after-open nil))
	(send tv:who-line-file-state-sheet
	      :add-server (make-instance 'net:generic-peek-bs-server :stream server-stream)
	      "3Chaos Name*" si:current-process)
	(when server-stream
	  (show-users-info server-stream
			   (getf (chaos:conn-plist (send server-stream :connection)) 'chaos:rfc-arguments))))
    (sys:network-error nil)))


(defun 4give-finger* (&aux connection idle pkt str)
  "2This function is called to handle the LISP Machine fingers (over CHAOS, TCP or UDP).*"
  ;1; This one doesn't call 2show-users-info* because it's supposed to return a one-line string (as Terminal-F).*
  ;1; So we do the notification magic in here.*
  (declare (function give-finger (&aux STREAM FIXNUM T STRING) T))
  (condition-case ()
      (unwind-protect
	  (progn
	    (setf connection (host:listen-for-connection-on-medium :datagram "3lispm-finger*"))
	    (when connection
	      (when (eq *notify-when-fingered* :always)
		;1; Figure out what kind of medium was in use, and what host called us.*
		(multiple-value-bind (medium host) (find-host-and-medium connection)
		  (unless host (setq host "3an unknown host*"))
		  (process-run-function "3Notify*" 'w:notify nil "3~A LISPM Finger from: ~A*" medium host))))
	    (setq idle (floor (time-difference (time) tv:kbd-last-activity-time) 3600))	 ;1;Minutes*
	    ;1; Making the string is expensive in terms of paging, and it is almost*
	    ;1; always the same as last time.  So try to use a saved string.*
	    (cond ((or (not (eq give-finger-saved-idle idle))
		       (not (eq give-finger-saved-user-id user-id)))
		   (setq give-finger-saved-idle idle
			 give-finger-saved-user-id user-id
			 give-finger-saved-string
			 (format () "3~A~%~A~%~:[~3*~;~:[~D:~2,48D~;~*~D~]~]~%~A~%~C~%*" user-id
				 si:local-finger-location (not (zerop idle)) (zerop (floor idle 60))
				 (floor idle 60) (rem idle 60) fs:user-personal-name-first-name-first
				 fs:user-group-affiliation))))
	    (multiple-value-setq (pkt str) (send connection :get-empty-pkt))
	    (copy-array-contents give-finger-saved-string str)
	    (send (prog1 connection (setf connection nil)) :answer
		  (prog1 pkt (setf pkt nil)) (length give-finger-saved-string)))
	(when pkt (send connection :return-output-pkt pkt))
	(when connection (send connection :return-connection "3Connection aborted*")))
    (sys:network-error nil)))


;1;; Adding phonetic matching to the Finger daemon.*


(defconstant 4SOUNDEX-CHARACTER-VALUES* #(7 1 2 3 7 1 2 7 7 2 2 4 5 5 7 1 2 6 2 3 7 1 7 2 7 2)
   "2Vector of character values accessible by* 2(aref ... (- (char-int (char-upcase x)) (char-int #\A)))*")

;1;; This is based on some code I got from David Forster, which is based on:*
;1;;*	2Huffman, Edna K. (1972) Medical Record Management.  Berwyn, Illonois: Physicians' Record Company.*
;1;;*
(defun 4soundex-hash* (string &optional (start 0) end)
  "2Return a number representing the given string; this number has the property that words which are 
 similar in pronounciation and words which rhyme will usually have the same number.*"
  (unless end (setq end (length string)))
  (setq start (position-if #'alpha-char-p string :start (or start 0)))
  (unless start (return-from soundex-hash -1))
  (let* ((out 0)
	 (first-character (char-int #\A))
	 (last-letter (char-upcase (char string start)))
	 (last-class  (aref SOUNDEX-CHARACTER-VALUES
			    (- (char-int last-letter) first-character)))
	 (this-letter #\Null)
	 (this-class #\Null))
    (declare (string-char last-letter last-class this-letter this-class))
    (do* ((i start (1+ i)))
	 ((>= i end))
      (setf this-letter (char-upcase (char string i)))
      (when (alpha-char-p this-letter)
	(setf this-class (aref soundex-character-values
			       (- (char-int this-letter) first-character)))
	(when (char/= this-class last-class)
	  (if (/= this-class 7)
	      (setq out (+ (* out 10) this-class))
	      ;1; could be H or W separating like letters:*
	      (when (and (or (char= this-letter #\H)
			     (char= this-letter #\W))
			 (< (1+ i) end)
			 (char= (aref soundex-character-values
				      (- (char-int (char-upcase (char string (1+ i))))
					 first-character))
				last-class))
		(incf i)
		(setf this-letter (char-upcase (char string i)))
		(setf this-class (aref SOUNDEX-CHARACTER-VALUES
				       (- (char-int this-letter) first-character))))))
	(setf last-letter this-letter
	      last-class  this-class)))
      out))


(defun 4name-spec-matches-namespace-user-object-p *(name-spec user-object
						        &optional (debug-p (eq *finger-matches-phonetically* :DEBUG)))
  "2True if NAME-SPEC matches the given object from the :USER class of the namespace, taking into account spelling errors and phonetics.
  NAME-SPEC may be a string, or a cons of (<first-name> . <last-name>).
  It is compared against the user's login name as well as their real name.*"
  (let* ((first (if (consp name-spec) (car name-spec) name-spec))
	 (last (when (consp name-spec) (cdr name-spec)))
	 (personal-name (find :PERSONAL-NAME (name:object-attributes user-object) :key #'name:attribute-name)))
    (macrolet ((match (string1 string2 &optional (from2 0) to2)
		 `(let* ((ok (= (soundex-hash ,string1)
				(soundex-hash ,string2 (or ,from2 0) ,to2))))
		    (when (and ok debug-p)
		      (format *trace-output* "3~&   ~8d: ~s = ~s~%*"
			      (soundex-hash ,string1) ,string1 (subseq ,string2 ,from2 ,to2)))
		    ok)))
      (when personal-name
	(when (name:attribute-p personal-name)
	  (setq personal-name (name:attribute-value personal-name)))
	(when personal-name
	  (let* ((space1 (position #\Space personal-name :test #'char=))
		 (space2 (and space1 (position #\Space personal-name :test #'char= :from-end t))))
	    (cond ((null last)
		   (or (match name-spec (name:object-name user-object))
		       (and space1 (match name-spec personal-name 0 space1))
		       (and space2 (match name-spec personal-name (1+ space2)))
		       (and (not space1) (match name-spec personal-name))))
		  (t
		   (if (and space1 space2)
		       (and (match first personal-name 0 space1)
			    (match last personal-name (1+ space2)))
		       (match first personal-name))))
	    ))))))


(defun 4finger-phonetic-match-p *(object name-pattern class attrs)
  "2Internal function to MATCH-USER-NAMES.*"
  (declare (ignore attrs))
  (and (eq class :USER)
       (let* ((dot (or (position #\. name-pattern :test #'char=)
		       (position #\Space name-pattern :test #'char=)))
	      (name-spec (if dot
			     (cons (subseq name-pattern 0 dot) (subseq name-pattern (1+ dot)))
			     name-pattern)))
	 (name-spec-matches-namespace-user-object-p name-spec object))))


(defun 4match-user-names *(name-spec)
  "2Returns a list of all login-ids in the :USER class of the namespace which phonetically match NAME-SPEC.
 If NAME-SPEC contains a period, it is considered to be a firstname-lastname pair.*"
  (mapcar #'name:object-name
	  (name:lookup-objects-from-properties :class :USER :name-pattern name-spec
					       :test 'FINGER-PHONETIC-MATCH-P :read-only t)))



;1;; A better definition of User-Activity-String.  The old version was just insane.*
;1;;*
(defun 4user-activity-string* ()
  (or (condition-call (c)
	  (let ((w (do ((w tv:selected-window (send w :superior))
			(w1 nil w))
		       ((or (null w) (typep w 'tv:standard-screen))
			w1))))
	    (when w
	      (if (send w :operation-handled-p :user-activity-string)
		  (send w :user-activity-string)
		  nil)))
	((errorp c) nil))
      "3??User Activity??*"))


(defmethod 4(w:label-mixin :user-activity-string)* ()
  "2If this window has a string in its label, return that.  Otherwise return the :name.*"
  (let ((s (and (consp tv:label) (w:label-string tv:label))))
    (if (stringp s) s
	(send self :send-if-handles :name))))

(defmethod 4(zwei:zmacs-frame    :user-activity-string)* () "2Zmacs*")
(defmethod 4(tv:peek-frame       :user-activity-string)* () "2Peek*")
(defmethod 4(zwei:converse-frame :user-activity-string)* () "2Converse*")
(defmethod 4(tv:inspect-frame    :user-activity-string)* () "2Inspect*")
(defmethod 4(w:lisp-listener     :user-activity-string)* () "2Lisp*")
(defmethod 4(user:telnet         :user-activity-string)* () "2Telnet*")
(defmethod 4(telnet:vt100-frame  :user-activity-string)* () "2VT100*")
(defmethod 4(fed:fed-frame       :user-activity-string)* () "2Font Edit*")

(defmethod 4(tv:pop-up-notification-window*	4:user-activity-string)* () "2Pop-Up Notification*")
(defmethod 4(tv:confirmation-window*		4:user-activity-string)* () "2Mouse-Confirm*")
