;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;; ChangeLog:
;;;
;;;  5 Mar 87  Jamie Zawinski  Created.
;;; 20 Mar 87  Jamie Zawinski  Added trapping of printing output on remote machine.  Made TIMEOUT able to be NIL.
;;; 25 Mar 87  Jamie Zawinski  Made EVALserver spawn processes to do the work, so that the main loop can always be ready
;;;                             for input, and so it can never hang.
;;; 14 Apr 87  Jamie Zawinski  Made EvalServer main loop ignore the condition "Bad Connection State", since it happens
;;;                             every time the machine is rebooted.
;;;                            Added FORCE-READABLE so that if the remote evaluation returns an unreadable object, the
;;;                             local contact won't die.
;;; 14 May 88  Jamie Zawinski  Cleaned up.
;;;  2 Nov 88  Jamie Zawinski  Made it not try to signal the error on the local machine.  That was silly.

;;;
;;; Synopsis:
;;;
;;; This utility allows you to evaluate forms on a remote Lisp Machine without actually typing it in there.
;;; If the result of the remote evaluation has a readable printed representation, then it will be returned by the client;
;;; This means that (EVAL-ON ("SomeHost") FS:LOCAL-HOST) will return an instance of the NET::HOST flavor.
;;; Any output to *standard-output* or *terminal-io* on the remote machine will be printed on the local machine instead.
;;; If an error occurred on the remote host, then a condition object is returned.
;;; 
;;; It would be nice if this code actually set up a bidirectional stream to the server, so that Y-OR-N-P (etc) would work.
;;; The way it works is that the server passes to the client a string of output done, and a list of values returned.
;;;
;;; Exports:
;;;
;;; (EVAL-ON-HOST (<host-name> &OPTIONAL <timeout>)    Evaluate BODY on the host HOST timing out after TIMEOUT seconds.
;;;               &BODY <body>)
;;;
;;; START-EVALSERVER     Start the process on the local machine.  This must be done before anyone can EvalServe off of you.
;;; STOP-EVALSERVER      Kill the EvalServer process.  If you do this, no one can EvalServe of of you.
;;;

(export '(eval-on-host remote-eval start-evalserver stop-evalserver))

(defvar *evalserver-process* nil)
(defvar *evalserver-contact-name* "NEWEVALSERVER")

(defun force-readable (object)
  "If OBJECT can br successfully read from a string, then it is returned.
Otherwise a cons of (:UNREADABLE-OBJECT . <printed-representation-of-object>) is returned."
  (let* ((printed-representation (prin1-to-string object)))
    (condition-call (condition-instance)
	(progn (read-from-string printed-representation)
	       object)
      (t condition-instance
	 (cons :unreadable-object printed-representation)))))


(defun evalserver-internal (conn)
  "Does EVALSERVER on a Chaos connection."
  (chaos:accept conn)
  (when (boundp 'tv:who-line-file-state-sheet)
    (send tv:who-line-file-state-sheet :add-server conn "EVAL"))
  (condition-call (condition-instance)
      (let* (foreign-host chaos-stream)
	(unwind-protect
	    (progn
	      (setq chaos-stream (chaos:make-stream conn))
	      (setq foreign-host (si:get-host-from-address (chaos:foreign-address conn) :chaos))
	      
	      (let* ((form (read chaos-stream))  ; The client connection will write us a form to evaluate.
		     (values nil)
		     (output-string nil))
		;; OUTPUT-STRING is a string of all output printed by the Eval Request.
		(setq output-string
		      (with-output-to-string (stringstream)
			(let* ((*terminal-io* stringstream)      ; Bind all streams to this, just in case...
			       (*standard-output* stringstream)
			       (*error-output* stringstream)
			       (*trace-output* stringstream)
			       (*query-io* stringstream)
			       (*debug-io* stringstream)
			       (*standard-input* :CANT-DO-INPUT)

			       (*print-array* t)     ; Print everything as readably as possible.
			       (*print-circle* t)
			       (*print-pretty* nil)
			       (*print-escape* t)
			       (*print-length* nil)
			       (*print-level* nil)
			       (*print-base* 10)
			       (*print-gensym* t))
			  
			  ;; VALUES is a list of all the values that the Eval Request returned, or
			  ;; a list of the form (:ERROR <condition-instance>) if an error occurred.  Conditions are readable.
			  (setq values
				(condition-call (condition-instance)
				    (multiple-value-list (eval form))
				  ((not (send condition-instance :dangerous-condition-p))
				   `(:error ,condition-instance)))))))
		(setq values (mapcar #'force-readable values))
		(let* ((*package* nil))  ; Forces printing of package prefix.
		  ;;
		  ;; Now we write a list back to the client connection.  This is a list of
		  ;; (<string-of-all-output> &REST <values>)
		  ;;
		  (print (cons output-string values) chaos-stream))))
	  (send chaos-stream :finish)
	  (if conn (chaos:remove-conn conn))))
    (t
     (tv:notify nil (format nil "Error from EvalServer: ~A" condition-instance)))))


(defun evalserver-function ()
  "The function which waits for packets to the Evalserver over the Chaosnet, and spawns handler processes."
  (loop
    (condition-call (condition-instance)
	(let ((conn (chaos:listen *evalserver-contact-name*)))
	  (process-run-function "EvalServer Spawn" #'evalserver-internal conn))
      ((condition-typep condition-instance 'system::bad-connection-state)       ; Ignore this kind of error.
       nil)
      (t
       (tv:notify nil (format nil "Error from EvalServer: ~A" condition-instance))))))


(defun remote-eval (host form &optional timeout-in-seconds)
  "Evaluate FORM on HOST, timeing out after TIMEOUT-IN-SECONDS.  NIL ==> never timeout.
  Returns the values that FORM returned, and prints what FORM printed."
  (with-open-stream (stream (chaos:open-stream host *evalserver-contact-name*))
    ;;
    ;; At this point STREAM is a stream to the #'EVALSERVER-INTERNAL function on the server host.
    ;; That function is waiting for us to write a form to it.  It then writes a list back to us -
    ;; The CAR of this list is a string, which is all of the output done at the client.
    ;; The CDR is the list of values returned by our form.
    ;;
    (let* ((*package* nil))  ; Binding *PACKAGE* to NIL forces printing of package prefix on all symbols.
      (print form stream))
    (send stream :force-output)
    (let* ((values (if timeout-in-seconds
		       (with-timeout ((* 60 timeout-in-seconds)
				      :timeout)
			 (read stream))
		       (read stream))))
      (when (eq values :timeout)
	(cerror "Proceed with no timeout." "Timed out waiting for response from EvalServer at ~S" host)
	(setq values (multiple-value-list (read stream))))
      ;;
      ;; The CAR of VALUES is a string of everything output by the code on the remote host.  Print it.
      (princ (pop values))
      ;; VALUES is now the list of values returned.
      (cond ((and (eq (first values) :ERROR)
		  (typep (second values) 'condition))
	     (format *debug-io* "~&An error occurred: ~A~2%" (second values))
	     (second values))
	    (t 
	       (apply #'values values))))))


(defmacro eval-on-host ((host &optional timeout) &body body)
  "Evaluate the contents of BODY on the network host HOST.  The values returned by BODY are returned by EVAL-ON-HOST.
If TIMEOUT seconds pass before the remote host finishes evaluating, then a correctable error is signalled on the local host.
If any errors occur during the evaluation of BODY, the errors are handled on the local host."
  `(remote-eval ',host '(progn ,@body) ,timeout))


(defun start-evalserver ()
  "Start the EVALSERVER process running."
  (when *evalserver-process* (send *evalserver-process* :kill))
  (setq *evalserver-process* (process-run-function '(:name "EvalServer" :restart-after-reset t)
						     'evalserver-function)))


(defun stop-evalserver ()
  "Kill the EVALSERVER process.  You can restart it with START-EVALSERVER."
  (when *evalserver-process* (send *evalserver-process* :kill))
  (setq *evalserver-process* nil))


(add-initialization "Start EvalServer" '(start-evalserver))
