;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:(WEBSTER use (LISP)); Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B CPTFONTBI HL12I) -*-

;1;; File "3DICTIONARY-CLIENT*".*
;1;; Copyright (c) 1989 University of California, Berkeley.*
;1;; Talks to the Webster's Dictionary server (by Ed James).*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   10 Jan 90*	1Jamie Zawinski*	1Created.*
;1;;*  116 Jan 90*	1Jamie Zawinski *	1Commented.*
;1;;*  129 Jan 90*	1Jamie Zawinski *	1Merged in Michael Braverman's hacks to get this running on Symbolics (Genera 7).*
;1;;*  131 Jan 90*	1Jamie Zawinski *	1Made it so that after listing words, the matches go on the completion alist.*
;1;;*   15 Feb 90*	1Jamie Zawinski *	1Made it possible to grind a definition into a buffer using only ASCII characters, instead*
;1;;*				1 of using the Lispm's special-characters as well.*
;1;;*

;1;;*
;1;; Comments about what needs to be done to make the Symbolics interface as cool as the Explorer one begin with "##".*
;1;;*


;1;; 5Zmacs Interface:**
;1;; *
;1;; This file defines five Zmacs commands for interrogating the dictionary server:*
;1;;*
;1;;   5Meta-X Dictionary**
;1;;*	1Prompts for a word or phrase, and prints out the definition of that word; bold words in the text of the definition*
;1;;*	1are mousable.  Clicking on them will show you their definitions.*
;1;;*
;1;;   5Meta-X Dictionary Search**
;1;;*	1Prompts for a list of words, separated by spaces or commas.  Prints out a list of all words in the dictionary which*
;1;;*	1begin with the given prefixes.  Caveat: if what you specified matches more than 100 words, you get nothing.*
;1;;*
;1;;   5Meta-X Dictionary Into Buffer**
;1;;*	1Just like 5Meta-X Dictionary*, but the definition will be inserted into the current buffer after the point.*
;1;;*
;1;;   5Meta-X Dictionary Reset**
;1;;*	1Resets the connection to the dictionary server in case the connection has gotten wedged.  You shouldn't need*
;1;;*	1to use this...*
;1;;*
;1;;   5Control-$* (aka 5Control-Shift-4*)*
;1;;*	1``Define word at point'' - this is analogous to the command 5Meta-$*, ``Correct Spelling of word at point,'' and does*
;1;;*	1the same thing as 5Meta-X Dictionary*, except you don't have to type the word.*
;1;;*

;1;; 5The Dictionary Server:**
;1;;*
;1;; Definitions are obtained by asking the dictionary server for them; one such server is on 5pasteur.berkeley.edu*.  *
;1;; You control which server you are talking to with the 5*dictionary-server-host** variable.  The following description*
;1;; of the dictionary protocol was mostly gleaned from reading the C source code to Ed James's 5dict* program, which*
;1;; runs under BSD Unix 4.2 and later.  This program is accessible by anonymous FTP from 5scam.berkeley.edu* (IP*
;1;; address 128.32.138.1) in the file 5/src/local/webster.tar.Z*.  However, the manual page for this program states:*
;1;;2 ``This client program allows you to access the dictionary server for Webster's Seventh Dictionary, published in 1965.**
;1;;2  This dictionary is licensed to the UC Berkeley EECS/CS research community for private use only.  The dictionary**
;1;;2  may not be reproduced whole or in part in any documents, research papers, etc.''**
;1;;*
;1;; The dictionary server has few commands:*
;1;;*
;1;; These three commands cause the server to find a list of words; once it has produced the list, the list is 7not* *
;1;; printed out; rather, a code indicating success or failure is returned.  The lists will never have more than 100 words.*
;1;;*
;1;;   4EXACT** 1[ 2word* ]+*	1- build a list of all of the given words which are in the dictionary.*
;1;;   4REGEX** 1[ 2regex* ]+*	1- build a list of all of the words in the dictionary which match the regular expressions.*
;1;;   4PREFIX** 1[ 2prefix* ]+*	1- build a list of all of the words in the dictionary which begin with the given strings.*
;1;;*
;1;; This next command causes the list built by the previous commands to be printed, one word per line.  The list is *
;1;; done when a line containing only one character, a period, is sent.*
;1;;*
;1;;   4LIST**		1- list the words found with the last search.*
;1;;*
;1;; This last command is for retreiving the definition(s) of a given word or phrase.  If the word or phrase has spaces *
;1;; in it, it must be sent with double-quotes around it (like "some words").*
;1;;*
;1;;   4WORD2 word***		1- send the definition of word (encoded format).*
;1;;*
;1;; All commands return status the way NNTP and SMTP do: a line beginning with a three-digit decimal number, followed*
;1;; by an optional comment (which is usually ignorable).*
;1;;*
;1;; If the 5Word* command is issued on a word for which there is a definition, an ``ok'' status code will be returned, followed*
;1;; by the encoded definition of the word.  There are two levels of encoding.  At the first level, structural elements of the*
;1;; definition are sent one per line, in a form like*
;1;;*
;1;;   2<character>5 : *<field-1>5 ; *<field-2>5 ; *<field-3> ...**
;1;;*
;1;; Where the 2character* says what kind of field this is (definition, label, cross-reference, etc).  Each kind of field has a fixed*
;1;; number of records in it, separated by semicolons.  No line will ever be more than 80 characters long; so a line beginning*
;1;; with 5C:* means ``append everything after the colon to the previous line.''  See the comments in 5#'parse-dictionary-entry**
;1;; for descriptions of what the different kind of fields are.*
;1;;*
;1;; Many of the fields have a 2text* record.  This is a string which has some typesetter information encoded in it.  Special*
;1;; characters and font-changes are encoded with ``overstruck'' characters, that is, a sequence like 2<char-1> <backspace> <char-2>**
;1;; will either change the font, or will map to one or more different characters.*
;1;;*
;1;; There are two bundles of hair to this: first, 2<char-2>* may actually be of the form 5(2 <backspace>* Q2 <character-name-text>* ) Q**
;1;; so that the sequence ``  5a 2<backspace>** 5(2 <backspace>* Qcedilla)Q* '' is to be interpreted as a control code consisting of*
;1;; the letter 5a*, a backspace, and the character called ``cedilla.''*
;1;;*
;1;; The second bundle of hair is that a control sequence which maps to a backspace may take the 2place* of a backspace.*
;1;; So, since `` 5| 2<backspace>** 5B* '' maps to backspace, the sequences `` 5a* 2<backspace>* 5b* '' and `` 5a|* 2<backspace>* 5Bb* '' must be*
;1;; treated as equivalent.  And likewise, `` 5a|* 2<backspace>* 5B(2 <backspace>* Qcedilla)Q* '' is the control code a-backspace-cedilla.*
;1;; Pretty gross, isn't it?*

(in-package "4WEBSTER*" :nicknames '("DICT-CLIENT") :use '("LISP"))
(export '(*dictionary-server-host* open-dict-server-stream flush-dict-server-stream
	  dictionary-list-words present-dict-definition dictionary-word))

#+LISPM (import '(sys:string-append sys:send))


(defvar 4*dictionary-server-host* *"3pasteur*"	;1 Pasteur.Berkeley.EDU, IP address 128.32.134.113 (HP/350, unix-ucb).*
  "2The name of the host which runs the Webster's Dictionary server.*")

(defvar 4*dictionary-server-port* *1964
  "2The Unix port number of the Webster's Dictionary server.*")

(defvar 4*dictionary-server-lock* *nil "2To prevent interfering access to the server.*")

(defvar 4*dictionary-server-stream* *nil "2Bidirectional telnet stream to the dict server.*")

(defmacro 4with-dict-server-lock *(&body body)
  #+EXPLORER  `(sys:with-lock (*dictionary-server-lock* :whostate "2DictServer Lock*") ,@body)
  #+SYMBOLICS `(progn ,@body)	;1 ## how do we do locking on symbolics?*
  #-(or EXPLORER SYMBOLICS) `(progn ,@body))
  
(defvar 4*dictionary-debug* *nil "2If T, dictionary client prints all interactions with the server to *trace-output*.*")


(defun 4open-dict-server-stream *()
  "2Open or re-open the connection to the dictionary server.*"
  (with-dict-server-lock
    (when *dictionary-server-stream*
      (when *dictionary-debug* (format *trace-output* "3~&Closing dict stream...*"))
      #+LISPM (sys:ignore-errors (close *dictionary-server-stream* :abort t))
      #-LISPM (close *dictionary-server-stream* :abort t)
      (setq *dictionary-server-stream* nil))
    (when *dictionary-debug* (format *trace-output* "3~&Opening dict stream...*"))
    (setq *dictionary-server-stream*
	  #+EXPLORER  (ip:open-telnet-stream *dictionary-server-host* :remote-port *dictionary-server-port*)
	  #+SYMBOLICS (tcp:open-tcp-stream *dictionary-server-host* *dictionary-server-port* nil :ascii-translation t)
	  #-(or EXPLORER SYMBOLICS) (error "2Don't know how to open a TCP/IP connection in this Lisp.*")
	  )
    ;1; I don't understand - after I open the stream, there should be a 200 waiting for me, but there's nothing.*
    ;1; And if I write anything, I get a 500 - unrecognised command.  Telnet-stream must be dumping some junk*
    ;1; out there before returning.  So... I write a newline, and wait for the 500 to come back before returning.  Ick.*
    (terpri *dictionary-server-stream*)
    (force-output *dictionary-server-stream*)
    (read-char *dictionary-server-stream*) ;1 wait for it...*
    (clear-input *dictionary-server-stream*) ;1 then ditch it.*
    *dictionary-server-stream*))

#-EXPLORER
(defun 4flush-dict-server-stream *()
  "2Discard all queued input and output on the stream.*"
  (clear-input *dictionary-server-stream*)
  (clear-output *dictionary-server-stream*))

#+EXPLORER
(defun 4flush-dict-server-stream *()
  "2Discard all queued input and output on the stream.*"
  ;1; 5clear-input* and 5clear-output* don't seem to work right on Explorer telnet-streams, so do it the hard way...*
  (terpri *dictionary-server-stream*)
  (force-output *dictionary-server-stream*)
  (read-char *dictionary-server-stream*) ;1 wait for it...*
  (clear-input *dictionary-server-stream*) ;1 then ditch it.*
  (do* () ((not (read-char-no-hang *dictionary-server-stream* nil nil)))
    (clear-input *dictionary-server-stream*)))


(defun 4send-dictionary-command *(command &rest arguments)
  (ecase command
    ((:EXACT :PREFIX :REGEX)
     (assert arguments (arguments) "3The ~A command needs at least one argument.*" command))
    ((:LIST :HELP)
     (assert (null arguments) () "3The ~A command takes no arguments.*" command))
    (:WORD
     (assert (= 1 (length arguments)) (arguments) "3The ~A command takes exactly one argument.*" command)
     (when (position #\Space (car arguments) :test #'char=)
       (setq arguments (list (string-append #\" (car arguments) #\"))))))
  (when *dictionary-debug* (format *trace-output* "3~&***> writing  \"~A~{ ~A~}\"*" command arguments))
  (with-dict-server-lock
    (write-string (symbol-name command) *dictionary-server-stream*)
    (dolist (arg arguments)
      (write-char #\Space *dictionary-server-stream*)
      (princ arg *dictionary-server-stream*))
    (write-char #\Newline *dictionary-server-stream*)
    (force-output *dictionary-server-stream*))
  command)


(defvar 4*dict-server-reply-codes**
	'(			;1 5Debug codes.**
	  (100 . INF-TEXT)	;1 Server pleasantry, please print*
	  (199 . INF-DEBUG)	;1 Debug output*
				;1 5OK response codes.**
	  (200 . OK-HELLO)	;1 Hello, what be happenin'?*
	  (201 . OK-HELP)	;1 Help info*
	  (205 . OK-GOODBYE)	;1 Closing connection*
	  (210 . OK-ECHO)	;1 Sample echo command*
	  (211 . OK-PREFIX)	;1 Prefix Search Succeeded*
	  (212 . OK-EXACT)	;1 Exact Search Succeeded*
	  (213 . OK-REGEX)	;1 Regex Search Succeeded*
	  (220 . OK-LIST)	;1 List of words follows*
	  (221 . OK-WORD)	;1 Word info follows*
				;1 5Continue response codes.**
	  (350 . CONT-POST)	;1 Continue to post article*
				;1 5Error response codes.**
	  (400 . ERR-GOODBYE)	;1 Have to hang up for some reason*
	  (401 . ERR-TOO-MANY)	;1 Too many matchs on the query*
	  (402 . ERR-USAGE)	;1 wrong arguments*
	  (403 . ERR-NOT-FOUND)	;1 no such word*
	  (404 . ERR-FORMAT)	;1 internal index format error*
	  (405 . ERR-FILE)	;1 couldn't read a dict file*
	  (406 . ERR-SERVER)	;1 some internal error*
				;5 Serious error response codes.*
	  (500 . ERR-COMMAND)	;1 Command not recognized*
	  (501 . ERR-CMDSYN)	;1 Command syntax error*
	  (502 . ERR-CNI)	;1 Command not implemented*
	  (503 . ERR-ACCESS)	;1 Access to server denied*
	  )
  "2The reply codes generated by the Dictionary Server (and, coincidentally, the NNTP server...)*")
	  

(defun 4parse-dict-server-reply-code-1 *(stream)
  "2Parse a line of the form ``nnn some text'' where nnn is a three digit reply code, as found in **dict-server-reply-codes*2.
  Returns the symbol corresponding to the code; a symbol representing how important this response is; and the response code itself.*"
  (declare (values reply-symbol severity reply-number))
  (let* ((str (make-string 3)))
    (setf (char str 0) (read-char stream)
	  (char str 1) (read-char stream)
	  (char str 2) (read-char stream))
    (let* ((n (parse-integer str :junk-allowed t)))
      (cond (n
	     (peek-char t stream) ;1 discard trailing whitespace.*
	     (let* ((severity (case (digit-char-p (char str 0))
				(1 'INFORMATIVE)
				(2 'OK)
				(3 'OK-SO-FAR)
				(4 'ERROR)
				(5 'SERIOUS-ERROR)
				(t nil))))
	       (values (cdr (assoc n *dict-server-reply-codes* :test #'=))
		       severity
		       n)))
	    (t (error "3Expected numeric reply code!*"))))))


(defun 4parse-dict-server-reply-code *(stream)
  "2Parse a line of the form ``nnn some text'' where nnn is a three digit reply code, as found in **dict-server-reply-codes*2.
  Returns the symbol corresponding to the code, if everything is ok; signals an error if the reply says that is appropriate.*"
  (loop
    (multiple-value-bind (reply severity n) (parse-dict-server-reply-code-1 stream)
      (let* ((text (read-line stream)))
	(when *dictionary-debug* (format *trace-output* "3~&<***  reading \"~D ~A\"~%*" n text))
	(case severity
	  (INFORMATIVE  nil)
	  (OK-SO-FAR    (error "3I don't know what this means: ~D ~A*" n text))
	  (OK
	   (ecase reply
	     ((OK-HELLO OK-GOODBYE OK-LIST OK-WORD OK-PREFIX OK-EXACT OK-REGEX)
	      (return reply))
	     (OK-HELP    (if (string-equal "3End of help*" text :end2 11)
			     (return nil)
			     (write-line text)))
	     ;1 *OK-ECHO1 <-- what does this one mean?*
	     ))
	  (t
	   (case reply
	     (ERR-TOO-MANY  (format t "3~&The given pattern matched more than 100 words.*")
			    (return reply))
	     (ERR-NOT-FOUND (format t "3~&That word is not in the dictionary.*")
			    (return reply))
	     (ERR-USAGE	    (error "3Improper command usage.~%~A ~A*" n text))
	     (ERR-CMDSYN    (error "3Improper command syntax.~%~A ~A*" n text))
	     (ERR-COMMAND   (error "3Unknown command.~%~A ~A*" n text))
	     (ERR-CNI       (error "3Command not implemented.~%~A ~A*" n text))
	     (ERR-GOODBYE   (error "3Dictionary server says it is hanging up.*"))
	     (ERR-ACCESS    (error "3Access to server has been denied.~%~A ~A*" n text))
	     ((ERR-FORMAT ERR-FILE ERR-SERVER) (error "3Internal dictionary server error.~%~A ~A*" n text))
	     (t (error "3Unknown error code ~A: ~A ~A*" reply n text)))))))))


(defun 4read-dictionary-reply *(command &optional (stream *dictionary-server-stream*))
  "2Assuming the the given command has just been issued, read and process the server's response.*"
  (ecase command
    ((:EXACT :PREFIX :REGEX)
     (let* ((response (parse-dict-server-reply-code stream)))
       (ecase response
	 ((ERR-TOO-MANY ERR-NOT-FOUND)  (values nil response))
	 ((OK-PREFIX OK-EXACT OK-REGEX) (values t response)))))
    (:HELP
     (when *dictionary-debug* (format *trace-output* "3~&Snarfing help text.*"))
     (parse-dict-server-reply-code stream) ;1 this will print on stdout.*
     )
    (:HELLO
     (let* ((reply (parse-dict-server-reply-code stream)))
       (when (or *dictionary-debug* (not (eq reply 'OK-HELLO)))
	 (format *trace-output* "3~&Hello got ~A*" reply))
       reply))
    (:LIST
     (when *dictionary-debug* (format *trace-output* "3~&Snarfing list output...*"))
     (let* ((response (parse-dict-server-reply-code stream)))
       (ecase response
	 (OK-LIST
	  (let* ((words '()))
	    (loop
	      (let* ((word (read-line stream)))
		(when (string-equal word "3.*") (return))		;1 string-equal in case of font lossage.*
		(push word words)))
	    (when *dictionary-debug* (format *trace-output* "3~&Done reading list output.*"))
	    (nreverse words))))))
    (:WORD
     (let* ((response (parse-dict-server-reply-code stream)))
       (ecase response
	 (ERR-NOT-FOUND nil)
	 (OK-WORD (parse-dictionary-entry stream)))))
    ))


(defun 4read-dictionary-def-line *(stream)
  "2This bogus file format doesn't let lines be longer than 80 characters, with continuations marked 
  by ``C:'' as the first chars on the next line.  This function lets us not need to worry about that...*"
  (let* ((line nil)
	 #+EXPLORER (sys:*new-line-delimiter* '(#\Newline #\Linefeed)))
    (loop
      (cond ((and line (string-equal line "3.*"))
	     (when *dictionary-debug* (format *trace-output* "3~&<*** reading ~S - returning.*" line))
	     (return nil))
	    ((and line (char-equal #\C (peek-char t stream)))
	     (when *dictionary-debug* (format *trace-output* "3~&<*** reading ~S - continuation line.*" line))
	     (setq line (string-append line (subseq #-SYMBOLICS (read-line stream)
						    #+SYMBOLICS (zl:read-delimited-string '(#\Newline #\Linefeed) stream)
						    2))))
	    (line
	     (return line))
	    (t
	     (peek-char t stream) ;1 ditch whitespace and blank lines.*
	     #-SYMBOLICS (setq line (read-line stream))
	     #+SYMBOLICS (setq line (zl:read-delimited-string '(#\Newline #\Linefeed) stream))
	     (when *dictionary-debug* (format *trace-output* "3~&<*** reading ~S.*" line))
	     )))))


(defun 4dict-server-tokenize *(string)
  "2Divide up the string at semicolons; discard the first two chars, and map emptystring to NIL.*"
  (let* ((tokens '())
	 (pos 2)
	 (last-token-pos pos)
	 (length (length string)))
    (do* ()
	 ((>= pos length)
	  (push (if (= last-token-pos length) nil (subseq string last-token-pos))
		tokens))
      (when (char= #\; (char string pos))
	(push (if (= last-token-pos pos) nil (subseq string last-token-pos pos))
	      tokens)
	(setq last-token-pos (1+ pos)))
      (incf pos))
    (nreverse tokens)))


(defstruct 4(dict-entry *(:print-function %print-dict-entry))
  name			;1 text*
  homonym		;1 number - for identifying distinct defs that sound alike*
  prefix		;1 keyword*
  dots			;1 ???*
  accents		;1 indices of chars with accents*
  part-of-speech	;1 keyword*
  alt-part-of-speech	;1 keyword or nil*
  pos-preferred-p	;1 whether primary p.o.s. is preferred over secondary*
  body			;1 *( ( 2<field-name-keyword>* 1&rest* 2<data>* ) ... )
  )

(defun 4%print-dict-entry *(struct stream ignore)
  (format stream "3#<~S ~A(~D) ~A ~D>*" (type-of struct)
	  (dict-entry-name struct) (or (dict-entry-homonym struct) "3?*") (or (dict-entry-part-of-speech struct) "3?*")
	  #+LISPM (sys:%pointer struct) #-LISPM nil))


(defun 4parse-dictionary-entry *(stream)
  "2Parse dict-entry fields off of the stream until a line consisting of \".\" is seen.  Returns a list of DICT-ENTRY structures.*"
  (let* ((entries '())
	 entry)
    (loop
      (let* ((line (read-dictionary-def-line stream)))
	(unless line
	  (when entry (setf (dict-entry-body entry) (nreverse (dict-entry-body entry))))
	  (return))
	(case (char line 0)
	  (#\F
	   ;1; First record:*  4F:2entname*;2homono*;2prefsuf*;2dots*;2accents*;2pos*;2posjoin*;2pos2**
	   ;1;*
	   (push (make-dict-entry) entries)
	   (when entry (setf (dict-entry-body entry) (nreverse (dict-entry-body entry))))
	   (setq entry (car entries))
	   (let* ((tokens (dict-server-tokenize line)))
	     (setf (dict-entry-name entry) (pop tokens)
		   (dict-entry-homonym entry) (if (car tokens) (parse-integer (pop tokens)) (pop tokens))
		   (dict-entry-prefix entry) (pop tokens)
		   (dict-entry-dots entry) (parse-dots-spec (pop tokens))
		   (dict-entry-accents entry) (pop tokens)
		   (dict-entry-part-of-speech entry) (parse-part-of-speech (pop tokens))
		   (dict-entry-pos-preferred-p entry) (pop tokens)
		   (dict-entry-alt-part-of-speech entry) (parse-part-of-speech (pop tokens)))
	     (when (dict-entry-accents entry)
	       (setf (dict-entry-accents entry) (parse-accent-spec (dict-entry-accents entry))))
	     (when (dict-entry-prefix entry)
	       (cond ((string-equal "3p*" (dict-entry-prefix entry)) (setf (dict-entry-prefix entry) :prefix))
		     ((string-equal "3s*" (dict-entry-prefix entry)) (setf (dict-entry-prefix entry) :suffix))
		     (t (cerror "3ignore it*" "3unknown prefix code ~s*" (dict-entry-prefix entry)))))
	     (when (dict-entry-pos-preferred-p entry)
	       (cond ((string-equal "32*" (dict-entry-pos-preferred-p entry)) (setf (dict-entry-pos-preferred-p entry) nil))
		     ((string-equal "3_*" (dict-entry-pos-preferred-p entry)) (setf (dict-entry-pos-preferred-p entry) t))
		     (t (cerror "3ignore it*" "3unknown posjoin code ~s*" (dict-entry-pos-preferred-p entry)))))
	     ))
	  (#\P		;1; Pronunciation*	4P:2text**
	   (push (list :PRONUNCIATION (subseq line 2)) (dict-entry-body entry)))
	  (#\E		;1; Etymology*	4E:2text**
	   (push (list :ETYMOLOGY (subseq line 2)) (dict-entry-body entry)))
	  (#\L ;1 ##*
	   ;1; Label*		4L:2snsnumber*;2snsletter*;2snssubno*;2text**
	   (let* ((tokens (dict-server-tokenize line))
		  (num (pop tokens))
		  (sub (pop tokens))
		  (subnum (pop tokens))
		  (txt (pop tokens)))
	     (push (list :LABEL num sub subnum txt) (dict-entry-body entry))
	     ))
	  (#\V ;1 ##*
	   ;1; Variant*		4V:2name*;2dots*;2accents*;2level15()*level2**
	   (let* ((tokens (dict-server-tokenize line))
		  (name (pop tokens))
		  (dots (parse-dots-spec (pop tokens)))
		  (accents (parse-accent-spec (pop tokens)))
		  (levels (pop tokens)))
	     (push (list :VARIANT name dots accents levels) (dict-entry-body entry))))
	  (#\D
	   ;1; Definition*		4D:2snsnumber*;2snsletter*;2snssubno*;2pos*;2text**
	   (let* ((tokens (dict-server-tokenize line))
		  (num (pop tokens))
		  (sub (pop tokens))
		  (subnum (pop tokens))
		  (pos (pop tokens))
		  (txt (pop tokens)))
	     (push (list :DEF num sub subnum pos txt) (dict-entry-body entry))))
	  (#\R
	   ;1; Run-on*		4R:2name*;2dots*;2accents*;2pos1*;2posjoin*;2pos2**
	   (let* ((tokens (dict-server-tokenize line))
		  (name (pop tokens))
		  (dots (parse-dots-spec (pop tokens)))
		  (accents (parse-accent-spec (pop tokens)))
		  (pos1 (parse-part-of-speech (pop tokens)))
		  (pos1-pref (pop tokens))
		  (pos2 (parse-part-of-speech (pop tokens))))
	     (cond ((null pos1-pref) nil)
		   ((string-equal "32*" pos1-pref) (setf pos1-pref nil))
		   ((string-equal "3_*" pos1-pref) (setf pos1-pref t))
		   (t (cerror "3ignore it*" "3unknown posjoin code ~s*" pos1-pref)))
	     (push (list :RUN-ON name dots accents pos1 pos2 pos1-pref) (dict-entry-body entry))))
	  (#\S		;1; Synonym*	4S:2text**
	   (push (list :SYNONYM (subseq line 2)) (dict-entry-body entry)))
	  (#\X
	   ;1; Cross Reference*	4X:2word*;2wrdsuper*;2wrdsubs*;2type*;2word2**
	   (let* ((tokens (dict-server-tokenize line)))
	     (setf (fourth tokens)
		   (ecase (parse-integer (fourth tokens))
		     ;1; This is kind of evil - the overstrike characters are embedded in these strings so that certain words will be bold.*
		     (0 "see (Y~A)Y")
		     (1 "see (Y~A)Y table")
		     (2 (error "illegal xref code 2"))
		     (3 "see ~*(Y~A)Y at ~2:*(Y~A)Y table")
		     (4 "compare (Y~A)Y")
		     (5 "compare (Y~A)Y table")
		     (6 "called also (Y~A)Y")
		     (7 (error "illegal xref code 7"))
		     (8 "(Ysyn)Y see in addition (Y~A)Y")
		     (9 "(Ysyn)Y see (Y~A)Y")))
	     (push (cons :XREF tokens) (dict-entry-body entry))))
	  )))
    (nreverse entries)))


(defun 4parse-accent-spec *(string)
  "2Return a list whose elements are of the form *( <char-pos> . <accent-keyword> )2.*"
  (let* ((result '()))
    (do* ((len (length string))
	  (i 0 (+ i 2)))
	 ((>= i len))
      (let* ((nth-char (char string i))
	     (accent (char string (1+ i))))
	(setq nth-char (or (digit-char-p nth-char)
			   ; 0..9, `=10, [=11, ]=12, {=13, }=14, ^A=15, ^H=16, ^]=17
			   (ecase nth-char
			     (#\` 10) (#\[ 11) (#\] 12) (#\{ 13) (#\} 14)
			     (#.(int-char 1) 15) (#.(int-char 8) 16)
			     (#.(int-char 29) 17))))
	(setq accent (ecase accent
		       (#\( :accent) (#\` :grave)
		       (#\' :caret) (#\: :umlaut)))
	(push (cons (1- nth-char) accent) result)))
    (nreverse result)))

(defun 4parse-part-of-speech *(string)
  (and string (intern (string-upcase string) "3KEYWORD*")))

(defun 4parse-dots-spec *(string)
  "2Return a list of character positions after which a syllable break occurs.*"
  (let* ((last 0))
    (map 'list #'(lambda (x) (1- (incf last (digit-char-p x))))
	 string)))


(defun 4dictionary-list-words *(words &optional prefix-p)
  "2See if the given words are in the dictionary - returns a list of the words that are.
  If PREFIX-P is T, then return any words beginning with the given strings.
  If PREFIX-P is :REGEX, then interpret the words as regular expressions.
  If the words match more than 100 definitions, an error is signalled.
  If there are no matches, NIL is returned.*"
  (when (stringp words) (setq words (list words)))
  (let* ((command (ecase prefix-p
		    (:regexp :REGEX)
		    (nil :EXACT)
		    ((t) :PREFIX))))
    (with-dict-server-lock
      (unless *dictionary-server-stream* (open-dict-server-stream))
      (flush-dict-server-stream)
      (apply 'send-dictionary-command command words)
      (read-dictionary-reply command)
      (send-dictionary-command :LIST)
      (read-dictionary-reply :LIST))))

(defvar 4*dictionary-cache-p* *t
  "2If true, then the dictionary client remembers what it has asked; this makes looking up the same definition twice much faster.*")

(defvar 4*dictionary-cache-alist* *nil "2Previous definitions.*")
  
(defun 4dictionary-get-def *(word)
  "2Given a word, return a structure representing its definition.  If the word is not in the dictionary, returns NIL.*"
  (or (cdr (assoc word *dictionary-cache-alist* :test #'string-equal))
      (with-dict-server-lock
	(unless *dictionary-server-stream* (open-dict-server-stream))
	(flush-dict-server-stream)
	(send-dictionary-command :WORD word)
	(let* ((result (read-dictionary-reply :WORD)))
	  (when *dictionary-cache-p*
	    (push (cons word result) *dictionary-cache-alist*))
	  result))))



;1;; Presenting the data*

(defmacro 4current-font *(stream)
  #+LISPM `(send ,stream :send-if-handles :current-font)
  #-LISPM nil)

#+SYMBOLICS
(defvar 4*current-char-style** '(NIL NIL NIL))

(defmacro 4set-current-font *(stream new-font)
  #+EXPLORER  `(send ,stream :send-if-handles :set-current-font ,new-font t)
  #+SYMBOLICS `(setq *current-char-style* (if (equal ,new-font 0) '(:FIX :ROMAN :NORMAL) (si:backtranslate-font ,new-font)))
  #-(or EXPLORER SYMBOLICS) nil)

(defsetf 4current-font *set-current-font)

(defun 4draw-label *(num sub subnum txt &optional nofonts)
  "2Draw a section number label and the (indented) text to go along with it.*"
  (let* ((width (cond (subnum 10)
		      (sub 6)
		      (t 4)))
	 (font (unless nofonts (current-font *standard-output*))))
    (and font (set-current-font *standard-output* FONTS:CPTFONTB))
    (when (equal num "30*") (setq num nil)) ;1 I think this is a bogus db entry.*
    (cond ((equal subnum "31*")
	   (format t "~&  ~v<~@3a~@3a~@3a~> : " width num sub subnum))
	  (subnum
	   (format t "~&  ~v<~@3a~> : " width subnum))
	  ((equal sub "3a*")
	   (format t "~&   ~v<~@3a~@3a~> : " width num sub))
	  (sub
	   (format t "~&   ~v<~@3a~> : " width sub))
	  (num
	   (format t "~&  ~v<~@3a~> : " width num))
	(t (format t "~&  ~vT: " width)))
    (and font (set-current-font *standard-output* font))
    (write-backspace-string txt nil (+ (if font 9 4) width) nil nofonts))
  (terpri))


(defun 4draw-record *(record &optional last-record-type last-record-pos nofonts)
  "2Record is an element of the BODY slot of a DICT-ENTRY structure; LAST-RECORD-TYPE is the car of the previous record drawn.
  (That is needed because a :DEF after a :LABEL shouldn't print it's label numbers).*"
  (let* ((type (pop record)))
    (values
      type
      (ecase type
	(:VARIANT
	 (let* ((name (pop record))
		(dots (pop record))
		(accents (pop record))
		(levels (pop record))
		(name2 (apply-dots (apply-accents name accents) dots)))
	   (declare (ignore levels))		;1 ##*
	   (write-backspace-string (string-append "3 also *" #\( #\Backspace #\Y name2 #\) #\Backspace #\Y)
				   nil nil name nofonts)
	   nil))
	(:PRONUNCIATION
	 (princ "3 /*")
	 (write-backspace-string (car record) nil 5 nil nofonts)
	 (princ "3/ *")
	 nil)
	(:ETYMOLOGY
	 (format t "3~&  [*")
	 (write-backspace-string (car record) nil 5 nil nofonts)
	 (princ "3] *")
	 nil)
	(:SYNONYM
	 (format t "3~&   *")
	 (write-backspace-string (car record) nil 8 nil nofonts)
	 nil)
	(:LABEL
	 (let* ((num (pop record))
		(sub (pop record))
		(subnum (pop record))
		(txt (pop record)))
	   (draw-label num sub subnum txt nofonts)
	   nil))
	(:DEF
	 (let* ((num (pop record))
		(sub (pop record))
		(subnum (pop record))
		(pos (parse-part-of-speech (pop record)))
		(txt (pop record)))
	   (when (and pos (not (eq pos last-record-pos)))
	     (format t "3~&~5t*")
	     (write-backspace-string (string-append #\( #\Backspace #\X pos #\) #\Backspace #\X) nil nil nil nofonts)
	     ;(setq txt (string-append #\( #\Backspace #\X pos #\) #\Backspace #\X "3:  *" txt))
	     )
	   (if (eq last-record-type :LABEL)
	       (draw-label (if num "" nil) (if sub "" nil) (if subnum "" nil) txt nofonts)
	       (draw-label num sub subnum txt nofonts))
	   pos))
	(:XREF
	 (let* ((word (pop record))
		(super (pop record))
		(sub (pop record))
		(format-string (pop record))
		(word2 (pop record)))
	   (format t "3~&   *")
	   (write-backspace-string (format nil format-string word word2))
	   (when super (write-backspace-string (format nil "3 ~A*" super)))
	   (when sub (write-backspace-string (format nil "3 ~A*" sub)))
	   nil))
	(:RUN-ON
	 (let* ((name (pop record))
		(dots (pop record))
		(accents (pop record))
		(pos1 (pop record))
		(pos2 (pop record))
		(pos1-pref (pop record))
		(name2 (apply-dots (apply-accents name accents) dots)))
	   (format t "3~&   *")
	   (write-backspace-string (string-append #\( #\Backspace #\Y name2 #\) #\Backspace #\Y #\Space
						  #\( #\Backspace #\X (or pos1 "") #\) #\Backspace #\X)
				   nil nil name nofonts)
	   (when pos2
	     (write-backspace-string (string-append (if pos1-pref "3(or *" "3or *")
						    #\( #\Backspace #\X pos2 #\) #\Backspace #\X)
				     nil nil nil nofonts))
	   pos1))
	))))


(defun 4draw-dict-definition *(entry &optional nofonts)
  "2Format the given dictionary entry onto *standard-output*.*"
  (let* ((name (dict-entry-name entry))
	 (accents (dict-entry-accents entry))
	 (dots (dict-entry-dots entry))
	 (name2 (apply-dots (apply-accents name accents) dots)))
    (format t "3~2&  *")
    (when (dict-entry-homonym entry) (format t "3~D  *" (dict-entry-homonym entry)))

    (let* ((pos1 (or (dict-entry-part-of-speech entry) ""))
	   (pos2 (dict-entry-alt-part-of-speech entry)))
      (write-backspace-string (string-append #\( #\Backspace #\Y name2 #\) #\Backspace #\Y #\Space
					     #\( #\Backspace #\X pos1 #\) #\Backspace #\X)
			      nil nil name nofonts)
      (when pos2
	(write-backspace-string (string-append (if (dict-entry-pos-preferred-p entry) "3(or *" "3or *")
					       #\( #\Backspace #\X pos2 #\) #\Backspace #\X)
				nil nil nil nofonts))
      (let* ((last-type nil)
	     (last-pos pos1))
	(dolist (record (dict-entry-body entry))
	  (multiple-value-bind (lt lp) (draw-record record last-type last-pos nofonts)
	    (setq last-type lt last-pos (or lp last-pos))))))))


(defun 4present-dict-definition *(word &optional nofonts print-after)
  "2Look up the definitions of WORD in the online dictionary, and format them to *standard-output*.*"
  (let* ((defs (dictionary-get-def word)))
    (when print-after (write-string print-after *query-io*))
    (dolist (def defs)
      (draw-dict-definition def nofonts))))

#+LISPM
(defun 4apply-accent *(char accent)
  "2Given a character and an accent keyword, return a new character which has that accent, or the original char if there isn't one.*"
  (let* ((x (position accent '(:UMLAUT :ACCENT :GRAVE :CARET) :test #'eq))
	 (y (position char "3AaEeIiOoUuYy*" :test #'char=))
         (array '#2a((#\3* #\ #\3* #\ #\3* #\ #\ #\ #\ #\ nil nil)
                     (#\3* #\ #\3* #\ #\ #\ #\3* #\ #\ #\ #\ #\)
                     (#\3* #\ #\3* #\ #\3* #\ #\3* #\ #\ #\ nil nil)
                     (#\ #\ #\3* #\ #\ #\ #\ #\ #\ #\ nil nil))))
    (or (and x y
	     (not (eq *overstrike-char-mappings* *overstrike-ascii-char-mappings*))
	     (aref array x y))
	char)))

#-LISPM
(defun 4apply-accent *(char accent)
  (declare (ignore accent))
  char)

(defun 4apply-accents *(string accent-spec)
  "2Returns a new string, which has the accents inserted.*"
  (setq string (copy-seq string))
  (dolist (spec accent-spec)
    (let* ((pos (car spec))
	   (type (cdr spec)))
      (setf (char string pos) (apply-accent (char string pos) type))))
  string)

(defun 4apply-dots *(string dots-spec)
  "2Returns a new string, which has the dots inserted.*"
  (let* ((new-str (make-string (+ (length string) (length dots-spec))))
	 (out-index 0))
    (dotimes (i (length string))
      (setf (char new-str out-index) (char string i))
      (incf out-index)
      (when (eql i (car dots-spec))
	(setf (char new-str out-index) #+LISPM #\Center-Dot #-LISPM #\.)
	(incf out-index)
	(pop dots-spec)))
    new-str))


(defvar 4*unknown-backspace-codes* *() "2For debugging - these are the codes we saw and ignored.*")

2#+Comment*
(defun 4draw-pron *()
  "2for debugging - draw all of the Pronunciation strings in the cache.*"
  (dolist (x *dictionary-cache-alist*)
    (dolist (def (cdr x))
      (dolist (cons (dict-entry-body def))
	(when (eq (car cons) :PRONUNCIATION)
	  (show-bs-string (second cons))
	  (format t "3~30t*")
	  (write-backspace-string (second cons))
	  )))))

2#+Comment*
(defun 4show-bs-string *(str)
  "2For debugging - show where the backspace codes are, and what they turn in to.*"
  (fresh-line)
  (let* ((last 0))
    (do* ((i 0 (1+ i)))
	 ((>= i (length str))
	  (set-current-font *standard-output* 0)
	  (unless (= last (length str)) (write-string (subseq str last))))
      (cond ((char= #\Backspace (char str i))
	     (set-current-font *standard-output* 0)
	     (unless (= last (1- i)) (write-string (subseq str last (1- i))))
	     (set-current-font *standard-output* FONTS:CPTFONTBI t)
	     (write-string (string-append (char str (1- i)) #\Left-Arrow (char str (1+ i))))
	     (setq last (incf i 2)))))))


(defun 4bs-codes-to-fonts *(c1 c2 greek-p mini-p italic-p bold-p super-p sub-p subitalic-p superitalic-p shift font)
  "2Internal function of WRITE-BACKSPACE-STRING.*"
  ;1; '(' <bs> <c>  = turn font on*
  ;1; ')' <bs> <c>  = turn font off*
  (let* ((state (char= c1 #\()))
    (setq shift 0)
    (ecase c2
      (#\G (setq greek-p state))
      (#\R nil)  ;1 APL*
      (#\Q nil)  ;1 symbol*
      (#\M (setq mini-p state))
      (#\X (setq italic-p state))
      (#\Y (setq bold-p state))
      (#\A (setq super-p state		shift (if state -5 +5)))
      (#\B (setq superitalic-p state	shift (if state -5 +5)))
      (#\I (setq sub-p state		shift (if state +5 -5)))
      (#\J (setq subitalic-p state	shift (if state +5 -5)))
      )
    (setq font (cond (greek-p FONTS:ILGREEK10)
		     (mini-p FONTS:HL10B)
		     ((and bold-p italic-p) FONTS:HL12BI)
		     (italic-p FONTS:HL12I)
		     (bold-p FONTS:HL12B)
		     (sub-p FONTS:HL10)
		     (subitalic-p FONTS:TR10I)
		     (superitalic-p FONTS:TR10I)
		     (super-p FONTS:HL10)
		     (t FONTS:HL12)))
    (values greek-p mini-p italic-p bold-p super-p sub-p subitalic-p superitalic-p shift font)))


(defun 4collapse-symbols *(string)
  "2Modifies string, replacing all occurences of the sequence *'(' 'backspace' 'Q' <word> ')' 'backspace' 'Q'2 with a single character,
  where the character is determined from what the <word> is.
  A new string is returned, but the passed-in string is destroyed nonetheless.*"
  (do* ((len (length string))
	(open-target  #.(string-append #\( #\Backspace #\Q))
	(close-target #.(string-append #\) #\Backspace #\Q))
	(pos (lisp:search open-target string) (lisp:search open-target string :start2 (1+ pos))))
       ((null pos)
	(string-right-trim #.(string (code-char 0)) string))
    (let* ((end (lisp:search close-target string :start2 pos))
	   (name (subseq string (+ pos 3) end))
	   (symbol (symbol-name-to-char name)))
      (setf (char string pos) symbol)
      (sys:copy-array-portion string (+ end 3) len string (1+ pos) len)
      )))


#+LISPM
(defun 4symbol-name-to-char *(symbol-name)
  (when (stringp symbol-name) (setq symbol-name (intern (string-upcase symbol-name) "3KEYWORD*")))
  (case symbol-name
    (:breve			#\Down-Horseshoe)
    ((:cedilla :cidilla)	#\Cedilla)
    (:hachek			#\And-Sign)
    (:macron-tilda		#\~)
    (:sub-breve			#\Up-Horseshoe)
    (:sub-dot			#\Center-Dot)
    #+TI ((:cedilla :cidilla)	#\Cedilla)	;1 ##  It doesn't matter what these characters are, so long as their char-codes are less*
    #+TI (:sub-diaeresis	#\Diaresis)	;1 ##  than 32 or greater than 127 (out of the ASCII range).  These are just dummy*
    #+TI (:sub-macron		#\Macron)	;1 ##  tags which are used for a unique key in the char-mappings table.  For Symbolics,*
    #+TI (:sup-circle		#\Degree-Sign)	;1 ##  find some unused characters to use for these...*
    (t				#\Null)))

#-LISPM
(defun 4symbol-name-to-char *(symbol-name)
  (declare (ignore symbol-name))
  #\Null)


(defun 4collapse-backspaces *(string)
  "2Modifies string, replacing all occurences of the character-sequence '|' 'backspace' 'B' with a single 'backspace'.
  A new string is returned, but the passed-in string is destroyed nonetheless.*"
  (let* ((done-once nil))
    (do* ((len (length string))
	  (pos (lisp:search #.(string-append #\| #\Backspace #\B) string)
	       (lisp:search #.(string-append #\| #\Backspace #\B) string :start2 (1+ pos))))
	 ((null pos)
	  (if done-once
	      (string-right-trim #.(string (code-char 0)) (collapse-backspaces string))
	      string))
      (setq done-once t)
      (setf (char string pos) #\Backspace)
      (sys:copy-array-portion string (+ pos 3) len string (1+ pos) len))))


(defun 4write-backspace-string *(string &optional output-width indent typeout-item nofonts)
  "2Draw the string on standard output, interpreting <c1><Backspace><c2> codes as font-changes and special characters.
  Also wraps lines and indents.*"
  ;1; 5------------------------------**
  ;1; 5This function is way too big and hairy.**
  ;1; 5------------------------------**
  (setq string (collapse-backspaces (collapse-symbols (copy-seq string))))
  (let* ((len (length string))
	 (use-fonts (and (not nofonts)
			 #+EXPLORER (send *standard-output* :operation-handled-p :set-current-font)
			 #+SYMBOLICS nil  ;1 ## We get this for free on symbolics, I think...*
			 ))
	 (last-font-change-pos 0)
	 (last-word-break-pos 0)
	 super-p greek-p sub-p subitalic-p superitalic-p mini-p italic-p bold-p
	 (output-char-pos (or (send *standard-output* :send-if-handles :read-cursorpos :character) 0))
	 (safe-to-break t)
	 )
    (or output-width (setq output-width (or #+EXPLORER (send *standard-output* :send-if-handles :size-in-characters)
					    ;1; ## There must be a #+SYMBOLICS thing to do here....*
					    70)))
    (and use-fonts (set-current-font *standard-output* fonts:hl12))
    (do* ((i 0 (1+ i))
	  (font (and use-fonts (current-font *standard-output*)))
	  shift)
	 ((>= i len)
	  #-SYMBOLICS (write-string string *standard-output* :start (max last-font-change-pos last-word-break-pos))
	  #+SYMBOLICS (scl:with-character-style (*current-char-style*)
			(write-string string *standard-output* :start (max last-font-change-pos last-word-break-pos)))
	  (and use-fonts (set-current-font *standard-output* 0)))
      
      (labels (4(maybe-break-line* ()
		 (let* ((last-output (max last-font-change-pos last-word-break-pos))
			(subseq-length (- i last-output)))
		   (when (and safe-to-break
			      (or (char= (char string i) #\Newline)
				  (>= (+ output-char-pos subseq-length) output-width)))
		     (setq output-char-pos (or indent 0))
		     (if indent (format t "3~%~vT*" indent) (terpri)))))
	       
	       4(dump-queue* (&optional (i i))
		 (let* ((last-output (max last-font-change-pos last-word-break-pos)))
		   (maybe-break-line)
		   (when (char= (char string last-output) #\Newline) (incf last-output))
		   #-SYMBOLICS (write-string string *standard-output* :start last-output :end i)
		   #+SYMBOLICS (scl:with-character-style (*current-char-style*)
				 (write-string string *standard-output* :start last-output :end i))
		   (incf output-char-pos (- i last-output))
		   (incf i)))
	       
	       #+LISPM
	       4(mouse-sensitize* (in-font)
		 #+EXPLORER
		 (let* ((type 'DICTIONARY-WORD)
			(item (string-trim "3 *" (or typeout-item (subseq string last-font-change-pos (- i 3)))))
			(x (send *standard-output* :cursor-x))
			(y (send *standard-output* :cursor-y))
			(width (send *standard-output* :string-length string last-font-change-pos (- i 3) nil in-font))
			(height (send *standard-output* :line-height)))
		   (decf x width)
		   (send *standard-output* :primitive-item type item x y (+ x width) (+ y height)))
		 
		 #+SYMBOLICS nil	;1; ## We should slam down a presentation here....  I don't know how.*
		 ))
	(cond ((char= (char string i) #\Backspace)
	       ;1;*
	       ;1; We are in the midst of some kind of control code - maybe font change, maybe special character.*
	       ;1;*
	       (let* ((c1 (char string (1- i)))
		      (c2 (char string (1+ i)))
		      (wasbold (when (or bold-p mini-p) font)))
		 (cond ((and (member c1 '(#\( #\)) :test #'char=)
			     (alpha-char-p c2))
			;1;*
			;1; This is a font-change code.  Dump what is queued, then mung the current font.*
			;1;*
			(dump-queue (1- i))
			(incf i 2)
			(multiple-value-setq (greek-p mini-p italic-p bold-p super-p sub-p subitalic-p superitalic-p shift
					      font)
					     (bs-codes-to-fonts c1 c2 greek-p mini-p italic-p bold-p super-p sub-p
								subitalic-p superitalic-p shift font))
			(and use-fonts (set-current-font *standard-output* font))
			#+LISPM
			(unless (or (null shift) (zerop shift))
			  ;1; ## how to do increment-cursorpos on Symbolics??*
			  (and use-fonts (send *standard-output* :increment-cursorpos 0 shift :pixel)))
			;1;*
			;1; Add a mouse-sensitive item around bold words.*
			#+LISPM
			(when (and wasbold (not (or bold-p mini-p))
				   #+EXPLORER (send *standard-output* :operation-handled-p :primitive-item)
				   #+SYMBOLICS t	;1 ## need a Genera way to tell if this is a window or a file/buffer stream.*
				   )
			  (mouse-sensitize wasbold))
			(setq last-font-change-pos i)
			)
		       
		       ((char= c1 c2)			;1; <c> <bs> <c>  = overstrike <c> (to get bold).*
			(dump-queue (1- i))
			(and use-fonts (set-current-font *standard-output* fonts:HL12B))
			(write-char c2)
			(and use-fonts (set-current-font *standard-output* font))
			(setq last-font-change-pos (incf i 2)))
		       
		       ((char= c1 #\/)				;1; '/' <bs> <c>  = print <c> in greek.*
			(dump-queue (1- i))
			(and use-fonts (set-current-font *standard-output* fonts:ILGREEK10))
			(write-char c2)
			(and use-fonts (set-current-font *standard-output* font))
			(setq last-font-change-pos (incf i 2)))
		       
		       (t
			;1; This is a special character.*
			;1;*
			(let* ((cons (get-overstrike-mapping c1 c2))
			       (char (cdr cons)))
			  (dump-queue (1- i))
			  (write-backspace-symbol-char c1 c2 char font use-fonts string)
			  (setq last-font-change-pos (incf i 2))))))
	       (setq safe-to-break nil))
	      
	      ((or (char= (char string i) #\Space)
		   (char= (char string i) #\Newline))
	       (setq safe-to-break t)
	       (dump-queue)
	       (setf last-word-break-pos i))
	      )))))


(defun 4write-backspace-symbol-char *(c1 c2 char font use-fonts string)
  "2Internal function of WRITE-BACKSPACE-STRING.*"
  (let* (#+EXPLORER  (alu w:alu-transp)
	 #+SYMBOLICS (alu tv:alu-ior)
	 )
    (cond ((characterp char) (write-char char))
	  
	  ((stringp char)
	   (cond #+EXPLORER
		 ((and use-fonts (string-equal char "3nj*"))
		  ;1; If we can position pixel-by-pixel, turn the "nj" code into one character (as in le5NG*then).*
		  ;1; ## don't know how to do pixel positioning on Symbolics.*
		  (let* ((x (send *standard-output* :cursor-x))
			 (y (send *standard-output* :cursor-y))
			 w)
		    (write-char #\n)
		    (setq w (round (- (send *standard-output* :cursor-x) x) 2))
		    (send *standard-output* :string-out-explicit "3j*" (+ x w) y nil nil font alu)))
		 (t #-SYMBOLICS (write-string char)
		    #+SYMBOLICS (scl:with-character-style (*current-char-style*) (write-string char)))))
	  
	  (char (error "2~S is not a string or a character.*" char))
	  
	  #+EXPLORER
	  ;1; ## don't know how to do pixel positioning on Symbolics.*
	  ((and (null char) use-fonts)
	   (let* ((c1-width (send *standard-output* :string-length (string c1)))
		  (c2-width (send *standard-output* :string-length (string c2)))
		  (cx (send *standard-output* :cursor-x))
		  (cy (send *standard-output* :cursor-y))
		  (top (if (lower-case-p c1)
			   (- cy (- (tv:font-baseline font) 5))
			   (- cy (tv:font-baseline font))))
		  (mid (- cy (floor (* 2/3 (tv:font-baseline font))))))
	     (write-char c1)
	     (incf cx (- (round (+ c1-width c2-width) 2) c2-width))
	     (case c2
	       (#\.				;1 dot-over*
		(send *standard-output* :string-out-explicit #.(string #\Center-Dot) cx mid nil nil font alu))
	       (#\-				;1 line-over*
		(send *standard-output* :string-out-explicit "3-*" cx top nil nil font alu))
	       ((#\_ #\Macron)			;1 line-under*
		(send *standard-output* :string-out-explicit "3_*" cx cy nil nil font alu))
	       (#\Down-Horseshoe		;1 breve over*
		(send *standard-output* :string-out-explicit #.(string #\Down-Horseshoe) cx top nil nil font alu))
	       (#\Cedilla			;1 cedilla under*
		(send *standard-output* :string-out-explicit #.(string #\Cedilla) cx cy nil nil font alu))
	       (#\Section-Symbol		;1 hachek: circumflex upside-down over*
		(send *standard-output* :string-out-explicit #.(string #\Or-Sign) cx mid nil nil font alu))
	       (#\~				;1 macron-tilda: line and tilda over*
		(send *standard-output* :string-out-explicit "3~*" cx mid nil nil font alu))
	       (#\Up-Horseshoe			;1 sub-breve*
		(send *standard-output* :string-out-explicit #.(string #\Up-Horseshoe) cx (+ cy 6) nil nil font alu))
	       (#\Diaresis			;1 sub-diaresis: umlaut under*
		(send *standard-output* :string-out-explicit #.(string #\Diaresis) cx (+ cy 8) nil nil font alu))
	       (#\Center-Dot			;1 sub-dot*
		(send *standard-output* :string-out-explicit #.(string #\Center-Dot) cx (+ cy 6) nil nil font alu))
	       (#\Degree-Sign			;1 sup-circle*
		(send *standard-output* :string-out-explicit #.(string #\Degree-Sign) cx mid nil nil font alu))
	       
	       (t (write-char c2)
		  ;1; Record it as unknown.*
		  (push (cons (string-append c1 c2) string) *unknown-backspace-codes*)
		  (tv:beep) (format *query-io* "3~&Unknown backspace code ~C-~C*" c1 c2))
	       )))
	  (t (write-char c1)
	     (write-char c2)
	     ))))

;(defun 4cn *(c)
;  (dolist (x sys:xr-special-character-names)
;    (when (char= c (cdr x)) (print x))))

(defvar 4*overstrike-ascii-char-mappings**
	'(("*o" . #\.) ("-/" . #\\) ("<(" . "3<<*") (")>" . "3>>*") ("-X" . #\x) ("-:" . #\/) ("-m" . "--") ("-3" . "3...*")
	  ("(|" . #\[) ("|(" . #\]) ("'\"" . #\`) ("-n" . #\-) ("=\"" . #\$) ("|B" . #\Backspace) ("n_" . "nj"))
  "2The table mapping backspace codes to special characters, using only ASCII characters.*")

(defvar 4*overstrike-char-mappings**
	#+EXPLORER
        '(("*o" . #\Center-Dot) ("+=" . #\Plus-Minus) ("-/" . #\\) ("<(" . #\Angle-Quotation-Left)
	  (")>" . #\Angle-Quotation-Right) ("c|" . #\Cent) ("|S" . #\Section-Symbol) ("'o" . #\Degree-Sign)
	  ("+=" . #\Plus-Minus-Sign) ("|q" . #\Paragraph-Symbol) ("-X" . #\Multiplication-Sign) ("-:" . #\Division-Sign)
	  (">A" . #\) ("<A" . #\) (";C" . #\) (">E" . #\) ("<E" . #\) ("~E" . #\) ("I:" . #\) ("~O" . #\)
	  ("\"U" . #\) ("<a" . #\) ("~a" . #\) ("a:" . #\) ("\"a" . #\) (";c" . #\) (">e" . #\) ("<e" . #\)
	  ("~e" . #\) ("\"e" . #\) ("i)" . #\) ("i(" . #\) ("~i" . #\) ("\"i" . #\) ("o)" . #\) ("o(" . #\)
	  ("~o" . #\) ("o:" . #\) ("\"o" . #\) (">u" . #\) ("u)" . #\) ("u:" . #\) ("~u" . #\) ("y(" . #\)
	  ("|o" . #\) ("-m" . "--") ("-3" . #\Soft-Hyphen) ("(|" . #\[) ("|(" . #\]) ("'\"" . #\`) ("-n" . #\-)
	  ("=\"" . #\$) ("|B" . #\Backspace) ("A~" . #\) ("N~" . #\) ("n~" . #\) ("n_" . "nj")
	  (#.(string-append #\a #\Degree-Sign) . #\) (#.(string-append #\A #\Degree-Sign) . #\)
	  (#.(string-append #\a #\And-Sign) . #\) (#.(string-append #\e #\And-Sign) . #\)
	  (#.(string-append #\u #\And-Sign) . #\) (#.(string-append #\o #\And-Sign) . #\)
	  (#.(string-append #\A #\And-Sign) . #\) (#.(string-append #\E #\And-Sign) . #\)
	  (#.(string-append #\U #\And-Sign) . #\) (#.(string-append #\O #\And-Sign) . #\)
	  (#.(string-append #\I #\And-Sign) . #\)
	  ("|-" . #\Up-Arrow)	;1 dagger.*
	  ("|=" . #\Diamond)	;1 double-dagger.*
	  )
	#+SYMBOLICS
        '(("*o" . #\Center-Dot) ("+=" . #\Plus-Minus) ("-/" . #\\) (">A" . #\) ("<A" . #\) (";C" . #\) (">E" . #\)
	  ("<E" . #\) ("~E" . #\) ("I:" . #\) ("~O" . #\) ("\"U" . #\) ("<a" . #\) ("~a" . #\) ("a:" . #\)
	  ("\"a" . #\) (";c" . #\) (">e" . #\) ("<e" . #\) ("~e" . #\) ("\"e" . #\) ("i)" . #\) ("i(" . #\)
	  ("~i" . #\) ("\"i" . #\) ("o)" . #\) ("o(" . #\) ("~o" . #\) ("o:" . #\) ("\"o" . #\) (">u" . #\)
	  ("u)" . #\) ("u:" . #\) ("~u" . #\) ("y(" . #\) ("|o" . #\) ("-m" . "--") ("+=" . #\Plus-Minus-Sign)
	  ("(|" . #\[) ("|(" . #\]) ("'\"" . #\`) ("-n" . #\-) ("=\"" . #\$) ("|B" . #\Backspace) ("A~" . #\) ("N~" . #\)
	  ("n~" . #\) ("n_" . "nj")
	  
	  (#.(string-append #\a #\And-Sign) . #\) (#.(string-append #\e #\And-Sign) . #\)
	  (#.(string-append #\u #\And-Sign) . #\) (#.(string-append #\o #\And-Sign) . #\)
	  (#.(string-append #\A #\And-Sign) . #\) (#.(string-append #\E #\And-Sign) . #\)
	  (#.(string-append #\U #\And-Sign) . #\) (#.(string-append #\O #\And-Sign) . #\)
	  (#.(string-append #\I #\And-Sign) . #\)
	  
	  ("|-" . #\Up-Arrow)	;1 dagger.*
	  ("|=" . #\Diamond)	;1 double-dagger.*

	  ;1; ## Don't know the character-names for these, if in fact they exist:*
	  ;1;*
	  ;("<(" . #\Angle-Quotation-Left)
	  ;(")>" . #\Angle-Quotation-Right)
	  ;("c|" . #\Cent)
	  ;("|S" . #\Section-Symbol)
	  ;("'o" . #\Degree-Sign)
	  ;("|q" . #\Paragraph-Symbol)
	  ;("-X" . #\Multiplication-Sign)
	  ;("-:" . #\Division-Sign)
	  ;("-3" . #\Soft-Hyphen)
	  ;(#.(string-append #\a #\Degree-Sign) . #\)
	  ;(#.(string-append #\A #\Degree-Sign) . #\)
	  ;1;*
	  ;1; ## So use these instead:*
	  ;1;*
	  ("*o" . #\.) ("-/" . #\\) ("<(" . "<<") (")>" . ">>") ("-X" . #\x) ("-:" . #\/) ("-m" . "--") ("-3" . "...")
	  )
	#-(or EXPLORER SYMBOLICS)	;1 Simple ASCII mappings...*
	*overstrike-ascii-char-mappings*
  "2The table mapping backspace codes to special characters, including whatever non-ascii chars this system supports.*")


(defun 4get-overstrike-mapping *(c1 c2)
  "2Like ASSOC.  Given two characters, look up their special-char mapping in the **overstrike-char-mappings*2 table.*"
  (declare (string-char c1 c2))
  (dolist (cons (the list *overstrike-char-mappings*))
    (when (and (char= c1 (char (the simple-string (car cons)) 0))
	       (char= c2 (char (the simple-string (car cons)) 1)))
      (return cons))))




;1;; Zmacs interface*

#+LISPM
(defun 4zwei:dictionary-1 *(word &optional nofonts)
  (setq word (string-trim '(#\Space #\Tab #\Newline) word))
  ;1; Hack for Meta-X Dict Search mousable output - discard everything after a semicolon.*
  (let* ((s (position #\; word :test #'char=)))
    (when s (setq word (string-right-trim '(#\Space #\Tab) (subseq word 0 s)))))
  (nsubstitute #\Space #\Newline (the string word) :test #'char-equal)
  (nsubstitute #\Space #\Tab     (the string word) :test #'char-equal)
  (cond ((string-equal "" word) (zwei:barf))
	(t
	 (zwei:typein-line (string-append "2Looking for *"#\" word #\" "2 in the dictionary...*"))
	 (present-dict-definition word nofonts "2 done.*"))))

#+LISPM
(zwei:defcom 4zwei:com-dictionary* "2Look up a word in the online dictionary.*" (KM)
  (multiple-value-bind (ignore ignore interval)
		       (zwei:completing-read-from-mini-buffer "2Word to look up:*"
			 (mapcar #'(lambda (x) (cons (car x) (car x))) *dictionary-cache-alist*)
			 t nil (format nil "2There are ~D definition~:P in the cache.*" (length *dictionary-cache-alist*)))
    (zwei:dictionary-1 (zwei:string-interval interval)
		       #+SYMBOLICS t  ;1 ## no fonts on symbolics for now...*
		       ))
  ZWEI:DIS-NONE)

(zwei:defcom 4zwei:com-dictionary-into-buffer*
	     "2Look up a word in the online dictionary and grind its definition into the current buffer.
 With a numeric argument, do it using only ASCII characters.*" ()
  (multiple-value-bind (ignore ignore interval)
		       (zwei:completing-read-from-mini-buffer "2Word to look up:*"
			 (mapcar #'(lambda (x) (cons (car x) (car x))) *dictionary-cache-alist*)
			 t nil (format nil "2There are ~D definition~:P in the cache.*" (length *dictionary-cache-alist*)))
    (zwei:with-bp (bp (zwei:point) :moves)
      (ticl:let-if zwei:*numeric-arg-p* ((*overstrike-char-mappings* *overstrike-ascii-char-mappings*))
	(with-open-stream (*standard-output* (zwei:interval-stream-into-bp bp))
	  (zwei:dictionary-1 (zwei:string-interval interval)
			     #+SYMBOLICS t  ;1 ## no fonts on symbolics for now...*
			     )))))
  ZWEI:DIS-TEXT)

#+SYMBOLICS	;1 Symbolics-Zmacs doesn't have this....*
ZWEI:
(defun 4current-word* (point)
  (let*((*mode-word-syntax-table* *spell-word-syntax-table*)
	(char-before (bp-char-before point))
	(char-after (bp-char point))
	(syntax-before (and (not (bp-= point (interval-first-bp *interval*)))
			    (char-syntax char-before *spell-word-syntax-table*)))
	(syntax-after (and (not (bp-= point (interval-last-bp *interval*)))
			   (char-syntax char-after *spell-word-syntax-table*)))
	start-of-word-bp)
    (cond ((and (null syntax-before)
		(not (eql syntax-after word-alphabetic)))
	   (barf "2There is no current word; point is at the beginning.*"))
	  ((and (not (eql syntax-before word-alphabetic))
		(eql syntax-after word-alphabetic))
	   ;1; We're at the beginning of a word.*
	   (setq start-of-word-bp point))
	  (t
	   ;1; All other cases are handled by going backward over a word,*
	   ;1; and then going forward over a word.*
	   (setq start-of-word-bp (forward-word (point) -1))
	   (unless start-of-word-bp
	     ;1; This can happen if syntax-before is word-delimiter.*
	     (barf "2There is no current word; point is at the beginning.*"))))
    start-of-word-bp))



#+LISPM
(zwei:defcom 4zwei:com-dictionary-word-after-point* "2Look up the word after point in the online dictionary.*" ()
  (let* ((zwei:*mode-word-syntax-table* (if (boundp #+EXPLORER  'zwei:*spelling-word-syntax-table*
					            #+SYMBOLICS 'zwei:*spell-word-syntax-table*)
					    #+EXPLORER  zwei:*spelling-word-syntax-table*
					    #+SYMBOLICS zwei:*spell-word-syntax-table*
					    zwei:*word-syntax-table*))
	 (bp1 (zwei:current-word (zwei:point)))
	 (bp2 (zwei:forward-word bp1 1 t))
	 (word (zwei:string-interval bp1 bp2)))
    (zwei:dictionary-1 word))
  ZWEI:DIS-NONE)

#+LISPM
(zwei:defcom 4zwei:com-dictionary-search* "2List all words in the dictionary which begin with a given string.*" ()
  (multiple-value-bind (ignore ignore interval)
		       (zwei:edit-in-mini-buffer zwei:*mini-buffer-comtab* "" 0
						 (if zwei:*numeric-arg-p*
						     '("3Regular Expressions:*")
						     '("3Prefixes:*")))
    (let* ((string (zwei:string-interval interval)))
      (nsubstitute #\Space #\- (the string string) :test #'char-equal)
      (let* ((i 0)
	     (ncols (1- (floor (send *standard-output* :size-in-characters) 20)))
	     (words (dictionary-list-words (list string) (if zwei:*numeric-arg-p*
							     :regexp
							     t))))
	(let* ((font (current-font *standard-output*)))
	  (fresh-line *standard-output*)
	  (set-current-font *standard-output* FONTS:HL12)
	  (unwind-protect
	      (dolist (word words)
		;1; Make the words matched by this be on the completion list.*
		(unless (member word *dictionary-cache-alist* :test #'string-equal :key #'car)
		  (push (cons word nil) *dictionary-cache-alist*))
		(when (>= i ncols) (terpri) (setq i 0))
		#+EXPLORER  (format t "3~vM~20,20T*" 'DICTIONARY-WORD word)
		#+SYMBOLICS (scl::with-character-style (*current-char-style*) (format t "3~a~20,20T*" word))
		(incf i))
	    (set-current-font *standard-output* font))))))
  ZWEI:DIS-NONE)

#+LISPM
(zwei:defcom 4zwei:com-dictionary-reset* "2Reset the connection to the dictionary server in case it's hosed.*" ()
  (open-dict-server-stream)
  (format *query-io* "3~:|Dictionary server connection reset.*")
  ZWEI:DIS-NONE)

#+LISPM
(zwei:set-comtab ZWEI:*STANDARD-COMTAB*
		 '(#\Control-$ zwei:com-dictionary-word-after-point)
		 (zwei:make-command-alist '(zwei:com-dictionary
					     zwei:com-dictionary-into-buffer
					     zwei:com-dictionary-search
					     zwei:com-dictionary-reset)))

#+EXPLORER	;1 ## On symbolics, this should be a Presentation-Type definition of some kind...*
(w:add-typeout-item-type zwei:*typeout-command-alist*
			 DICTIONARY-WORD "2View Definition*" zwei:dictionary-1 t
			 "2Print the definitions of this word.*")
