;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:SYSTEM -*-

;;; File "DESCRIBE-SYMBOL"
;;; Redefining SYS:DESCRIBE-SYMBOL, an internal function of DESCRIBE to be better.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 14 Jul 88  Jamie Zawinski    Created.
;;;


;;; This version of DESCRIBE-SYMBOL doesn't describe NIL, and it describes a DEFSTRUCT which the symbol names.


(defun describe-symbol (symbol)
  (cond ((null symbol)
	 (format t "~&NIL is both a symbol and the empty list.  It is magic.~%"))
	(t
	 (describe-basic-symbol symbol)
	 (when (and (boundp symbol) (not (keywordp symbol)))
	   (describe-1 (symbol-value symbol)))
	 (when (fboundp symbol)
	   (describe-symbol-function symbol))
	 (when (symbol-plist symbol)
	   (describe-symbol-plist symbol))
	 (unless (or (boundp symbol) (fboundp symbol) (symbol-plist symbol))
	   (format t "~%It has no value, definition or properties."))
	 (describe-symbol-defstruct symbol)
	 )))

(defun describe-basic-symbol (sym)
  (if (SYMBOL-PACKAGE sym)
      (format t "~%Symbol ~S is in ~A package." sym (package-name (symbol-package sym)))
      (format t "~%Symbol ~S is uninterned." sym))
  (let ((tem nil))
    (do-all-packages (p)
      (multiple-value-bind (s flag)
	  (find-symbol sym p)
	(when (and (not (eq :inherited flag)) (eq s sym) (not (eq p (symbol-package sym)))
		   (not (member p (package-used-by-list (symbol-package sym)) :test #'eq)))
	  (push p tem))))
    (when tem
      (format t "~% It is also interned in package~P ~{~A~^, ~}" (length tem) tem)))
  
  (when (and (boundp sym) (not (keywordp sym)))
    (let ((*print-level* 2)
	  (*print-length* 3)
	  (*print-pretty* t))
      (format t "~%The value of ~S is ~S" sym (symbol-value sym)))))

(defun describe-symbol-function (sym)
  (let ((*print-level* 2)
	(*print-length* 3))
    (let* ((macro (macro-function sym)))
      (format t "~%The ~A definition of ~S is ~S: "
	      (if macro "macro" "function")
	      sym
	      (or macro (symbol-function sym)))
      (zwei:print-arglist-internal (arglist sym) *standard-output*)
      (describe-1 (or macro (symbol-function sym))))))

(defun describe-symbol-plist (sym)
  (do ((pl (symbol-plist sym) (cddr pl))
       (*print-level* 2)
       (*print-length* 3))
      ((null pl))
    (format t "~%~S has property ~S: ~S" sym (car pl) (cadr pl))
    (describe-1 (cadr pl))))

(defun describe-symbol-defstruct (symbol)
  (let* ((desc (get symbol 'SYS:DEFSTRUCT-DESCRIPTION)))
    (when desc
      (format t "~&~%~S is the name of a structure.~%" symbol)
      (flet ((print-slot (slot val)
	       (unless (eq val SYS:DEFSTRUCT-EMPTY)
		 (format t "~&   ~A:~30t ~S~&" slot val))))
	(print-slot 'conc-name (defstruct-description-conc-name desc))
	(print-slot 'predicate (defstruct-description-predicate desc))
	(print-slot 'copier    (defstruct-description-copier desc))
	(print-slot 'printer   (defstruct-description-print desc))
	(print-slot 'include   (or (car (defstruct-description-include desc)) SYS:DEFSTRUCT-EMPTY)))
      ;;
      ;; Describe the constructors.
      ;;
      (let* ((constructors (defstruct-description-constructors desc)))
	(format t "~&It has ~D constructor~:P:" (length constructors))
	(dolist (cdesc constructors)
	  (let* ((name (if (consp cdesc) (car cdesc) cdesc)))
	    (format t "~&   ~S: " name)
	    (when (macro-function name) (princ "(MACRO): "))
	    (zwei:print-arglist-internal (arglist name) *standard-output*))))
      ;;
      ;; Describe the slots.
      ;;
      (let* ((slots (defstruct-description-slot-alist desc)))
	(format t "~&It has ~D slot~:P:" (length slots))
	(format t "~& slot: name: ~30Tdefault value: ~50Ttype:~70Taccessor:~%~%")
	(dolist (slot slots)
	  (let* ((name (car slot))
		 (slotd (cdr slot))
		 (number (defstruct-slot-description-number slotd))
		 (init (defstruct-slot-description-init-code slotd))
		 (type (defstruct-slot-description-type slotd))
		 (accessor (defstruct-slot-description-ref-macro-name slotd))
		 (read-only-p (defstruct-slot-description-read-only slotd))
		 (*print-pretty* t)
		 )
	    (format t "~&~5D  ~A ~30T~S ~50T~S ~70T~A"
		    number name
		    (if (eq init SYS:DEFSTRUCT-EMPTY) '- init)
		    (if (eq type SYS:DEFSTRUCT-EMPTY) '- type)
		    accessor)
	    (when read-only-p (format t "~85t read-only."))
	    (terpri)))))))
