function y = mm(li,ri,fl,freqs,disps,angles,min_mag,pixel,thresh,lscalogram,rscalogram)

% MWM(left,right,fl,freqs,disps,angles,min_mag,pixel,thresh,ls,rs)
%
%  -- MWM's stereo matcher
%
%	SCANLINES ONLY!!  No images yet
%
%	left -- Left image scanline
%	right - Right image scanline
%	fl    - focal length as multiple of pixel width
%	freqs - Set of frequencies to use [1 ./ (2:(len/4))]
%	disps - Set of disparities to check [0:50];
%	angles - Set of foreshortening angles (in degrees) to consider [0]
%	min_mag - Minimum magnitude value (must be > this) [0]
%	pixel - Save the error matrix for a pixel in global PIXEL [0]
%	thresh - Set cutoff for Fleet threshold [.05]
%	ls - initial left scalogram [will be computed]
%	rs - Initial right scalogram [will be computed]

global PIXEL

tic;
%%% Set defaults
num_sigma = 6;
num_periods = 4;
full_scalogram = 0;
percent_update = 10;

%%% Check arguments
if nargin < 3,
  error ('M:  must enter two signals to be matched and a focal length');
end;
if nargin < 4,
  freqs = 1 ./ [2:length(li)/num_periods];
%%  freqs = [num_periods/length(li):num_periods/(2*length(li)):.5];
end;
if nargin < 5,
  disps = [0:50];
end;
if nargin < 6,
  angles = [0];
end;
if nargin < 7,
  min_mag = 0;
end;
if nargin < 8,
  pixel = 0;
end;
if nargin < 9,
  thresh = .05;
end;
if nargin < 10,
  lscalogram = 0;
end;
if nargin < 11,
  rscalogram = 0;
end;

%%%  Check the input images, they better be the same size
[lr lc] = size(li);
[rr rc] = size(ri);

if lr ~= 1,
  li = li';
  [lr lc] = size(li);
end;
if rr ~= 1,
  ri = ri';
  [rr rc] = size(ri);
end;
if lr ~= rr | lc ~= rc,
  error ('M:  left and right scanlines must be same size!');
end;

if lr ~= 1,
  error ('M:  Sorry, cannot handle images yet');
end;

%%%  Run a quick a dirty global translation method to make sure the images
%%%  are in the right order.

sh = kuglin(li, ri);
if sh(2) ~= 0,
  [ 'WARNING: Y shift not zero! (' num2str(sh(2)) ')']
end;
if sh(1) < 0,
  'Images out of order, swapping left and right images'
  t = li;
  li = ri;
  ri = t;
end;

%%%  Make sure the focal length is a scalar

if (size(fl) ~= [1 1]),
  error ('M:  Focal length must be a scalar');
end;

%%%  Allocate space for the pixel matrix, if needed

if pixel ~= floor(pixel),
  ['WARNING!  Non-integer pixel ' num2str(pixel) ' mapped to ' ...
   num2str(floor(pixel))]
  pixel = floor(pixel);
end;

if pixel > 0,
  PIXEL = zeros(length(freqs),length(disps));
end;

%%%  Make sure the input frequencies are stored in the right order

freqs = -sort(-freqs);

num_freqs = length(freqs);
num_angles = length(angles);
tan_angles = tan(angles ./ (pi/180));
col_center = lc/2;
max_freq = max(freqs);

%%%  The crux of the algorithm is the computation of the scalogram, which
%%%  caches the outputs of Gabor filters at many scales

if size(lscalogram) ~= [1 1] & size(rscalogram) ~= [1 1],
  'Using input scalograms'
else
  'Computing Scalograms'
  lscalogram = scalogram (li, freqs, num_sigma, num_periods, full_scalogram);
  rscalogram = scalogram (ri, freqs, num_sigma, num_periods, full_scalogram);
end;

['Computing Fleet''s Filter, threshold is ' num2str(thresh)]
lfilter = fleet(lscalogram, freqs)' < thresh;
rfilter = fleet(rscalogram, freqs)' < thresh;
% lfilter = ones(size(lscalogram))';
% rfilter = lfilter;

%%%  We'll also cache as much of the frequency corrections as we can.  A
%%%  full cache would require num_disps * num_angles * num_freqs * len
%%%  (typically 500 * 90 * 128 * 512 for a single scanline), but we'll
%%%  compromise and just cache angles and locations

corr = zeros (lc, num_angles);
if max(angles) ~= 0 | min(angles) ~= 0,
  ['Caching ' num2str(num_angles*lc) ' frequency corrections']
  for j = 1:num_angles,
    fore_tan = tan(angles(j)*pi/180);
    for k = 1:lc,
      corr(k,j) = fore_tan / (fl - (k-col_center) * fore_tan);
    end;
  end;
end;

%%%  We don't want to use frequencies outside the legal range.  Compute the
%%%  lowest frequency available in each column.  Since we compute disparity
%%%  relative to the left image, we only need to find the lowest indices for
%%%  the RIGHT image.

'Computing low frequency bounds'
min_freq_index = zeros (1,lc);
for x = 1:lc,
  for fi = num_freqs:-1:1,
    if abs(rscalogram(fi,x)) > 0,
      min_freq_index(1,x) = fi;
      break;
    end;
  end;
end;

%%%  Impossible disparities are marked -1

res = -ones (length(disps),lc);

'Beginning the matching process'
this_percent = floor(lc/percent_update);

xstart = 1;
xend = lc;
if (pixel > 0 & pixel <= lc),
  ['Generating global PIXEL matrix at pixel ' num2str(pixel)]
  xstart = pixel;
  xend = pixel;
end;

%%%  Loop over each column in the left image
for x = xstart:xend,
  left_mag = abs(lscalogram(:,x))';
  left_pha = angle(lscalogram(:,x))';
  if (mod(x,this_percent) == 0),
    ['Finished ' num2str(x) ' columns (' num2str(floor(100*x/lc+.5)) '%)' ]
  end;

%%%  Make sure we have enough usable magnitude values
  lvalid = (left_mag > min_mag) .* lfilter(x,:);
  sumup = sum(lvalid);
  if sumup > 0,

%%%  Check all possible disparities
    for di = 1:length(disps)
      disp = disps(di);
      if x-disp >= 1 & x-disp <= lc,
	ideal_diff = mod2pi ((disp-floor(disp)) .* freqs);

%%%  Compute X coordinate in right image
	rightx = x-floor(disp);

%%%  Check all possible angles
	for ai = 1:num_angles
	  correction = freqs .* (1 + disp * corr(x,ai));

%%%  interp1 doesn't let you access anything outside the range of X, so
%%%  we have to map any illegal values to max_freq, the only guaranteed
%%%  frequency

	  if min_freq_index(1,rightx) > 0,
	    rvalid = rfilter(rightx,:) & ...
		     (abs(rscalogram(:,rightx))' > min_mag) & ...
		     (correction >= freqs(min_freq_index(1,rightx))) & ...
		     (correction <= max_freq);
	    correction = (rvalid .* correction) + (~rvalid) .* max_freq;
	    if angles(ai) == 0,
		if rightx < lc,
		    right_pha = interpolate2pi(angle(rscalogram(:,rightx))' ...
					, angle(rscalogram(:,rightx+1))', ...
					1-(disp-floor(disp)));
		else
		    right_pha = angle(rscalogram(:,rightx))';
		end;
	    else
		right_pha = interp1 (freqs, angle(rscalogram(:,rightx)), ...
				     correction)';
	    end;
	    useful_freqs = sum(lvalid .* rvalid);
	    error = lvalid .* rvalid .* left_mag .* ... %% left_mag .* ...
		    absdiffmod(ideal_diff,left_pha - right_pha);
	    if (useful_freqs > 0),
		res(di,x) = sum(error) / useful_freqs;
	    end;
	    if pixel == x,
		PIXEL(:,di) = error';
	    end;
	  end;
	end;
      end;
    end;
  end;
end;

y = res;
toc
end;

% This software provided by Mark Maimone (mwm@cmu.edu) as part of his Ph.D.
% thesis.  For more info, please see http://www.cs.cmu.edu/~mwm/thesis
%
% Release Date: 9 October 1997
%
% 	Carnegie Mellon University and the individuals associated with this
% project assume no liability for the fitness or performance of this software
% or any damages resulting from its use.  It's free, you get what you pay
% for...
