function y = img (x,yaxis,xaxis,axis_str,om)

% img(x,yaxis,xaxis,axis_str,om) - Show an image of x, scaling the display
%	to use the full colormap.  Returns the scale factor used, and plots
%	it in a log-scale sidebar when it's signficant (> om/200).  If you
%	need to find the offset used, just find the minimum value in the
%	matrix.
%
%	This function tries to be smart enough to let you omit interior
%	parameters, though you must always give them in the correct order.
%	So all these examples will work:
%
%	img(x,5)		Sets om=5
%	img(x,'xy')		Sets axis_str='xy'
%	img(x,[1:2:60],5)	Sets yaxis=[1:2:60], om=5
%
%	x	-- Image matrix
%	yaxis	-- Y Axis labels
%	xaxis	-- X Axis labels
%	axis_str - String parameter to optional axis() command
%	om	-- Orders of magnitude to use in sidebar (zero means none) [2]

% Written 2 Sep 95 by Mark Maimone (mwm@cs.cmu.edu), CMU Computer Science
% Corrections and comments appreciated

%%%  Parse the input parameters, assign default values to any missing ones.

if nargin < 1,
  error ('img: Must specify an input image')
end;

if nargin < 2,
  yaxis = [1:length(x(:,1))];
elseif isstr(yaxis),
  axis_str = yaxis;
  yaxis = [1:length(x(:,1))];
elseif size(yaxis) == 1,
  om = yaxis;
  yaxis = [1:length(x(:,1))];
end;

if nargin < 3,
  xaxis = [1:length(x(1,:))];
elseif isstr(xaxis),
  axis_str = xaxis;
  xaxis = [1:length(x(1,:))];
elseif size(xaxis) == 1,
  om = xaxis;
  xaxis = [1:length(x(1,:))];
end;

if nargin < 4 & exist('axis_str') == 0,
  axis_str = '';
elseif exist('axis_str'),
  if isstr(axis_str) == 0 & size(axis_str) == 1,
    om = axis_str;
    axis_str = '';
  end;
end;

if nargin < 5 & exist('om') == 0,
  om = 2;
end;

%%%  Compute the scaling factor

im = max(max(abs(imag(x))));
if (im > 0),
 'img:  WARNING!  Input image has complex values!'
end;
mx = max(max(x));
mn = min(min(x));
y = length (colormap) / (mx-mn);

%%%  Display the image

if nargin < 2,
	image((x-mn)*y);
else
	image(xaxis,yaxis,(x-mn)*y);
end;

%%%  Correct for the axes (if needed) *before* plotting the side bar

if size(axis_str) ~= 0,
  axis(axis_str);
end;

if om == 0,
  return;
end;

%%%  Time to display the sidebar.  Read the current plot information

xlim = get(gca,'XLim');
ylim = get(gca,'YLim');
if (strcmp (get(gca,'YDir'),'reverse') == 1),
  ylim= [ylim(2) ylim(1)];
end;
xscale = get(gca,'XScale');
yscale = get(gca,'YScale');

if (strcmp(xscale, 'linear') == 0 | strcmp(yscale, 'linear') == 0),
  ['WARNING!  Img() needs both axes linear (X ' xscale ', Y ' yscale ')']
end;

%%%  Looks like all Matlab plots use 20% of the screen real estate for
%%%  horizontal padding (basically evenly distributed between left and right).
%%%  So let's assume we have the next 12.5% of the IMAGE (as opposed to
%%%  screen) to work with on the right.  We'll draw the box from 4-8% horiz,
%%%  and center the text at 6% horiz.


xp = (xlim(2)-xlim(1))/100;		%%%  1 percent in current graph
yp = (ylim(2)-ylim(1))/100;		%%%  1 percent in current graph

if (y == 0),
  y = 10.^(-om);
end;

%%%  Compute the size of the scaling box

scale = log10(abs(y));
scale = max(-om, min(scale, om));

%%%  If there has been any appreciable scaling, draw a thermometer.

if abs(scale) >= om/200,
  color = 'r';
  if scale < 0,
    color = 'b';
  end;

  t=sprintf('%4.2g', y);
  text(xlim(2) + xp * 6, ylim(1) + yp * 5, t, 'HorizontalAlignment', 'center');
  text(xlim(2) + xp * 6, ylim(1) + yp * 55, 'Log Intensity Scaling', ...
	'Rotation', 90);

%%%  Draw the thermometer in [104xp,10yp] to [108xp,50yp].  Shrinking the
%%%  range will cause a blue thermometer to grow up, expanding the range
%%%  draws a red thermometer down.

  patch(xlim(1) + xp * [104 104 108 108], ...
	ylim(1) + yp * (30-(20/om)*scale*[0 1 1 0]), color, 'Clipping', 'off');
  line(xlim(1) + xp * [104 104 108 108 104], ...
	ylim(1) + yp * [10 50 50 10 10], 'Clipping', 'off');
  line(xlim(2) + xp * [4 8], ylim(1) + yp * [30 30], 'Clipping', 'off');
  for i=-(om-1):(om-1),
    line(xlim(2) + xp * [3.8 4.2], ylim(1) + yp * ([30 30]+i*20/om), ...
	'Clipping', 'off');
    line(xlim(2) + xp * [7.8 8.2], ylim(1) + yp * ([30 30]+i*20/om), ...
	'Clipping', 'off');
  end;
end;

end;

% This software provided by Mark Maimone (mwm@cmu.edu) as part of his Ph.D.
% thesis.  For more info, please see http://www.cs.cmu.edu/~mwm/thesis
%
% Release Date: 9 October 1997
%
% 	Carnegie Mellon University and the individuals associated with this
% project assume no liability for the fitness or performance of this software
% or any damages resulting from its use.  It's free, you get what you pay
% for...
