function [res, var] = findpeaks (sig,percentthreshold,maxcount,plotp,offset,exthresh)
%   [indices,variances] = findpeaks (signal, percentthresh, maxcount, ...
%					plotp, offset, exthresh)
%
%		-- Returns the list of peaks in the signal, up to a
%		   percentage of the maximum value, or until the max
%		   number of peaks have been found.  First result is a vector
%		   with peak indices, optional second result the variances.
%		   You can specify an offset to extend the no-more-peaks
%		   region, and optionally generate a figure to watch its
%		   progress.
%		-- Default percentthreshold is 10
%		-- Default maxcount is none
%		-- Default plotp is 0
%		-- Default offset is 2
%		-- Default exthresh is 0.3
if (nargin < 1),
  error ('findpeaks:  Must give some input');
end;

if (nargin < 2),
  percentthreshold = 10;
end;

if (nargin < 3),
  maxcount = 0;
end;

if (nargin < 4),
  plotp = 0;
end;
if (nargin < 5),
  offset = 2;
end;

if (nargin < 6),
  exthresh = 0.3;
end;

marked_line_type = '-.';		%%%  Used to highlight marked lines

[len cols] = size(sig);

if len == 1,
  sig = sig';
end;
[len cols] = size(sig);

if min(min(sig)) < 0,
  ['findpeaks: WARNING!!  Input has negative values; shifting up']
  sig = sig - min(min(sig));
end;

if percentthreshold < 0 | percentthreshold > 100,
  percentthreshold = max(0,min(100,percentthreshold));
  ['findpeaks:  WARNING!  percentthreshold outside legal range, using ' ...
   num2str(percentthreshold)]
end;

%%c = 'ymcrgbwk';			%%% Default color cycle
c = 'y';
[thismax thismaxindex] = max(sig);
thresh = thismax * percentthreshold / 100;
peakno = 1;
previ = ones(len,cols);
startsig = sig;
startsigmax = max(startsig); startsigmin = min(startsig);
ylimmin = startsigmin - .1 * (startsigmax - startsigmin);
ylimmax = startsigmax + .1 * (startsigmax - startsigmin);
%%%  Show the progress of the peak finder, on as many as 9 plots.

if (plotp ~= 0),
  if (cols > 1),
    error ('findpeaks:  Can only generate plots for a SINGLE input function')
  end;
  plots = min(9,max(plotp,maxcount));
  clf
  subplot(plots,1,1);
end;
global save1 save2

while (sum(colextract(previ,thismaxindex) .* (thismax > thresh)) > 0 & ...
	(maxcount < 1 | peakno <= maxcount)),

  indices = marktop (sig,exthresh);   %%%  Allow slight dips in 2nd derivative
  ['Working on peak ' int2str(peakno) ', ' ...
    int2str(sum(colextract(previ,thismaxindex) .* (thismax > thresh))) ...
   '/' int2str(cols) ' active columns']

%%%  Even though   fitgauss   returns a mean, it's better to center it at
%%%  the maximum value instead.  Would be nice to do this to subpixel
%%%  precision, but oh well.

  for i=1:cols,
    if (thismax(i) > thresh(i) & previ(thismaxindex(i),i) == 1),

%%%  Don't want to find peaks exactly where the function was subtracted.

      pmin = max(1,indices(1,i)-offset);
      pmax = min(len,indices(2,i)+offset);
      previ(pmin:pmax,i) = zeros(1+pmax-pmin,1);

      if (plotp ~= 0),
	subplot(plots,1,min(peakno,plots));
	notdone = char2indices(previ);
	nlen = length(notdone);
	start=1;
	hold on;
	for j=1:nlen-1,
	  if (j == nlen-1)
	    where = notdone(start:nlen)';
	    if (where(1) > 1),		%%% connect back to last marked pixel
	      where = [where(1)-1 where];
	    end;
	    plot (where,sig(where));
	  elseif (notdone(j) ~= notdone(j+1)-1),
	    where = [notdone(start:j)' notdone(j)+1];
	    if (where(1) > 1),		%%% connect back to last marked pixel
	      where = [where(1)-1 where];
	    end;
	    plot (where,sig(where));
	    start = j+1;
	  end;
	end;
	set (gca, 'Ylim', [ylimmin ylimmax]);
	set (gca, 'Xlim', [0 len+1]);
	set (gca, 'XTick', []);
	set (gca, 'YTick', []);
	ylabel (['Step ' int2str(peakno)]);
	plot (thismaxindex, thismax, ['o' c(rem(peakno-1,length(c))+1)], ...
	    'LineWidth',2);
	alreadydone = char2indices(1-previ);
	alen = length(alreadydone);
	start=1;
	for j=1:alen-1,
	  if (j == alen-1)
	    plot (alreadydone(start:alen),sig(alreadydone(start:alen)), ...
		marked_line_type);
	  elseif (alreadydone(j) ~= alreadydone(j+1)-1),
	    where = alreadydone(start:j)';
	    plot (where,sig(where), marked_line_type);
	    start = j+1;
	  end;
	end;
      end;

      [gaussmean, var(peakno,i)] = fitgauss (sig(indices(1,i):indices(2,i),i));
      res(peakno,i) = thismaxindex(i);
      if (var(peakno,i) < .001),	%%% Zero variance around a spike pixel
	var(peakno,i) = .001;
      end;
      g = gauss (var(peakno,i), 6, thismaxindex(i));
      gpad = zeros(len,1);
      glen = min (len, length(g));
      gpad(1:glen) = g(1:glen);
      sig(:,i) = max(0, sig(:,i) - thismax(i) * gpad);
    end;
  end;
  peakno = peakno + 1;
  [thismax thismaxindex] = max(sig);
end;

if (plotp ~= 0),
  subplot(plots,1,min(peakno,plots));
  p = plot ([1:len], startsig, c(rem(peakno-1,length(c))+1)); hold on;
  set (gca, 'Ylim', [ylimmin ylimmax]);
  set (gca, 'Xlim', [0 len+1]);
  set (gca, 'YTick', []);
  xlabel ('Final Result');
  plot (res, startsig(res), ['o' c(rem(peakno-1,length(c))+1)],'LineWidth',2);
  hold off;
end;

end;

% This software provided by Mark Maimone (mwm@cmu.edu) as part of his Ph.D.
% thesis.  For more info, please see http://www.cs.cmu.edu/~mwm/thesis
%
% Release Date: 9 October 1997
%
% 	Carnegie Mellon University and the individuals associated with this
% project assume no liability for the fitness or performance of this software
% or any damages resulting from its use.  It's free, you get what you pay
% for...
