/*
 *  rt_parse.C - contains parsing implementations
 *
 *
 *  Created by Sriram Vaidhyanathan on Thu Feb 26 2004.
 *  15-462 Computer Graphics Spring 2004 Programming Assignment 3
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "rt_group.h"
#include "rt_light.h"
#include "rt_sphere.h"
#include "rt_triangle.h"
#include "rt_trimesh.h"
#include "rt_material.h"
#include "rt_parse.h"
#include "rt_vectors.h"
/******************************************/
/* Include all other necessary files here */
/******************************************/

/* Constructor for class rt_parse */
rt_parse::rt_parse()
{
    /* Set parsing up */
    rt_parse_initialize();
}

/* Constructor for class rt_parse */
rt_parse::rt_parse(const char *filename)
{
    /* Set parsing up */
    rt_parse_initialize();

    /* The file-extension needs to be "ray" */
    assert(filename != NULL);
    const char *ext = &filename[strlen(filename)-4];
    assert(!strcmp(ext,".ray"));
    file = fopen(filename, "r");
    assert(file != NULL);

    /* Top-level call to begin parsing */
    parseFile();

    /* Completed parsing */
    fclose(file);
    file = NULL;
}

/* Destructor for class rt_parse */
rt_parse::~rt_parse()
{
    /* Basically, deallocate anything we "new"-ed here */
    /* Good programming practice! */
    
    if (group != NULL)
      delete group;

    std::vector<rt_Material *>::iterator mat_it;
    for (mat_it = materials.begin(); mat_it != materials.end(); mat_it++)
    {
      delete (*mat_it);
    }
    materials.clear();

    std::vector<rt_Light *>::iterator light_it;
    for (light_it = lights.begin(); light_it != lights.end(); light_it++)
    {
      delete(*light_it);
    }
    lights.clear();
}

/* Set up parsing process */
void rt_parse::rt_parse_initialize()
{
    /* Basically, throw in some reasonable values for members */
    
    group = NULL;
    background_color = Vec3f(0.5,0.5,0.5);
    eye = Vec3f(0,0,0);
    center = Vec3f(0,0,-1);
    up = Vec3f(0,1,0);
    fovy = 45;
    current_material = NULL;
    file = NULL;
    curline = 1;
}

/* Top-level parser */
void rt_parse::parseFile()
{
    char token[MAX_PARSER_TOKEN_LENGTH];
    
    //prime the parser pipe
    parse_char = fgetc(file);

    while (getToken(token))
    {
        if(!strcmp(token, "Lights"))
            parseLights();
        else if(!strcmp(token, "Materials"))
            parseMaterials();
        else if(!strcmp(token, "Background")) 
            parseBackground();
        else if(!strcmp(token, "Group"))
            group = parseGroup();
	else if(!strcmp(token, "Camera"))
	    parseCamera();
        else
        {
            cout<<"Unknown token in parseFile: '"<<token<<
                  "' at input line "<<curline<<"\n";
            exit(-1);
        }
    }        
}

/* Parses the "Group" token */
rt_Group* rt_parse::parseGroup()
{
  char token[MAX_PARSER_TOKEN_LENGTH];
  
  
  getToken(token);
  assert (!strcmp(token, "{"));
  
  /**********************************************/
  /* Instantiate the group object               */
  /**********************************************/    
  rt_Group *answer     = new rt_Group(this);
  
  bool working=true;
  while (working)
  {
    getToken(token);
    if (!strcmp(token,"}"))
    {
      working=false;
    }
    else 
    {
      if (!strcmp(token, "Sphere"))
      {
	rt_Sphere *sceneElem = parseSphere();
	assert(sceneElem != NULL);
	answer->add(sceneElem);          
      }
      else if (!strcmp(token, "Triangle"))
      {
	rt_Triangle *sceneElem = parseTriangle();
	assert(sceneElem != NULL);
	answer->add(sceneElem);
      }
      else if (!strcmp(token, "TriMesh"))
      {
	rt_TriMesh *sceneElem = parseTriMesh();
	assert(sceneElem != NULL);
	answer->add(sceneElem);
      }
      else
      {
	cout << "Unknown token in Group: '" << token << "' at line "
	     << curline << "\n";
	exit(0);
      }
    }
  }
  
  /* Return the group */
  return answer;
}

/* Parse the "Lights" token */
void rt_parse::parseLights()
{
    char token[MAX_PARSER_TOKEN_LENGTH];
    getToken(token); assert (!strcmp(token, "{"));

    /* Loop over the lights in the scene */
    bool working=true;
    while (working)
    {
        getToken(token);
        if (!strcmp(token,"}"))
        {
          working=false;
        }
        else if (!strcmp(token, "Light"))
        {
            getToken(token); assert (!strcmp(token, "{"));
            getToken(token); assert (!strcmp(token, "position"));
            Vec3f position = readVec3f();
            getToken(token); assert (!strcmp(token, "color"));
            Vec3f color = readVec3f();
            float attenuation_1 = 1;
            float attenuation_2 = 0;
            float attenuation_3 = 0;
            while(1)
            {
                getToken(token);
                if (!strcmp(token,"attenuation"))
                {
                    attenuation_1 = readFloat();
                    attenuation_2 = readFloat();
                    attenuation_3 = readFloat();
                }
                else if (!strcmp(token, "}"))
                {
                    break;
                }
                else
                {
                    cout<< "Unknown token in parseLight: "<<token<<"\n";
                    exit(0);
                }
            }

            /**********************************************/
            /* The call to your own constructor goes here */
            /**********************************************/            
            lights.push_back(new rt_Light(this,position,color,attenuation_1,
                                           attenuation_2,attenuation_3));
        }
        else
        {
            cout<< "Unknown token in parseGroup: "<<token<<"\n";
            exit(0);
        }
    }
}

/* Parse the "Camera" token */
void rt_parse::parseCamera()
{
    char token[MAX_PARSER_TOKEN_LENGTH];
    float x,y,z;
    getToken(token); assert (!strcmp(token, "{"));

    /* Get the eye, center, and up vectors (similar to gluLookAt) */
    getToken(token); assert (!strcmp(token, "eye"));
    x = readFloat();
    y = readFloat();
    z = readFloat();
    eye.Set(x,y,z);
    
    getToken(token); assert (!strcmp(token, "center"));
    x = readFloat();
    y = readFloat();
    z = readFloat();
    center.Set(x,y,z);
    
    getToken(token); assert (!strcmp(token, "up"));
    x = readFloat();
    y = readFloat();
    z = readFloat();
    up.Set(x,y,z);
    
    getToken(token); assert (!strcmp(token, "fovy"));
    fovy = readFloat();
    
    getToken(token); assert (!strcmp(token, "}"));
}

/* Parse the "Materials" token */
void rt_parse::parseMaterials()
{
    char token[MAX_PARSER_TOKEN_LENGTH];
    char texname[MAX_PARSER_TOKEN_LENGTH];
    getToken(token); assert (!strcmp(token, "{"));

    /* Loop over each Material */
    bool working=true;
    while (working)
    {
        getToken(token);
        if (!strcmp(token, "}"))
        {
          working = false;
        } 
        else if (!strcmp(token, "Material"))
        {
            getToken(token); assert (!strcmp(token, "{"));
            texname[0] = '\0';
            Vec3f diffuseColor(1,1,1);
            Vec3f specularColor(0,0,0);
            float shininess = 1;
            Vec3f transparentColor(0,0,0);
            Vec3f reflectiveColor(0,0,0);
            float indexOfRefraction = 1;

            while (1)
            {
                getToken(token);
                if (!strcmp(token, "textureFilename"))
		{
		  getToken(token);
                  strcpy(texname, token);
		}
                else if (!strcmp(token, "diffuseColor"))
                    diffuseColor = readVec3f();
                else if (!strcmp(token, "specularColor"))
                    specularColor = readVec3f();
                else if  (!strcmp(token, "shininess"))
                    shininess = readFloat();
                else if (!strcmp(token, "transparentColor"))
                    transparentColor = readVec3f();
                else if (!strcmp(token, "reflectiveColor"))
                    reflectiveColor = readVec3f();
                else if (!strcmp(token, "indexOfRefraction"))
                    indexOfRefraction = readFloat();
                else
                {
                    assert (!strcmp(token, "}"));
                    break;
                }
            }

            /**********************************************/
            /* The call to your own constructor goes here */
            /**********************************************/
            materials.push_back (new rt_Material
                (this,texname, diffuseColor, specularColor, shininess,
                 transparentColor, reflectiveColor, indexOfRefraction));
            
        }
    }
}

/* Parses the "Background" token */
void rt_parse::parseBackground()
{
    char token[MAX_PARSER_TOKEN_LENGTH];

    getToken(token); assert (!strcmp(token, "{"));
    while (1)
    {
        getToken(token);
        if (!strcmp(token, "}"))
        {
            break;
        }
        else if (!strcmp(token, "color"))
        {
            background_color = readVec3f();
        }
        else if (!strcmp(token, "ambientLight"))
        {
            ambient_light = readVec3f();
        }
        else
        {
            cout<< "Unknown token in parseBackground: "<<token<<"\n";
            assert(0);
        }
    }
}

/* Parse the "Sphere" token */
rt_Sphere* rt_parse::parseSphere()
{
    char token[MAX_PARSER_TOKEN_LENGTH];

    getToken(token); assert (!strcmp(token, "{"));
    getToken(token); assert (!strcmp(token, "materialIndex"));
    int sphere_material_index = readInt();    
    getToken(token); assert (!strcmp(token, "center"));
    Vec3f center = readVec3f();
    getToken(token); assert (!strcmp(token, "radius"));
    float radius = readFloat();
    getToken(token); assert (!strcmp(token, "}"));

    /**********************************************/
    /* The call to your own constructor goes here */
    /**********************************************/
    return new rt_Sphere(this,center,radius,sphere_material_index);
}

/* Parse out the "Triangle" token */
rt_Triangle* rt_parse::parseTriangle()
{
  char token[MAX_PARSER_TOKEN_LENGTH];
  getToken(token); assert (!strcmp(token, "{"));

  /* Parse out vertex information */
  getToken(token); assert (!strcmp(token, "vertex0"));
  Vec3f v0 = readVec3f();
  getToken(token); assert (!strcmp(token, "normal_vertex0"));
  Vec3f n0 = readVec3f();
  getToken(token); assert (!strcmp(token, "materialIndex0"));
  int v0_mat = readInt();
  getToken(token); assert (!strcmp(token, "s_t_tex_0"));
  float s_0 = 0;
  float t_0 = 0;
  s_0 = readFloat();
  t_0 = readFloat();

  /* Parse out vertex information */
  getToken(token); assert (!strcmp(token, "vertex1"));
  Vec3f v1 = readVec3f();
  getToken(token); assert (!strcmp(token, "normal_vertex1"));
  Vec3f n1 = readVec3f();
  getToken(token); assert (!strcmp(token, "materialIndex1"));
  int v1_mat = readInt();
  getToken(token); assert (!strcmp(token, "s_t_tex_1"));
  float s_1 = 0;
  float t_1 = 0;
  s_1 = readFloat();
  t_1 = readFloat();

  /* Parse out vertex information */
  getToken(token); assert (!strcmp(token, "vertex2"));
  Vec3f v2 = readVec3f();
  getToken(token); assert (!strcmp(token, "normal_vertex2"));
  Vec3f n2 = readVec3f();
  getToken(token); assert (!strcmp(token, "materialIndex2"));
  int v2_mat = readInt();
  getToken(token); assert (!strcmp(token, "s_t_tex_2"));
  float s_2 = 0;
  float t_2 = 0;
  s_2 = readFloat();
  t_2 = readFloat();

  getToken(token); assert (!strcmp(token, "}"));

  /**********************************************/
  /* The call to your own constructor goes here */
  /**********************************************/  
  return new rt_Triangle(this,v0,v1,v2,n0,n1,n2,v0_mat, v1_mat, v2_mat,
                         s_0, t_0, s_1, t_1, s_2, t_2);
}

rt_TriMesh* rt_parse::parseTriMesh()
{
  char token[MAX_PARSER_TOKEN_LENGTH];
  char objfile[MAX_PARSER_TOKEN_LENGTH];
  getToken(token); assert (!strcmp(token, "{"));
  
  /* Parse mesh information */
  getToken(token); assert (!strcmp(token, "objfile"));
  getToken(objfile);
  const char * ext = &objfile[strlen(objfile)-4];
  assert (!strcmp(ext, ".obj"));
  getToken(token); assert (!strcmp(token, "materialIndex"));
  int mesh_mat = readInt();
  
  getToken(token); assert (!strcmp(token, "}"));

  /**********************************************/
  /* The call to your own constructor goes here */
  /**********************************************/
  return new rt_TriMesh(this,objfile,mesh_mat);
}

/* consume whitespace */
void rt_parse::eatWhitespace(void)
{
  bool working = true;
  
  do{    
    while(isspace(parse_char))
    {
      if(parse_char == '\n')
      {
        curline++;
      }
      parse_char = fgetc(file);
    } 
    
    if('#'==parse_char)
    {
      /* this is a comment... eat until end of line */
      while(parse_char != '\n')
      {
        parse_char = fgetc(file);
      }
      
      curline++;
    }
    else
    {
      working = false;
    }
    
  } while(working);
}

/* Parse out a single token */
int rt_parse::getToken(char token[MAX_PARSER_TOKEN_LENGTH])
{
    int idx=0;
    
    assert (file != NULL);
    eatWhitespace();
    
    if(parse_char == EOF)
    {
      token[0]='\0';
      return 0;
    }
    while((!(isspace(parse_char))) && (parse_char != EOF))
    {
      token[idx]=parse_char;
      idx++;
      parse_char = fgetc(file);
    }
    
    token[idx] = '\0';
    return 1;
}

/* Reads in a 3-vector */
Vec3f rt_parse::readVec3f()
{
    float x,y,z;
    
    x=readFloat();
    y=readFloat();
    z=readFloat();
    
    return Vec3f(x,y,z);
}

/* Reads in a single float */
float rt_parse::readFloat()
{
    float answer;
    char buf[MAX_PARSER_TOKEN_LENGTH];
    
    if(!getToken(buf))
    {
        cout<< "Error trying to read 1 float (EOF?)\n";
        assert (0);
    }
    
    int count = sscanf(buf,"%f",&answer);
    if (count != 1)
    {
        cout<< "Error trying to read 1 float\n";
        assert (0);

    }
    return answer;
}

/* Reads in a single int */
int rt_parse::readInt()
{
    int answer;
    char buf[MAX_PARSER_TOKEN_LENGTH];
    
    if(!getToken(buf))
    {
        cout<< "Error trying to read 1 int (EOF?)\n";
        assert (0);
    }
    
    int count = sscanf(buf,"%d",&answer);
    if (count != 1)
    {
        cout<< "Error trying to read 1 int\n";
        assert (0);

    }
    return answer;
}
