// NOTE: For visual studio 2005, modify the working directory and command arguments in the project properties
// in order to launch internally
#include <stdlib.h>

#include <gfx/gl.h>
#include <GL/glext.h>
#include <GL/glut.h>
#include <gfx/raster.h>
#include <gfx/vec3.h>
#include <gfx/vec2.h>

using namespace gfx;

/* spline struct which contains how many control points, and an array of control points */
struct spline {
  int length;
  Vec3f *points;
};

/* the spline array */
struct spline *g_Splines;

/* total number of splines */
int g_iNumOfSplines;

//////////////////////////////////////////////////////////////////////
// Global variables

// glut handle for the menu
int g_menuId;

// the window size -- this is updated during the reshape function
int g_windowSize[] = {640, 480}; 

// the mouse position -- this is updated whenever the mouse moves
int g_mousePosition[] = {0, 0};

// flags for if mouse buttons are pressed
bool g_leftMouseButtonDown = false;
bool g_middleMouseButtonDown = false;
bool g_rightMouseButtonDown = false;

GLuint g_utexSky = 0;
int g_skyWidth, g_skyHeight;

// Global variable for setting up lighting:
const GLfloat g_lightpos[4] = { -0.5, 1.0, 1.0, 0.0 };
bool g_lighting = true;

//////////////////////////////////////////////////////////////////////
// Take a screen shot of the GL context

void screenShot(char* filename)
{
  ByteRaster raster(g_windowSize[0], g_windowSize[1], 3);

  glReadPixels(0, 0, g_windowSize[0], g_windowSize[1], 
               GL_RGB, GL_UNSIGNED_BYTE, raster.head() );

  raster.vflip();

  write_image(filename, raster);

}

int loadSplines(char *argv)
{
  char *cName = (char *)malloc(128 * sizeof(char));
  FILE *fileList;
  FILE *fileSpline;
  int iType, i = 0, j, iLength;

  /* load the track file */
  fileList = fopen(argv, "r");
  if (fileList == NULL) {
    printf ("can't open file\n");
    exit(1);
  }

  /* stores the number of splines in a global variable */
  fscanf(fileList, "%d", &g_iNumOfSplines);

  g_Splines = (struct spline *)malloc(g_iNumOfSplines * sizeof(struct spline));

  /* reads through the spline files */
  for (j = 0; j < g_iNumOfSplines; j++) {
    i = 0;
    fscanf(fileList, "%s", cName);
    fileSpline = fopen(cName, "r");

    if (fileSpline == NULL) {
      printf ("can't open file\n");
      exit(1);
    }

    /* gets length for spline file */
    fscanf(fileSpline, "%d %d", &iLength, &iType);

    /* allocate memory for all the points */
    g_Splines[j].points = (Vec3f*)malloc(iLength * sizeof(Vec3f));
    g_Splines[j].length = iLength;

    /* saves the data to the struct */
    while (fscanf(fileSpline, "%f %f %f", 
		  &g_Splines[j].points[i][0], 
		  &g_Splines[j].points[i][1],
		  &g_Splines[j].points[i][2]) != EOF) {
      i++;
    }
  }

  free(cName);

  return 0;
}

//////////////////////////////////////////////////////////////////////
// GLUT callback for when window is resized

void reshape(int width, int height) {

  // reset viewport
  glViewport(0, 0, width, height);
  g_windowSize[0] = width;
  g_windowSize[1] = height;

  // now make our projection matrix for our camera

  if (!height) { height = 1; } // prevent divide by zero

  double aspect = (double)width / (double)height;

  const double fov = 45.0; // 45 degree field-of-view
  const double fnear = 0.1; // distance to near clipping plane
  const double ffar = 100.0; // distance to far clipping plane

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov, aspect, fnear, ffar);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // At this point, the camera is at the origin staring down
  // the negative z axis.

  // let's set up a light in the scene

  glLightfv(GL_LIGHT0, GL_POSITION, g_lightpos);
  glEnable(GL_COLOR_MATERIAL);
  glEnable(GL_LIGHT0);

  // want depth buffering
  glEnable(GL_DEPTH_TEST);

  // might come in handy when we scale our height field
  glEnable(GL_RESCALE_NORMAL);

}

//////////////////////////////////////////////////////////////////////
// Load a texture into video memory and return its handle
GLuint loadTexture(char* filename, int* pwidth, int* pheight)
{
  // input image for the heighfield, recall that pixels range from 0 through 255
  // for some reason, debugging crashes here...
  ByteRaster* pSkyTexture = read_image(filename);
  if( pSkyTexture == NULL ) {
    printf("Failed to create %s\n", filename);
    return 0;
  }

  GLuint utex = 0;
  glGenTextures(1, &utex); // create a texture

  // GL_TEXTURE_RECTANGLE_NV specifies rectangular textures
  glBindTexture(GL_TEXTURE_RECTANGLE_NV, utex); // Bind the ID texture specified by the 2nd parameter

  // The next commands sets the texture parameters
  glTexParameterf(GL_TEXTURE_RECTANGLE_NV, GL_TEXTURE_WRAP_S, GL_REPEAT); // If the u,v coordinates overflow the range 0,1 the image is repeated
  glTexParameterf(GL_TEXTURE_RECTANGLE_NV, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_RECTANGLE_NV, GL_TEXTURE_MAG_FILTER, GL_LINEAR); // The magnification function ("linear" produces better results)
  glTexParameterf(GL_TEXTURE_RECTANGLE_NV, GL_TEXTURE_MIN_FILTER, GL_LINEAR); //The minifying function

  // select modulate to mix texture with color for shading
  glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );

  GLenum format = pSkyTexture->channels() == 4 ? GL_RGBA : GL_RGB; // assume by default that GL_RGB.

  glTexImage2D(GL_TEXTURE_RECTANGLE_NV, 0, pSkyTexture->channels(), pSkyTexture->width(), pSkyTexture->height(), 0, format, GL_UNSIGNED_BYTE, pSkyTexture->pixel(0,0));

  if( pwidth )
    *pwidth = pSkyTexture->width();
  if( pheight )
    *pheight = pSkyTexture->height();

  delete pSkyTexture;

  return utex;
}

//////////////////////////////////////////////////////////////////////
// TODO: Draw a correct sky box
void drawBackground()
{
  glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
  glClear( GL_COLOR_BUFFER_BIT );

  // draw some primitives in white
  glColor3f(1,1,1);

  glMatrixMode(GL_MODELVIEW);

  glDisable(GL_LIGHTING); // Optional
  glDepthMask(GL_FALSE); // disable writing to the depth (the bkgnd is at infinity)
	
  glEnable(GL_TEXTURE_RECTANGLE_NV); // enable texture mapping
  glBindTexture(GL_TEXTURE_RECTANGLE_NV, g_utexSky);

  // draw a big cube that defines the environment
  glBegin(GL_QUADS);

  float sz = 2;
  glTexCoord2f(g_skyWidth, 0);
  glVertex3d(sz, sz, -5);
  glTexCoord2f(g_skyWidth, g_skyHeight);
  glVertex3d(sz, -sz, -5);
  glTexCoord2f(0, g_skyHeight);
  glVertex3d(-sz, -sz, -5);
  glTexCoord2f(0, 0);
  glVertex3d(-sz, sz, -5);

  glEnd();

  // Restore the settings we had going in to this
  glDisable(GL_TEXTURE_RECTANGLE_NV);
  glDepthMask(GL_TRUE);
  glDepthRange(0.0, 1.0);

  if (g_lighting) { glEnable(GL_LIGHTING); }
}

//////////////////////////////////////////////////////////////////////
// GLUT callback to draw the window

void display()
{
  // ...replace this code with your height field implementation...

  if (g_lighting) {
    glEnable(GL_LIGHTING);
  } else {
    glDisable(GL_LIGHTING);
  }

  // clear buffers
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  drawBackground();

  // swap buffers
  glutSwapBuffers();
}

//////////////////////////////////////////////////////////////////////
// GLUT idle callback, called whenever GLUT has nothing else to do

void idle()
{
  // ...do animation stuff...

  // force the window to re-display
  glutPostRedisplay();
}

//////////////////////////////////////////////////////////////////////
// GLUT callback for when the menu is clicked

void menu(int value)
{
  switch(value) {
  case 0:
    exit(0);
    break;
  case 1:
    g_lighting = !g_lighting;
    break;
  }
}

//////////////////////////////////////////////////////////////////////
// GLUT callback for when the mouse is moved while a button is held

void mouseMotionActive(int x, int y) {

  int mouseDelta[2] = {x - g_mousePosition[0], y - g_mousePosition[1]};

  g_mousePosition[0] = x;
  g_mousePosition[1] = y;

}

//////////////////////////////////////////////////////////////////////
// GLUT callback for mouse motion while no buttons are pressed

void mouseMotionPassive(int x, int y) {
  g_mousePosition[0] = x;
  g_mousePosition[1] = y;
}

//////////////////////////////////////////////////////////////////////
// GLUT callback for when a mouse button is clicked

void mouseButtonPressed(int button, int state, int x, int y)
{
  // Check which button was pressed and update stored mouse state 
  switch (button) {
  case GLUT_LEFT_BUTTON:
    g_leftMouseButtonDown = (state==GLUT_DOWN);
    break;
  case GLUT_MIDDLE_BUTTON:
    g_middleMouseButtonDown = (state==GLUT_DOWN);
    break;
  case GLUT_RIGHT_BUTTON:
    g_rightMouseButtonDown = (state==GLUT_DOWN);
    break;
  }

  // update stored mouse position
  g_mousePosition[0] = x;
  g_mousePosition[1] = y;
}

//////////////////////////////////////////////////////////////////////
// GLUT callback for when a keyboard key is pressed

void keyboardKeyPressed(unsigned char key, int x, int y)
{
  switch (key) {
  case 'q':
  case 'Q':
    exit(0);
  case ' ':
    screenShot("test.jpg");
    break;
  }
}

//////////////////////////////////////////////////////////////////////
// Main function

int main(int argc, char* argv[])
{
  if (argc<2)
    {  
      printf ("usage: %s trackfile\n", argv[0]);
      exit(0);
    }

  // initialize GLUT
  glutInit(&argc, argv);

  // specify our pixel format, etc
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);

  // specify our window size
  glutInitWindowSize(g_windowSize[0], g_windowSize[1]);

  loadSplines(argv[1]);

  // give our window a dorky title
  glutCreateWindow("My Awesome 15-462 Project 2");

  // create the menu and add a ``quit'' option
  g_menuId = glutCreateMenu(menu);
  glutSetMenu(g_menuId);
  glutAddMenuEntry("Quit", 0);
  glutAddMenuEntry("Toggle Lighting", 1);
  glutAttachMenu(GLUT_MIDDLE_BUTTON);

  // set a pile of callbacks
  glutReshapeFunc(reshape);
  glutDisplayFunc(display);
  glutIdleFunc(idle);
  glutKeyboardFunc(keyboardKeyPressed);
  glutMouseFunc(mouseButtonPressed);
  glutMotionFunc(mouseMotionActive);
  glutPassiveMotionFunc(mouseMotionPassive);

  // load txture
  g_utexSky = loadTexture("sky.jpg", &g_skyWidth, &g_skyHeight);

  // start the main loop
  glutMainLoop();

  return 0;
}
