import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;
import java.util.List;

public class Classifier extends JPanel {

    boolean showMargin = true;	// should we display the margin if it's +ve?
    double winRange = 3;	// maximum coordinate shown

    // a point in homogeneous coordinates, used to represent both
    // examples and weights
    static class Point3 {
	double x, y, z;
    }

    // classification examples and classification weights
    List<Point3> examples = new ArrayList<Point3>();
    Point3 weights = new Point3();

    // Convert a halfplane to a polygon in screen coordinates.  The
    // poly will extend outside the given clip area so that it appears
    // to be a full halfplane.  Results returned in xi, yi (which must
    // be preallocated to size >= 3).
    public void scaleHalfPlane(double x, double y, double z,
			       boolean flip,
			       int left, int top, int width, int height,
			       int [] xi, int [] yi) {

	// compute unit normal to halfplane
	double xn, yn;
	if (z == 0) {
	    xn = x * 1e12;
	    yn = y * 1e12;
	} else {
	    xn = x/z;
	    yn = y/z;
	}
	double len = Math.sqrt(xn * xn + yn * yn + 1e-6);
	xn /= len;
	yn /= len;

	// which side of the halfplane should be filled
	double sgn = 1;
	if (z < 0) sgn = -1;
	if (flip) sgn *= -1;

	// point on separator closest to origin
	double xc = xn / len;
	double yc = yn / len;

	// variables to hold corners
	double x0, y0, x1, y1, x2, y2;

	// split into 2 cases: whole screen is same color, or not
	if (len < .7 / winRange) { // .7 is a bit less than 1/sqrt(2)

	    // whole screen is same color; is it filled?
	    if (sgn == -1) {	// screen is filled (violating)
		x0 = -1.1 * winRange;
		y0 = -3 * winRange;
		x1 = x0;
		y1 = 3 * winRange;
		x2 = 3 * winRange;
		y2 = 0;
	    } else {		// screen is empty (satisfying)
		x0 = -2*winRange;
		x1 = x0;
		x2 = x0;
		y0 = 0;
		y1 = 0;
		y2 = 0;
	    }

	} else {

	    // a point very far away in the constraint-violating direction
	    x0 = xc + sgn * xn * 5 * winRange;
	    y0 = yc + sgn * yn * 5 * winRange;

	    // two points on separator outside of clip rect
	    x1 = xc + yn * 3 * winRange;
	    y1 = yc - xn * 3 * winRange;
	    x2 = xc - yn * 3 * winRange;
	    y2 = yc + xn * 3 * winRange;

	}

	// scale the polygon into screen coordinates
	int halfw = width/2;
	int halfh = height/2;
	int xzero = left + halfw;
	int yzero = top + halfh;
	xi[0] = xzero + (int)(x0 * halfw / winRange);
	yi[0] = yzero - (int)(y0 * halfh / winRange);
	xi[1] = xzero + (int)(x1 * halfw / winRange);
	yi[1] = yzero - (int)(y1 * halfh / winRange);
	xi[2] = xzero + (int)(x2 * halfw / winRange);
	yi[2] = yzero - (int)(y2 * halfh / winRange);
    }


    // Convert a point from screen coordinates to homogeneous
    // coordinates; return true if in the left (example) half of the
    // screen.  Result is returned in p, which must be preallocated.
    public boolean unscalePoint(int x, int y, int w, int h, Point3 p) {
	boolean left = true;
	p.x = (4.0 * winRange * x) / w - winRange;
	p.y = winRange - (2.0 * winRange * y) / h;
 	p.z = 1;
	if (p.x > winRange) {
	    p.x -= 2 * winRange;
	    left = false;
	}
	return left;
    }


    public void drawHalfPlane(Graphics2D g, double x, double y, double z,
			      boolean flip, float weight,
			      int left, int top, int width, int height) {
	int xi[] = new int[3];
	int yi[] = new int[3];

	scaleHalfPlane(x, y, z, flip, left, top, width, height, xi, yi);

	g.setClip(left, top, width, height);
	g.setStroke(new BasicStroke(weight, BasicStroke.CAP_BUTT,
				    BasicStroke.JOIN_BEVEL));
	g.setColor(new Color(0f,0f,0f,1f));
	g.drawPolygon(xi, yi, 3);
	g.setClip(null);
    }


    public void fillHalfPlane(Graphics2D g, double x, double y, double z,
			      boolean flip,
			      int left, int top, int width, int height) {
	int xi[] = new int[3];
	int yi[] = new int[3];

	scaleHalfPlane(x, y, z, flip, left, top, width, height, xi, yi);

	g.setClip(left, top, width, height);
	g.setColor(new Color(1f,0f,0f,.25f));
	g.fillPolygon(xi, yi, 3);
	g.setClip(null);
    }


    public void drawPoint(Graphics2D g, double x, double y, double z,
			  int left, int top, int width, int height) {

	// compute screen coordinates
	double xn = x/z;
	double yn = y/z;
	int halfw = width/2;
	int halfh = height/2;
	int xzero = left + halfw;
	int yzero = top + halfh;
	int plotx = xzero + (int)(xn * halfw / winRange);
	int ploty = yzero - (int)(yn * halfh / winRange);

	// which class?
	double sgn = 1;
	if (z < 0) sgn = -1;

	// draw the point
	g.setStroke(new BasicStroke(1f, BasicStroke.CAP_BUTT,
				    BasicStroke.JOIN_BEVEL));
	g.setClip(left, top, width, height);
	g.setColor(new Color(0f,0f,0f,1f));
	if (sgn < 0)
	    g.fillOval(plotx-5, ploty-5, 10, 10);
	else
	    g.drawOval(plotx-5, ploty-5, 10, 10);
	g.setClip(null);
    }


    // draw a point with a circle around it to represent margin
    public void framePoint(Graphics2D g, double x, double y, double z,
			   double radius,
			   int left, int top, int width, int height) {

	// compute screen coordinates
	double xn = x/z;
	double yn = y/z;
	int halfw = width/2;
	int halfh = height/2;
	int xzero = left + halfw;
	int yzero = top + halfh;
	int plotx = xzero + (int)(xn * halfw / winRange);
	int ploty = yzero - (int)(yn * halfh / winRange);

	// compute radii
	int xradius = (int) (radius * halfw / winRange);
	int yradius = (int) (radius * halfh / winRange);

	// draw the point
	g.setStroke(new BasicStroke(1f, BasicStroke.CAP_BUTT,
				    BasicStroke.JOIN_BEVEL));
	g.setClip(left, top, width, height);
	g.setColor(new Color(0f,0f,0f,1f));
	g.drawOval(plotx-xradius, ploty-yradius, 2*xradius, 2*yradius);
	g.setClip(null);
    }


    // Draw ourself
    public void paintComponent(Graphics graph) {

	// get dimensions
	int w = getWidth();
	int h = getHeight();

	// background
	Graphics2D g = (Graphics2D) graph;
        super.paintComponent(g);
	g.setColor(new Color(.9f, .9f, .9f, 1f));
	g.fillRect(0, 0, w, h);
	g.setStroke(new BasicStroke(1f, BasicStroke.CAP_BUTT,
				    BasicStroke.JOIN_BEVEL));

	// line separating halves of canvas
	g.setColor(new Color(0f, 0f, 0f, 1f));
	g.drawLine(w/2, 0, w/2, h);

	// dot at origin
	g.fillOval(w/4, h/2, 3, 3);
	g.fillOval(3*w/4, h/2, 3, 3);

	// unit circles
// 	int xradius = (int)(w/(4*winRange));
// 	int yradius = (int)(h/(2*winRange));
// 	g.drawOval(3*w/4 - xradius, h/2 - yradius, 2*xradius, 2*yradius);
// 	g.drawOval(w/4 - xradius, h/2 - yradius, 2*xradius, 2*yradius);


	// draw examples and the fill for their constraint halfspaces,
	// while calculating margin
	double margin = 1e12;
	for (Iterator i = examples.iterator(); i.hasNext(); ) {
	    Point3 ex = (Point3) i.next();
	    fillHalfPlane(g, -ex.x * weights.z, -ex.y * weights.z, 
			  -ex.z * weights.z, false, w/2+1, 0, w/2, h);
	    double m = ex.x * weights.x + ex.y * weights.y - ex.z * weights.z;
	    m /= Math.abs(weights.z) + 1e-10;
	    if (m < margin)
		margin = m;
	}

	// draw the boundaries for all examples (so that these appear
	// in front of the fills) and their points in example space.
	// If desired draw margins around the boundaries of points
	// that are (nearly) support vectors.
	for (Iterator i = examples.iterator(); i.hasNext(); ) {
	    Point3 ex = (Point3) i.next();
	    drawPoint(g, ex.x, ex.y, ex.z, 0, 0, w/2, h);
	    drawHalfPlane(g, -ex.x * weights.z, -ex.y * weights.z,
			  -ex.z * weights.z, false, 3f, w/2+1, 0, w/2, h);
	    double m = ex.x * weights.x + ex.y * weights.y - ex.z * weights.z;
	    m /= Math.abs(weights.z) + 1e-10;
	    if (margin > .05 && showMargin && m < margin + .15) {
		drawHalfPlane(g, -ex.x * weights.z, -ex.y * weights.z,
			      -ex.z * weights.z + margin, false, 1f,
			      w/2+1, 0, w/2, h);
		drawHalfPlane(g, -ex.x * weights.z, -ex.y * weights.z,
			      -ex.z * weights.z - margin, false, 1f,
			      w/2+1, 0, w/2, h);
	    }
	}

	// draw the weight vector
	fillHalfPlane(g, weights.x, weights.y, weights.z, true, 0, 0, w/2, h);
	drawHalfPlane(g, weights.x, weights.y, weights.z, true, 
		      3f, 0, 0, w/2, h);
	drawPoint(g, weights.x, weights.y, weights.z, w/2+1, 0, w/2, h);

	// draw margins on weight vector if desired
	if (margin > .05 && showMargin) {
	    drawHalfPlane(g, weights.x, weights.y, weights.z + margin, 
			  true, 1f, 0, 0, w/2, h);	    
	    drawHalfPlane(g, weights.x, weights.y, weights.z - margin, 
			  true, 1f, 0, 0, w/2, h);	    
// 	    framePoint(g, weights.x, weights.y, weights.z, margin, 
// 		       w/2+1, 0, w/2, h);
	}
    }

    public void toggleMargin() {
	showMargin = !showMargin;
	repaint();
    }

    public void clearPoints() {
	examples = new ArrayList<Point3>();
	repaint();
    }

    private class mouser extends MouseAdapter {

        public void mousePressed(MouseEvent e) {

	    Point3 p = new Point3();
	    boolean left = unscalePoint(e.getX(), e.getY(), 
					getWidth(), getHeight(), p);

	    if ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0) {
		
		for (Iterator i = examples.iterator(); i.hasNext(); ) {
		    Point3 ex = (Point3) i.next();
		    if ((Math.abs(ex.x / ex.z - p.x) < .05) &&
			(Math.abs(ex.y / ex.z - p.y) < .05)) {
			i.remove();
		    }
		}


	    } else {

		if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
		    p.x *= -1;
		    p.y *= -1;
		    p.z = -1;
		}

		if (left) {
		    examples.add(p);
		} else {
		    weights = p;
		}

	    }

	    repaint();
        }

        public void mouseReleased(MouseEvent e) {
        }
    }


    // Support for key actions
    public void addKey(char key, String name, AbstractAction act) {
        this.getInputMap().
            put(KeyStroke.getKeyStroke(key), name);
        this.getActionMap().put(name, act);
    }

 
    // Constructor
    public Classifier() {
        super();
        this.addMouseListener(new mouser());
	this.weights.x = 1;
	this.weights.y = 0;
	this.weights.z = 1;

        addKey(' ', "toggle", new AbstractAction() {
                public void actionPerformed(ActionEvent e) {
		    toggleMargin();
                }
            });
        addKey('x', "clear", new AbstractAction() {
                public void actionPerformed(ActionEvent e) {
		    clearPoints();
                }
            });
    }


    // Main method: just make one of me and put it in a window.
    public static void main(String[] args) {
        JFrame frame = new JFrame("Linear classifier");
	Classifier content = new Classifier();
        content.addKey('q', "quit", new AbstractAction() {
                public void actionPerformed(ActionEvent e) {
                    System.exit(0);
                }
            });
        frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
        frame.setSize(1000,530);
        frame.setContentPane(content);
        frame.setVisible(true);
    }
}
