% [x,y,wt] = candmatch(fr, sx, sy, matches)
%
% Generate candidate matches for a dot.
%   fr: the next frame in the movie
%   sx, sy: the starting position for the search
%   matches: how many candidates to return
%
% Returns:
%   x, y: candidate match locations (best first)
%   wt: candidate match scores

%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [x,y,wt] = candmatch(fr, sx, sy, matches)

dist = 5;				% std dev of dot motion in
                                        %   each frame
contrast = 8;				% this many graylevels =
                                        %   factor of 1/e in likelihood
k = 31;					% size of filter to use for
                                        %   identifying dots (in px)
res = 10;				% ratio of outer to inner
                                        %   radius in filter

[n, m] = size(fr);

% build a center-surround filter
span = 2.5;
xs = -span+(span/k):(2*span/k):span;
filt1 = exp(-xs.^2/2);
filt2 = exp(-(xs*res).^2/2);
filt = filt2/sum(filt2) - filt1/sum(filt1);

% pick an image block to search
k2 = floor(k/2);
minx = max(1, floor(sx - dist * 3 - k2));
maxx = min(m, ceil(sx + dist * 3 + k2));
miny = max(1, floor(sy - dist * 3 - k2));
maxy = min(n, ceil(sy + dist * 3 + k2));

% apply the filter and find large local maxima
fra = conv2(filt, filt, fr(miny:maxy, minx:maxx), 'valid');
%clf;
%imagesc(fra);
%pause
rx = 2:maxx-minx-k+1;
ry = 2:maxy-miny-k+1;
m1 = fra(ry,rx) > fra(ry-1,rx);
m2 = fra(ry,rx) > fra(ry+1,rx);
m3 = fra(ry,rx) > fra(ry,rx-1);
m4 = fra(ry,rx) > fra(ry,rx+1);
m5 = fra(ry,rx) > fra(ry-1,rx-1);
m6 = fra(ry,rx) > fra(ry+1,rx-1);
m7 = fra(ry,rx) > fra(ry-1,rx+1);
m8 = fra(ry,rx) > fra(ry+1,rx+1);
fra = max(0,fra(ry,rx)) .* m1 .* m2 .* m3 .* m4 .* m5 .* m6 .* m7 .* m8;

% select and weight initial candidates
numcand = min(prod(size(fra)), 10*matches);
[val, idx] = sort(fra(:));
[y, x] = ind2sub(size(fra), idx(end-numcand+1:end));
x = x + minx + k2;
y = y + miny + k2;
wt = val(end-numcand+1:end);
wt = exp((wt-max(wt))/contrast);

% reweight by distance from start pos
wt = wt .* exp(-.5 * ((x - sx).^2 + (y - sy).^2) ./ dist^2);

% find best reweighted candidates
[val, idx] = sort(wt);
x = x(idx(end:-1:end-matches+1));
y = y(idx(end:-1:end-matches+1));
wt = val(end:-1:end-matches+1);

