
%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [xs,ys,zs,wts] = track3improved(movie, npoints, scale)
% Inputs:
% movie: cell array of 3-D frames, 1 per time slice
% npoints: the number of dots being tracked
% scale: 3-element array (x,y,z) of per-pixel scaling factors
%
% Outputs:
% xs, ys, zs: (npoints X numTimeSlices) matrices of dot locations
% wts: (1 X numTimeSlices) matrix of reliabilities

% Display the image projected through the Z-axis
clf
imagesc(sum(movie{1},3));
axis image

% Get an image region
fprintf('Please select a box containing the points to track:\n')
waitforbuttonpress;
point1 = get(gca,'CurrentPoint');
bbox = rbbox;
point2 = get(gca,'CurrentPoint');
minp = round(min(point1(1,1:2), point2(1,1:2)));
maxp = round(max(point1(1,1:2), point2(1,1:2)));
xmin = minp(1);
xmax = maxp(1);
ymin = minp(2);
ymax = maxp(2);

% Display the selected region
imagesc(sum(movie{1}(ymin:ymax,xmin:xmax,:),3));
axis image

% Get a bunch of points (x and y are arrays, not scalars)
fprintf('Please select initial points to track:\n')
for i = 1:npoints
  [xi, yi] = ginput(1);
  line(xi, yi, 'Marker', 'x', 'Color', 'r')
  x(i) = xi + xmin - 1;
  y(i) = yi + ymin - 1;
end
set(gcf, 'Pointer', 'arrow')

% Derive initial z values by finding locations of max intensity
for i = 1:npoints
  [maxval, z(i)] = max(movie{1}(round(y(i)), round(x(i))) );
end

% Make sure x, y, and z are column vectors, and initialize weights
x = x(:);
y = y(:);
z = z(:);
bestwt = ones(size(x));

% Initialize return values
xs = [];
ys = [];
zs = [];
wts = [];

% Number of candidate dot locations to keep track of
ncandidates = npoints + 15;
nums = (1:ncandidates)';

% for each frame in the movie
for fr = 1:length(movie)

  % get the frame
  frame = movie{fr};

  % get candidate dots
  % "cx" = candidate x(max number of candidates, dot #n)
  % "wt" = weight a.k.a. reliability metric
  [cx, cy, cz, wt] = candregion(frame, xmin, xmax, ymin, ymax, ncandidates);
  
  % figure out which dots correspond to dots in last frame
  d = samplepost(cx, cy, cz, x, y, z, wt, npoints);

  % update positions
  mask = (d > 0);
  x(d(mask)) = cx(nums(mask));
  y(d(mask)) = cy(nums(mask));
  z(d(mask)) = cz(nums(mask));
  bestwt(d(mask)) = wt(nums(mask));

  % record tracks
  xs = [xs x];
  ys = [ys y];
  zs = [zs z];
  wts = [wts bestwt];

end
  
return


  % plot candidate matches
%  swt = 0.5*1.434./(1+exp(-15*(wt-.2)));
%  swt = swt ./ (0.5 + swt);
%  imagesc(sum(frame(ymin:ymax,xmin:xmax,:),3));
%  axis image;
%  for i = 1:length(cx)
%    lh(i) = line(cx(i)-xmin+1,cy(i)-ymin+1,'Marker','o','Color','r', ...
%		 'MarkerSize',2+8*swt(i));
%  end
%  line(x-xmin+1, y-ymin+1, 'Marker', 'x', 'Color', [1 1 0], ...
%       'LineStyle', 'none');
%  [nums cz cy cz wt]

%  while (1)
%    d = samplepost(cx, cy, cz, x, y, z, wt, npoints);
%    [nums(d>0) d(d>0)]
%    for i = 1:length(d)
%      if (d(i) < 0)
%	set(lh(i), 'Color', [1 0 0]);
%	h = line([cx(i) cx(-d(i))]-xmin+1, [cy(i) cy(-d(i))]-ymin+1);
%	set(h, 'Color', [1 0 0]);
%      elseif (d(i) > 0)
%	set(lh(i), 'Color', [0 0 1]);
%	h = line([cx(i) x(d(i))]-xmin+1, [cy(i) y(d(i))]-ymin+1);
%	set(h, 'Color', [0 0 1]);
%      else
%	set(lh(i), 'Color', [0 1 0]);
%      end
%    end
%    waitforbuttonpress; cmd = get(gcf, 'CurrentCharacter');
%    if (cmd == 'n')
%      fprintf('next frame\n');
%      break;
%    end
%  end
