% We are given some observed track candidates at positions (xs, ys,
% zs), along with their estimated reliabilities (wts, in [0,1]).  We
% believe that numreal of the tracks are "real" (ie, having high
% reliability scores); the rest are either "noise" (generated at
% random locations with low reliability) or "extra" (generated near
% real or noise tracks with random reliability).
%
% This function uses MCMC to get a sample from the posterior
% distribution of explanations for the data.  An explanation is a
% vector d, the same length as the data, each of whose elements is
% one of the following:
%   - a positive number, meaning "real"
%   - zero, meaning "noise"
%   - a negative number, meaning "extra"
%
% There will be exactly one "real" point with each index in 1:numreal.
% The "extra" data points are associated with "real" or "noise"
% points, so that an "extra" point i with d(i)=-11 is penalized for
% its distance from point 11 (which must not be extra).
%
% In addition to the sample d, we return a reliability score ep which
% measures how easy this weight vector is to explain.  (More
% precisely, ep is an estimate of (a constant multiple of) the
% normalizing constant in the posterior prior(d)*p(data|d).)

%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [d, ep] = oldsamplepost(xs, ys, zs, wts, numreal, lh)

% parameters
%steps = 1000;
%burnin = 20;
steps = 5000;
burnin = 1000;
n = length(xs);

% initialize to a reasonable guess: the biggest weights are
% explained as "real", others as "noise".
%[swts, topwts] = sort(wts);
%d = zeros(n, 1);
%d(topwts(end-numreal+1:end)) = 1:numreal;

% initialize so that random tracks are assigned to be real, and all
% others noise
d = zeros(n,1);
perm = randperm(n);
d(perm(1:numreal)) = 1:numreal;

p = density(d, xs, ys, zs, wts);
ep = 0;

for step = 1:steps

  % proposal distribution: pick a random index.  If it's currently
  % explained as a "real" (>0) observation, try to swap with a random
  % other index.  If it's currently explained as "extra" (<0) or
  % "noise" (=0), re-explain it with a different extra/noise cause.
  nd = d;
  i = 1+floor(rand*n);
  if (d(i) > 0)
    j = 1+floor(rand*n);
    nd(i) = d(j);
    nd(j) = d(i);
    if (d(j) > 0)
      nd(d==-d(i)) = -d(j);
      nd(d==-d(j)) = -d(i);
    end
  else
    newexpl = floor(rand*(1+numreal));
    nd(i) = -newexpl;
  end
  
  % MCMC step: accept nd with probability np/p
  np = density(nd, xs, ys, zs, wts);
  if (rand < np/p)
    d = nd;
    p = np;
  end
  
  % once burn-in period has passed, keep track of avg density
  if (step > burnin)
    ep = ep + p;
  end

%  eep = 0;
%  if (step > burnin) ep/(step-burnin); end
%  fprintf('%4d: %g %g\n', step, p, eep);
%  for h = 1:length(d)
%    if (d(h) < 0)
%      set(lh(h), 'Color', [1 0 0]);
%    elseif (d(h) > 0)
%      set(lh(h), 'Color', [0 0 1]);
%    else
%      set(lh(h), 'Color', [0 1 0]);
%    end
%  end
%  set(lh(i), 'Marker', 'x');
%  pause
%  set(lh(i), 'Marker', 'o');

end

% normalize avg density by number of samples
ep = ep / (steps - burnin);

return


% New observation model: weights for "real" points are 1-w where w is
% exponentially distributed w/ mean 1/5.7; weights for "noise" or
% "extra" points are uniform on [0, 1]; prior of real v. extra
% v. noise is [.9,.05,.05].  (Chosen so that a weight of .3 is about
% 50% likely to be "real", while a weight of 1 is about 95% likely to
% be "real".)
%
% XXX Observation model: weights for "noise" points are normal with mean
% XXX -.4 and sigma^2=.05; weights for "real" points are normal with mean
% XXX 1 and sigma^2=.05; weights for "extra" points are uniform.

function p = density(d, xs, ys, zs, wts)

% compute likelihoods under each possible type of assignment for each
% point
lreal = 0.9*5.7*exp(-5.7*(1-wts));
lnoise = repmat(.05, size(wts));
lextra = repmat(.05, size(wts));
%lreal = exp(-(wts-1).^2/.1)/sqrt(pi*.1);
%lnoise = exp(-(wts+.4).^2/.1)/sqrt(pi*.1);
%lextra = .089*ones(size(wts));		% corresp. to width of 11.24

% Select likelihoods corresponding to d and multiply them.  For noise
% or extra, also multiply in the likelihood of (x,y,z).  (Since there
% are always the same number of real points, and since the likelihood
% of (x,y,z) for those points is always the same, we can ignore their
% location likelihood.)
p = 1;
area = 10;
%area = (xmax-xmin)*(ymax-ymin);
sig2 = 1;
for i = 1:length(xs)
  if (d(i) > 0)				% real
    p = p * lreal(i);
  elseif (0 == d(i))			% noise
    p = p * lnoise(i);
    p = p / area;
  else					% extra
    p = p * lextra(i);
    j = find(d == -d(i));
%    if (length(j) ~= 1)
%      warning(sprintf('length of j is %d, i=%d, d(i)=%d', length(j), ...
%		      i, d(i)));
%    end
    dist = (xs(i) - xs(j)).^2 + (ys(i) - ys(j)).^2 + (zs(i) - zs(j)).^2;
    p = p * exp(-dist/(2*sig2))/sqrt(2*pi*sig2);
  end
end

return
