% Polish some local maxima of a 3D function to get sub-pixel accuracy.
%
% Given the integral coordinates (x,y,z) of the maxima, as well as the
% sampled values (im) of the function, use quadratic interpolation in
% each coordinate separately to get improved coordinates for the
% maxima.
%
% Also works for local minima, but if the points are neither local
% maxima nor local minima weird things can happen.  (Maxima/minima are
% defined by being greater/less than all of the 6 neighboring points
% gotten by changing a single coordinate by 1.)
%
% Points which are on the edge of the image in some coordinates (and
% so can't be compared to some neighbors) are left unchanged in those
% coordinates.

%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [px, py, pz] = fastpolish(im, xs, ys, zs)

[n,m,d] = size(im);


%%% X interpolation

% coordinates for interpolation
lo = xs-1;
hi = xs+1;
mask = (lo >= 1) & (hi <= m);
lo(~mask) = xs(~mask);
hi(~mask) = xs(~mask);

% values for interpolation
v1 = im(sub2ind(size(im),ys,lo,zs));
v2 = im(sub2ind(size(im),ys,xs,zs));
v3 = im(sub2ind(size(im),ys,hi,zs));

% coeffs of interpolation (for orthogonal basis functions const, x,
% and 3x^2-2)
const = (v1 + v2 + v3) / 3;
lin = (v3 - v1) / 2;
quad = (v1 + v3 - 2*v2) / 6;

% subpixel location of maximum
dx = -lin ./ (6*quad+1e-5);
px = xs;
px(mask) = px(mask) + dx(mask);


%%% Y interpolation

% coordinates for interpolation
lo = ys-1;
hi = ys+1;
mask = (lo >= 1) & (hi <= n);
lo(~mask) = ys(~mask);
hi(~mask) = ys(~mask);

% values for interpolation
v1 = im(sub2ind(size(im),lo,xs,zs));
v2 = im(sub2ind(size(im),ys,xs,zs));
v3 = im(sub2ind(size(im),hi,xs,zs));

% coeffs of interpolation (for orthogonal basis functions const, x,
% and 3x^2-2)
const = (v1 + v2 + v3) / 3;
lin = (v3 - v1) / 2;
quad = (v1 + v3 - 2*v2) / 6;

% subpixel location of maximum
dy = -lin ./ (6*quad+1e-5);
py = ys;
py(mask) = py(mask) + dy(mask);


%%% Z interpolation

% coordinates for interpolation
lo = zs-1;
hi = zs+1;
mask = (lo >= 1) & (hi <= d);
lo(~mask) = zs(~mask);
hi(~mask) = zs(~mask);

% values for interpolation
v1 = im(sub2ind(size(im),ys,xs,lo));
v2 = im(sub2ind(size(im),ys,xs,zs));
v3 = im(sub2ind(size(im),ys,xs,hi));

% coeffs of interpolation (for orthogonal basis functions const, x,
% and 3x^2-2)
const = (v1 + v2 + v3) / 3;
lin = (v3 - v1) / 2;
quad = (v1 + v3 - 2*v2) / 6;

% subpixel location of maximum
dz = -lin ./ (6*quad+1e-5);
pz = zs;
pz(mask) = pz(mask) + dz(mask);


%%%%

% sanity check -- in case user gave us points that aren't actually
% local maxima, don't move them around too much.
thresh = 0.99999;
mask = (abs(px-xs) >= thresh);
mask = mask | (abs(py-ys) >= thresh);
mask = mask | (abs(pz-zs) >= thresh);
px(mask) = xs(mask);
py(mask) = ys(mask);
pz(mask) = zs(mask);

if (sum(mask) > 0)
  idx = find(mask);
  wrn = sprintf(['it appears that input (%g, %g, %g) was not a local' ...
		 ' optimum'], px(idx(1)), py(idx(1)), pz(idx(1)));
  warning('matlab:fastpolish', wrn);
  x = px(idx(1));
  y = py(idx(1));
  z = pz(idx(1));
  im2 = zeros(size(im)+4);
  im2(3:end-2,3:end-2,3:end-2) = im;
  im2(y:y+4,x:x+4,z:z+4)
end

