% [x,y,z,wt] = candregion(fr, xmin, xmax, ymin, ymax, matches)
%
% Search a region for candidate matches for a dot.
%   fr: the image to search
%   xmin, xmax, ymin, ymax: a box within the image (z range is
%     assumed to be full)
%   matches: how many candidates to return
%
% Returns:
%   x, y, z: candidate match locations (best first)
%   wt: candidate match scores

%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [x,y,z,wt] = candregion(fr, minx, maxx, miny, maxy, matches)

k = 31;					% size of filter to use for
                                        %   identifying dots (in px)
span = 2.5;				% edge of filter = this many std's
res = 10;				% ratio of outer to inner
                                        %   radius in filter

% These parameters appear to work better sometimes: nearly a box
% filter for surround, and a quite narrow Gaussian for center.  
% k = 31; res = 100; span = .2;

k2 = floor(k/2);

[n, m, d] = size(fr);

% build a center-surround filter
xs = -span+(span/k):(2*span/k):span;
filt1 = exp(-xs.^2/2);
filt2 = exp(-(xs*res).^2/2);
filt = filt2/sum(filt2) - filt1/sum(filt1);

% apply the filter to each z-slice, adding a tiny amount of noise
% so that local maxima will be single-pixel w.h.p.
for i = 1:d
  fra3(:,:,i) = ...
      conv2(filt, filt, fr(miny-k2:maxy+k2, minx-k2:maxx+k2, i), 'valid');
end
fra3 = fra3 + rand(size(fra3))*1e-8;
fra3 = max(fra3,0);

% find local maxima by zeroing out any non-maxima
% check against 8 neighbors in xy plane and 2 along z axis
rx = 1:size(fra3,2);
ry = 1:size(fra3,1);
rz = 1:size(fra3,3);
rxhi = min(rx+1,size(fra3,2));
rxlo = max(rx-1,1);
ryhi = min(ry+1,size(fra3,1));
rylo = max(ry-1,1);
rzhi = min(rz+1,size(fra3,3));
rzlo = max(rz-1,1);
m1 = fra3(ry,rx,rz) >= fra3(rylo,rx,rz);
m2 = fra3(ry,rx,rz) >= fra3(ryhi,rx,rz);
m3 = fra3(ry,rx,rz) >= fra3(ry,rxlo,rz);
m4 = fra3(ry,rx,rz) >= fra3(ry,rxhi,rz);
m5 = fra3(ry,rx,rz) >= fra3(rylo,rxlo,rz);
m6 = fra3(ry,rx,rz) >= fra3(ryhi,rxhi,rz);
m7 = fra3(ry,rx,rz) >= fra3(ryhi,rxlo,rz);
m8 = fra3(ry,rx,rz) >= fra3(rylo,rxhi,rz);
m9 = fra3(ry,rx,rz) >= fra3(ry,rx,rzlo);
m0 = fra3(ry,rx,rz) >= fra3(ry,rx,rzhi);
opt = fra3 .* m1 .* m2 .* m3 .* m4 .* m5 .* m6 .* m7 .* m8 .* m9 .* m0;

mz1 = fra3(ry,rx,rz) >= fra3(rylo,rx,rzlo);
mz2 = fra3(ry,rx,rz) >= fra3(ryhi,rx,rzlo);
mz3 = fra3(ry,rx,rz) >= fra3(ry,rxlo,rzlo);
mz4 = fra3(ry,rx,rz) >= fra3(ry,rxhi,rzlo);
mz5 = fra3(ry,rx,rz) >= fra3(rylo,rxlo,rzlo);
mz6 = fra3(ry,rx,rz) >= fra3(ryhi,rxhi,rzlo);
mz7 = fra3(ry,rx,rz) >= fra3(ryhi,rxlo,rzlo);
mz8 = fra3(ry,rx,rz) >= fra3(rylo,rxhi,rzlo);
opt = opt .* mz1 .* mz2 .* mz3 .* mz4 .* mz5 .* mz6 .* mz7 .* mz8;

mZ1 = fra3(ry,rx,rz) >= fra3(rylo,rx,rzhi);
mZ2 = fra3(ry,rx,rz) >= fra3(ryhi,rx,rzhi);
mZ3 = fra3(ry,rx,rz) >= fra3(ry,rxlo,rzhi);
mZ4 = fra3(ry,rx,rz) >= fra3(ry,rxhi,rzhi);
mZ5 = fra3(ry,rx,rz) >= fra3(rylo,rxlo,rzhi);
mZ6 = fra3(ry,rx,rz) >= fra3(ryhi,rxhi,rzhi);
mZ7 = fra3(ry,rx,rz) >= fra3(ryhi,rxlo,rzhi);
mZ8 = fra3(ry,rx,rz) >= fra3(rylo,rxhi,rzhi);
opt = opt .* mZ1 .* mZ2 .* mZ3 .* mZ4 .* mZ5 .* mZ6 .* mZ7 .* mZ8;

% select and weight candidates
idx = find(opt);
numcand = min(length(idx), matches);
[val, idx2] = sort(opt(idx));
idx = idx(idx2);
[y, x, z] = ind2sub(size(opt), idx(end-numcand+1:end));
wt = val(end-numcand+1:end);

%wt = exp((wt-max(wt))/contrast);

% linearly scale weights so that 1 is the highest weight and 0
% corresponds to the mean filter response (which would mean that
% the candidate is very likely to be just noise).
whi = max(wt);
wlo = mean(fra3(:));
wt = (wt - wlo) ./ (whi - wlo);


%cutoff = wlo + .4 * (whi-wlo);
%wt = (wt - cutoff) ./ (whi - cutoff);
%wt = 1./(1+exp(-4*wt));


% polish each candidate for subpixel accuracy
[x, y, z] = fastpolish(fra3, x, y, z);

% shift to original coords
x = x + minx - 1;
y = y + miny - 1;

