% [x,y,z,wt] = candmatch(fr, sx, sy, sz, matches, scale)
%
% Generate candidate matches for a dot.
%   fr: the next frame in the movie
%   sx, sy: the starting position for the search
%   matches: how many candidates to return
%   scale: [dx dy dz] values for aspect ratio
%
% Returns:
%   x, y, z: candidate match locations (best first)
%   wt: candidate match scores

%    Copyright (C) 2005
%    Geoff Gordon  ggordon@cs.cmu.edu
%    Andrew Gove
%
%    This file is part of DotTrack, dot tracking software for
%    fluorescence microscope images.
%
%    DotTrack is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published
%    by the Free Software Foundation; either version 2 of the License,
%    or (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%    General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
%    02110-1301 USA

function [x,y,z,wt] = candmatch(fr, sx, sy, sz, matches, scale)

dist = 5;				% std dev of dot motion in
                                        %   each frame
contrast = 8;				% this many graylevels =
                                        %   factor of 1/e in likelihood
k = 31;					% size of filter to use for
                                        %   identifying dots (in px)
res = 10;				% ratio of outer to inner
                                        %   radius in filter

[n, m, d] = size(fr);

% build a center-surround filter
span = 2.5;
xs = -span+(span/k):(2*span/k):span;
filt1 = exp(-xs.^2/2);
filt2 = exp(-(xs*res).^2/2);
filt = filt2/sum(filt2) - filt1/sum(filt1);

% pick an image block to search
k2 = floor(k/2);
minx = max(1, floor(sx - dist * 3 - k2));
maxx = min(m, ceil(sx + dist * 3 + k2));
miny = max(1, floor(sy - dist * 3 - k2));
maxy = min(n, ceil(sy + dist * 3 + k2));

% apply the filter to each z-slice, adding a tiny amount of noise
% so that local maxima will be single-pixel w.h.p.
for i = 1:d
  fra3(:,:,i) = conv2(filt, filt, fr(miny:maxy, minx:maxx, i), 'valid');
end
fra3 = fra3 + rand(size(fra3))*1e-8;
fra3 = max(fra3,0);

% find local maxima by zeroing out any non-maxima
% check against 8 neighbors in xy plane and 2 along z axis
rx = 1:size(fra3,2);
ry = 1:size(fra3,1);
rz = 1:size(fra3,3);
rxhi = min(rx+1,size(fra3,2));
rxlo = max(rx-1,1);
ryhi = min(ry+1,size(fra3,1));
rylo = max(ry-1,1);
rzhi = min(rz+1,size(fra3,3));
rzlo = max(rz-1,1);
m1 = fra3(ry,rx,rz) >= fra3(rylo,rx,rz);
m2 = fra3(ry,rx,rz) >= fra3(ryhi,rx,rz);
m3 = fra3(ry,rx,rz) >= fra3(ry,rxlo,rz);
m4 = fra3(ry,rx,rz) >= fra3(ry,rxhi,rz);
m5 = fra3(ry,rx,rz) >= fra3(rylo,rxlo,rz);
m6 = fra3(ry,rx,rz) >= fra3(ryhi,rxhi,rz);
m7 = fra3(ry,rx,rz) >= fra3(ryhi,rxlo,rz);
m8 = fra3(ry,rx,rz) >= fra3(rylo,rxhi,rz);
m9 = fra3(ry,rx,rz) >= fra3(ry,rx,rzlo);
m0 = fra3(ry,rx,rz) >= fra3(ry,rx,rzhi);
opt = fra3 .* m1 .* m2 .* m3 .* m4 .* m5 .* m6 .* m7 .* m8 .* m9 .* m0;

% select and weight initial candidates
idx = find(opt);
numcand = min(length(idx), 10*matches);
[val, idx2] = sort(opt(idx));
idx = idx(idx2);
[y, x, z] = ind2sub(size(opt), idx(end-numcand+1:end));
wt = val(end-numcand+1:end);
wt = exp((wt-max(wt))/contrast);

% polish each candidate for subpixel accuracy
%for i = 1:5
%  imagesc(fra3(:,:,i));
%  mm = (z == i);
%  line(x(mm),y(mm),'LineStyle','none','Marker','x','Color','r');
%  pause;
%  imagesc(opt(:,:,i));
%  line(x(mm),y(mm),'LineStyle','none','Marker','x','Color','r');
%  pause;
%end
%[minx + k2 - 1, miny + k2 - 1]
[x, y, z] = fastpolish(fra3, x, y, z);

% shift to original coords and reweight by distance from start pos
x = x + minx + k2 - 1;
y = y + miny + k2 - 1;
wt = wt .* exp(-.5 * (((x - sx).*scale(1)).^2 + ((y - sy).*scale(2)).^2 + ((z - sz).*scale(3)).^2) ./ dist^2);

% find best reweighted candidates
[val, idx] = sort(wt);
x = x(idx(end:-1:end-matches+1));
y = y(idx(end:-1:end-matches+1));
z = z(idx(end:-1:end-matches+1));
wt = val(end:-1:end-matches+1);

%[x(:) y(:) z(:) wt(:)]
