(* Translation from named to de Bruijn represention *)

signature TRANSLATE =
sig
    exception Error of string
    val translate: MinML.program Stream.stream -> DBMinML.program Stream.stream

end;  (* signature TRANSLATE *)

structure Translate :> TRANSLATE =
struct
    structure M = MinML
    structure DB = DBMinML
    structure S = Stream

    exception Error of string

    (* context for translation *)
    infix 1 ++
    fun (ctx ++ x') = x'::ctx
    (* lookup : M.var -> M.var list -> DB.var *)
    fun lookup x (x'::ctx) = if x = x' then 1 else 1 + lookup x ctx
      | lookup x nil = raise Error "Unbound variable"

    (* translation to DeBruijn representation *)
    (* translate' : M.var list -> M.exp -> DB.exp *)
    fun translate' ctx (M.Int n) = DB.Int n
      | translate' ctx (M.Bool b) = DB.Bool b
      | translate' ctx (M.If (e, e1, e2)) = DB.If (translate' ctx e,
                                                   translate' ctx e1,
                                                   translate' ctx e2)
      | translate' ctx (M.Primop (primop, elist)) =
        DB.Primop (primop, map (translate' ctx) elist)
      | translate' ctx (M.Fn (t, (x, e))) =
        DB.Fn (t, ((), translate' (ctx++x) e))
      | translate' ctx (M.Rec (t, (x, e))) =
        DB.Rec (t, ((), translate' (ctx++x) e))
      | translate' ctx (M.Let (e1, (x, e2))) =
        DB.Let (translate' ctx e1, ((), translate' (ctx++x) e2))
      | translate' ctx (M.Apply (e1, e2)) =
        DB.Apply (translate' ctx e1, translate' ctx e2)
      | translate' ctx (M.Var x) = DB.Var (lookup x ctx)
      | translate' ctx (M.Object(c,args)) = DB.Object(c,map(fn(l,e) => (l, translate' ctx e)) args)
      | translate' ctx (M.Proj(l,e)) = DB.Proj(l,translate' ctx e)
      | translate' ctx (M.Call(m,args)) = DB.Call(m,map (translate' ctx) args)

    fun translate_decl (M.Class x) = DB.Class x
      | translate_decl (M.Method x) = DB.Method x
      | translate_decl (M.Extend (n, args, e)) = DB.Extend (n, map (fn (x,y) => ((),y)) args, translate' (rev (map #1 args)) e)

    fun translateModule m = map translate_decl m

    fun translate (s : M.program S.stream) : DB.program S.stream = 
        S.map (fn (M.Expr e) => DB.Expr(translate' nil e)
		| (M.Module m) => DB.Module(translateModule m)) s

end;  (* structure Translate *)
