(* Printing expressions *)
(* Supports both named and de Bruijn form *)

signature PRINT =
sig
  type exp
  type program

  (* print a type *)
  val typToString : T.typ -> string

  (* print an expression *)
  val expToString : exp -> string

  (* print a program *)
  val programToString : program -> string

  (* print a list *)
  val listToString : string list -> string
end

signature NAMEDPRINT = PRINT 
    where type exp = MinML.exp
      and type program = MinML.program

signature DBPRINT = PRINT 
    where type exp = DBMinML.exp
      and type program = DBMinML.program

(*--------------------------------------------------*)
structure Print :> NAMEDPRINT =
struct

    structure M = MinML
    type exp = M.exp
    type program = M.program

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"
      | po P.Mod = "%"

    fun seqToString delim [] = ""
      | seqToString delim [h] = h
      | seqToString delim (h::tl) = h ^ delim ^ seqToString delim tl

    val listToString = seqToString ", "

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
      | typToString (T.CLASS s) = s

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Fn (t, (x, e))) = 
	"fn " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Rec (t, (x, e))) = 
	"rec " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Let (e1, (x, e2))) = 
	"let " ^ x ^ " = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.Apply (f, e)) = 
	"(" ^ expToString f ^ " " ^ expToString e ^ ")"
      | expToString (M.Var v) = v
      | expToString (M.Object (c, args)) = 
	"{"^c^": "^ (listToString 
			 (map (fn (l,e) => l ^ " = " ^ expToString e) args)) ^ "}"
      | expToString (M.Proj (lab, body)) = "#"^lab ^ " (" ^ expToString body ^ ")"
      | expToString (M.Call (lab, args)) = 
	"call "^lab ^ " (" ^ listToString (map expToString args) ^ ")"

    fun declToString (M.Class(name, abs, sc, repn)) = 
	(if abs then "abstract " else "") ^ "class " 
	^ name 
	^ (case sc of NONE => "" | SOME x => " extends " ^ x)
	^ " of {"
	^ listToString (map (fn (l,t) => l ^ " : " ^ typToString t) repn)
	^ "}"
      | declToString _ = "<decl>"

    fun programToString (M.Expr e) = expToString e
      | programToString (M.Module dl)  = "module\n" ^ seqToString "\n" (map declToString dl) ^ "\nend"

end;  (* structure Print *)

structure DBPrint :> DBPRINT =
struct

    structure M = DBMinML
    type exp = M.exp
    type program = M.program

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"
      | po P.Mod = "%"

    fun seqToString delim [] = ""
      | seqToString delim [h] = h
      | seqToString delim (h::tl) = h ^ delim ^ seqToString delim tl

    val listToString = seqToString ", "

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.CLASS s) = s
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.Fn (t, ((), e))) = 
	"fn _ : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Rec (t, ((), e))) = 
	"rec _ : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Let (e1, ((), e2))) = 
	"let - = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.Apply (e1, e2)) = 
	"(" ^ expToString e1 ^ " " ^ expToString e2 ^ ")"
      | expToString (M.Var x) = "DB[" ^ (Int.toString x) ^ "]"
      | expToString (M.Object (c, args)) = 
	"{"^c^": "^ (listToString 
			 (map (fn (l,e) => l ^ " = " ^ expToString e) args)) ^ "}"
      | expToString (M.Proj (lab, body)) = "#"^lab ^ " (" ^ expToString body ^ ")"
      | expToString (M.Call (lab, args)) = 
	"call "^lab ^ " (" ^ listToString (map expToString args) ^ ")"

    fun declToString (M.Class(name, abs, sc, repn)) = 
	(if abs then "abstract " else "") ^ "class " 
	^ name 
	^ (case sc of NONE => "" | SOME x => " extends " ^ x)
	^ " of {"
	^ listToString (map (fn (l,t) => l ^ " : " ^ typToString t) repn)
	^ "}"
      | declToString (M.Method(name, args, ret)) = "method "^name^"("^listToString args^") : "^typToString ret
      | declToString (M.Extend(name, args, impl)) = 
	"extend "^name^"("^listToString (map (fn (l,t) => "_ : "^t) args) ^ ") = "^expToString impl

    fun programToString (M.Expr e) = expToString e
      | programToString (M.Module dl)  = "module\n" ^ seqToString "\n" (map declToString dl) ^ "\nend"

end;  (* structure DBPrint *)
