signature MODULE =
sig
    type label = string

    (* should never have to be raised, by typing invariant. *)
    exception MissingCase
    exception Error of string

    val clearDecls : unit -> unit
    val install : DBMinML.module -> unit 

    val dispatch : label * label list -> DBMinML.exp
    (* Given a method name and a list of argument tags,
       returns the function body to be executed. *)

    (* returns the total set of bindings that would arise
     if a module were installed *)
    val imagine : DBMinML.module -> Bindings.bindings

    (* checks a module for the EML-specific conditions
       on default cases, ownership positions, etc. *)
    val check : DBMinML.module -> unit

    val curBindings : unit -> Bindings.bindings

    val lookupClassIn : Bindings.bindings -> label -> 
			bool * label option * (label * T.typ) list

    val lookupClass : label -> 
		      bool * label option * (label * T.typ) list


    val isSubclass : label * label -> bool
    val isSubclassIn : Bindings.bindings -> label * label -> bool
    val listSubclassIn : Bindings.bindings -> label list * label list -> bool

    val fieldType : label -> label -> T.typ option
    (* hasField <class> <field> returns SOME of
       the type of <field> iff <class> (or a superclass of it)
       indeed has field <field>, otherwise NONE *)
    val fieldTypeIn : Bindings.bindings -> label -> label -> T.typ option

    val methodType : label -> (label list * T.typ) option
    (* hasMethod<method> returns SOME of the type of 
       <method> iff <method> exists otherwise NONE *)
    val methodTypeIn : Bindings.bindings -> label -> (label list * T.typ) option

    val isClassNamed : DBMinML.label -> DBMinML.decl list -> bool
    val isMethodNamed : DBMinML.label -> DBMinML.decl list -> bool
    val methodSpec : DBMinML.label -> DBMinML.decl list -> (label list * T.typ) option
end

structure Module :> MODULE =
struct

    open Util

    structure M = DBMinML

    type label = string

    exception MissingCase
    exception Error of string

    val decls = ref Bindings.noBindings
    fun curBindings () = !decls

    fun lookupClass cn = let val c = Bindings.namedClass (!decls) cn in
			     (#abstract c, #superclass c, #repn c) end

    fun lookupClassIn d cn = let val c = Bindings.namedClass d cn in
				 (#abstract c, #superclass c, #repn c) end

    fun isSubclassIn bind (c1, c2) = 
	if c1 = c2 
	then true 
	else
	    case #superclass (Bindings.namedClass bind c1) of
		NONE => false
	      | SOME x => isSubclassIn bind (x, c2)
    fun isSubclass (c1, c2) = isSubclassIn (!decls) (c1, c2)
    fun listSubclassIn d = ListPair.all (isSubclassIn d)

    fun clearDecls () = decls := Bindings.noBindings

    (* existsExtend p md =
       is there a method extention in md whose name and args satisfy p? *)
    fun existsExtend p [] = false
      | existsExtend p (M.Extend (m, args, _)::tl) = if p(m,map #2 args) 
						     then true
						     else existsExtend p tl
      | existsExtend p (_::tl) = existsExtend p tl

    fun imagine' cs (M.Class (name, abstract, superclass, repn)::md) = 
	let
	    fun ic decls = Bindings.installClass 
			       decls
			       {name = name,
				abstract = abstract,
				superclass = superclass,
				repn = repn}
	in
	     imagine' (name::cs) md o ic
	end
      | imagine' cs (M.Method(name, args, ret)::md) = 
	let
	    val internal = List.exists (fn x => x = hd args) cs
	    fun im decls = Bindings.installMethod
			       decls
			       {name = name,
				internal = internal,
				args = args,
				ret = ret}
	in
	    imagine' cs md o im
	end
      | imagine' cs (M.Extend (m, args, impl)::md) = 
	let
	    val args = map #2 args
	    fun imi decls = Bindings.installImpl
				decls
				m
				{args = args,
				 impl = impl}
	in
	     imagine' cs md o imi
	end
      | imagine' cs [] = (fn x => x)
    fun imagineIn bind md = (imagine' [] md) bind
    fun imagine md = imagineIn (!decls) md

    fun check_local_defaults d md name = 
	let val (abs, sc, repn) = lookupClassIn d name 
	in case (abs,sc) of 
	       (true, _) => ()
	     | (false, NONE) => ()
	     | (false, SOME sc) => 
	       let
		   fun abstract_supers name =
		       let
			   val (abs, sc, _) = lookupClassIn d name
		       in
			   if abs 
			   then name :: (case sc of NONE => [] | SOME x => abstract_supers x)
			   else []
		       end
		   val req_methods = List.concat (map (Bindings.ownedMethods d) (abstract_supers sc))
		   fun check_method (m : Bindings.method) = 
		       let
			   val args = #args m 
			   val new_args = name::(tl args) (* def'n of "local default case" *)
		       in
			   if existsExtend (fn (m', args') => m' = #name m andalso args' = new_args) md
			   then ()
			   else raise Error ("Missing local default for class "^name^", method "^(#name m)^".")
		       end
	       in
		   app check_method req_methods
	       end
	end

    fun check_global_default md name args = 
	if existsExtend (fn (m', args') => m' = name andalso args' = args) md
	then ()
	else raise Error ("Missing global default for method "^name^".")

    fun dispatchIn d (m, args) = 
	let
	    val impls = Bindings.methodImpls d m
	    fun search NONE ([]:Bindings.methodimpl list) = raise MissingCase
	      | search (SOME impl) [] = impl
	      | search cur (h::tl) = 
		let
		    val new = if listSubclassIn d (args, #args h)
			      then case cur of NONE => SOME h
					     | SOME impl => if listSubclassIn d (#args h, #args impl)
							    then SOME h
							    else cur
			      else cur
		in 
		    search new tl
		end
	in
	    #impl (search NONE impls)
	end

    fun dispatch (m, args) = dispatchIn (!decls) (m,args)

(* XXX: annoying semi-bug. The following code is sensitive to the
order of extend declarations within a module. I'm pretty sure one can
always rearrange any valid EML program to get this code to accept it
(by moving more specific cases later) but it should be fixed. *)

    local 
	exception Ok 
    in
    fun check_ambiguous d m args (M.Extend (m', args', _)::tl) omd = 
	if m = m' then 
	let
	    val args' = map #2 args' (* discard useless binds *)
	    val _ = if args = args' 
		    then raise Error ("Ambiguity error: duplicate cases for "^m^"("^Print.listToString args^")" )
		    else ()
	    fun glbClass (a, b) = if isSubclassIn d (a,b) then a else if isSubclassIn d (b,a) then b else raise Ok
	    val glb = ListPair.map glbClass (args, args')
	    fun search [] = false
	      | search (M.Extend (m'', args'', _)::tl) = if m'' = m andalso listSubclassIn d (glb, map #2 args'')
						      then true
						      else search tl
	      | search (_::tl) = search tl
	in
	    if search omd 
	    then check_ambiguous d m args tl omd
	    else raise Error ("Ambiguity error: found "^m^"("^Print.listToString args^") and "
						^m^"("^Print.listToString args'^") but not "
						^m^"("^Print.listToString glb^").\n(You may need to move the most "
						^ "specific case later in the module)")
	end handle Ok => check_ambiguous d m args tl omd
	else ()
      | check_ambiguous d m args (_::tl) omd = check_ambiguous d m args tl omd
      | check_ambiguous d m args [] omd = ()
    end

    fun isClassNamed name (M.Class (name', _, _, _)::tl) = name = name' orelse isClassNamed name tl
      | isClassNamed name (_::tl) = isClassNamed name tl
      | isClassNamed name [] = false

    fun isAbstractClassNamed name (M.Class (name', b, _, _)::tl) = b andalso name = name' orelse isAbstractClassNamed name tl
      | isAbstractClassNamed name (_::tl) = isAbstractClassNamed name tl
      | isAbstractClassNamed name [] = false

    fun methodSpec name (M.Method (name', args, ret)::tl) = if name = name' 
							then SOME (args, ret)
							else methodSpec name tl
      | methodSpec name (_::tl) = methodSpec name tl
      | methodSpec name [] = NONE

    fun isMethodNamed name md = (case methodSpec name md of NONE => false | SOME _ => true)

    (* check' omd md makes sure all the declarations in md are correct in
       a module omd.
       If they are all okay, it returns ().
       Otherwise, raises an exception. *)

    fun check' omd (h::md) = 
	let val d = imagine omd
	    val _ = case h of 
			M.Class (name, _, _, _) => (check_local_defaults d md name )
		      | M.Method(name, args as fstarg::tl, _) => (if isAbstractClassNamed fstarg omd then () else check_global_default md name args)
		      | M.Method(name, [], _) => raise Error ("Can't have 0-arg methods")
		      | M.Extend (m, args, impl) =>
			let
			    val args = map #2 args (* get rid of spurious unit binds *)
			in 
			    if isClassNamed (hd args) omd (* if the extention is declared in the same module as its owner class... *)
			       orelse isMethodNamed m omd (* ...or the same module as the function is declared... *)
			    then check_ambiguous d m args md omd (* ...then we're okay proceeding with local ambiguity checking. *)
			    else raise Error ("Extention of method "^m^" doesn't satisfy nonambiguity constraint")
			end
	in
	     check' omd md
	end
      | check' omd [] = ()

    fun check md = check' md md



    fun install m = decls := imagine m

    fun fieldTypeIn bind cn f = 
	(let val c = Bindings.namedClass bind cn in
	     SOME (#2 (findItem (fn (l,t) => l = f) (#repn c)))
	     handle NotFound => (* field not found *)
		    (case #superclass c of
			 NONE => NONE
		       | SOME x => fieldTypeIn bind x f) 
	 end 
	     handle NotFound => NONE) (* class not found at all *)

    fun fieldType cn f = fieldTypeIn (!decls) cn f

    fun methodTypeIn bind mn = 
	(let val m = Bindings.namedMethod bind mn in
	     SOME(#args m, #ret m) end
	     handle NoteFound => NONE)
    fun methodType mn = methodTypeIn (!decls) mn

end
