(* Evaluation of MinML Expression *)
(* Uses de Bruijn representation *)

signature EVAL =
sig
  exception Stuck

  (* one-step evaluation, raises Stuck if impossible *)
  val step : DBMinML.exp -> DBMinML.exp

  (* ... returns NONE if impossible *)
  val stepOpt : DBMinML.exp -> DBMinML.exp option

  (* repeats step as many times as possible *)
  val multiStep : DBMinML.exp -> DBMinML.exp

  (* a stream of all steps of an evaluation *)
  val stepStream : DBMinML.exp -> DBMinML.exp Stream.stream

end;  (* signature EVAL *)

structure Eval :> EVAL =
struct

  open T
  open P
  open DBMinML
  structure S = Stream

  exception Stuck
  exception Done

  (* Substitute v for x in e. *)
  (* subst : exp -> var -> exp -> exp *)
  fun subst v x (Int i) = Int i
    | subst v x (Bool b) = Bool b
    | subst v x (If (e, e1, e2)) = If (subst v x e,
                                       subst v x e1,
                                       subst v x e2)
    | subst v x (Primop (primop, elist)) =
      Primop (primop, map (subst v x) elist)
    | subst v x (Fn (t, (b, e))) =
      Fn (t, (b, subst v (x + 1) e))
    | subst v x (Rec (t, (b, e))) =
      Rec (t, (b, subst v (x + 1) e))
    | subst v x (Let (e1, (b1, e2))) =
      Let (subst v x e1, (b1, subst v (x + 1) e2))
    | subst v x (Apply (e1, e2)) = Apply (subst v x e1, subst v x e2)
    | subst v x (Var x1) = if x = x1 then v else Var x1
    | subst v x (Object (c,args)) = Object(c, map (fn (l,e) => (l,subst v x e)) args)
    | subst v x (Proj (l, e)) = Proj (l, subst v x e)
    | subst v x (Call (m, args)) = Call (m, map (subst v x) args)

  fun step_list (e::elist) = ((step e)::elist
			      handle Done => e::(step_list elist))
    | step_list [] = raise Done
  and step (Int _) = raise Done
    | step (Bool _) = raise Done
    | step (If (e, e1, e2)) =
      (If (step e, e1, e2) (* IfCond *)
       handle Done => 
              (case e
                of Bool (b) => if b then e1 else e2 (* IfTrue, IfFalse *)
                 | _ => raise Stuck))
    | step (Primop (primop, elist)) =
      (Primop (primop, step_list elist) (* OpArg *)
       handle Done => case evalPrimop (primop, elist) (* OpVals *)
                       of SOME (e) => e
			| NONE => raise Stuck)
    | step (Fn _) = raise Done
    | step (Rec (t, (b, e))) = subst (Rec (t, (b, e))) 1 e
    | step (Let (e1, (b, e2))) =
      (Let (step e1, (b, e2)) (* LetArg *)
       handle Done => subst e1 1 e2) (* Let *)
    | step (Apply (e1, e2)) =
      (Apply (step e1, e2) (* AppFun *)
       handle Done => (Apply (e1, step e2) (* AppArg *)
                       handle Done =>
                              (case e1 (* CallFun *)
                                of Fn (_, (_, e11)) => subst e2 1 e11 
                                 | _ => raise Stuck)))
    | step (Var x) = raise Stuck
    | step (Object (c,args)) = 
      let fun step_record ((l,e)::elist) =
              ((l,(step e))::elist
               handle Done => (l,e)::(step_record elist))
            | step_record (nil) = raise Done
      in
	  Object (c, step_record args)
      end
    | step (e' as Proj (l,e)) = 
      (Proj (l,step e) 
       handle Done => 
	      let  
		  fun search [] = raise Stuck
		    | search ((l',e)::tl) = if l = l' then e else search tl
	      in
		  case e of Object (_, args) => search args
			  | _ => raise Stuck
	      end)
    | step (Call (m, args)) = 
      (Call (m, step_list args)
       handle Done => 
	      let
		  fun tagof (Object (c, _)) = c
		    | tagof _ = raise Stuck
		  val body = Module.dispatch (m, map tagof args)
		  fun subst_list n [] e = e
		    | subst_list n (h::tl) e = subst_list (n+1) tl (subst h n e)
	      in
		 subst_list 1 (rev args) body
	      end)

  fun stepOpt e = SOME (step e) handle Stuck => NONE | Done => NONE
  fun multiStep e = let val e' = step e in multiStep e' end 
                        handle Done => e
                                       
  fun stepStream e = let val e' = step e 
                       in
                         S.delay (fn () => S.Cons (e', stepStream e'))
                       end
                         handle Done => S.delay (fn () => S.Nil)

end;  (* structure Eval *)
