signature LOOP =
sig
  type 'a action = 'a -> unit
  type state

  (* ... with its type *)
  val showType  : DBMinML.exp action

  (* apply an action to a completely evaluated expression *)
  val eval      : DBMinML.exp action -> DBMinML.exp action

  (* ... on a file *)
  val loopFile  : string -> DBMinML.exp action -> unit

end

functor Loop (structure G : GC) =
struct

  structure Eval = Eval (structure G = G)

  exception Error of string

  type 'a action = 'a -> unit

  fun typing e =
          (case Typing.typeOpt e
             of SOME t => " : " ^ DBPrint.typToString t
              | NONE => " has no type.")

  (* A few actions *)

  fun showType e =
      List.app print [DBPrint.expToString e, typing e, "\n"]

  structure DB = DBMinML

  fun buildFromsVal (G.Unit) = DB.Unit
    | buildFromsVal (G.Int i) = DB.Int i
    | buildFromsVal (G.True) = DB.Bool true
    | buildFromsVal (G.False) = DB.Bool false
    | buildFromsVal (G.Nil) = DB.Nil(T.INT) (* Best Guess :( *)
    | buildFromsVal (G.Loc l) = buildFromlVal (G.read l)

  and buildFromlVal (G.Cons(v1,v2)) = DB.Cons(buildFromsVal v1, buildFromsVal v2)
    | buildFromlVal (G.Pair(v1,v2)) = DB.Pair(buildFromsVal v1, buildFromsVal v2)
    | buildFromlVal (G.Closure(eta,e)) = e

  fun eval heapSize action e = 
      (case (Eval.evaluate heapSize e)
        of e' => 
           action (buildFromsVal e'))

  fun loopFile name action =
         (Stream.app action
         ((Translate.translate o Parse.parse o Lexer.lex) 
              (Input.readfile name))) handle e as Eval.Error s => (print ("Eval: "^s^"\n"); raise e)

end
