(* Translation from named to de Bruijn represention *)

signature TRANSLATE =
sig
    exception Error of string
    val translate: MinML.exp Stream.stream -> DBMinML.exp Stream.stream
end;  (* signature TRANSLATE *)

structure Translate :> TRANSLATE =
struct
    structure M = MinML
    structure DB = DBMinML
    structure S = Stream
    structure T' = DB.T

    exception Error of string

    (* context for translation *)
    infix 1 ++
    fun (ctx ++ x') = x'::ctx
    (* lookup : M.var -> M.var list -> DB.var *)
    fun lookup x (x'::ctx) = if x = x' then 1 else 1 + lookup x ctx
      | lookup x nil = raise Error "Unbound variable"

    fun trans_primop (P.Equals) = DB.P.Equals
      | trans_primop (P.LessThan) = DB.P.LessThan
      | trans_primop (P.Plus) = DB.P.Plus
      | trans_primop (P.Minus) = DB.P.Minus
      | trans_primop (P.Times) = DB.P.Times
      | trans_primop (P.Negate) = DB.P.Negate
      | trans_primop (P.Mod) = DB.P.Mod
      | trans_primop (P.Sqrt) = DB.P.Sqrt
      | trans_primop (P.FPlus) = DB.P.FPlus
      | trans_primop (P.FTimes) = DB.P.FTimes

    fun trans_type ctx (T.INT) = T'.INT
      | trans_type ctx (T.BOOL) = T'.BOOL
      | trans_type ctx (T.FLOAT) = T'.FLOAT
      | trans_type ctx (T.UNIT) = T'.UNIT
      | trans_type ctx (T.VOID) = T'.VOID
      | trans_type ctx (T.VAR x) = T'.VAR(lookup x ctx)
      | trans_type ctx (T.CROSS(t1,t2)) = T'.CROSS(trans_type ctx t1, trans_type ctx t2)
      | trans_type ctx (T.SUM(t1,t2)) = T'.SUM(trans_type ctx t1, trans_type ctx t2)
      | trans_type ctx (T.ARROW(t1,t2)) = T'.ARROW(trans_type ctx t1, trans_type ctx t2)
      | trans_type ctx (T.MU(b,t)) = T'.MU((),trans_type (ctx++b) t)
      | trans_type ctx (T.ALL(b,t)) = T'.ALL((),trans_type (ctx++b) t)
      | trans_type ctx (T.EXISTS(b,t)) = T'.EXISTS((),trans_type (ctx++b) t)

    (* translation to DeBruijn representation *)
    (* translate' : M.var list -> M.exp -> DB.exp *)
    fun translate' ctx (M.Int n) = DB.Int n
      | translate' ctx (M.Bool b) = DB.Bool b
      | translate' ctx (M.If (e, e1, e2)) = DB.If (translate' ctx e,
                                                   translate' ctx e1,
                                                   translate' ctx e2)
      | translate' ctx (M.Primop (primop, elist)) =
        DB.Primop (trans_primop primop, map (translate' ctx) elist)
      | translate' ctx (M.Fn (x, e)) =
        DB.Fn ((), translate' (ctx++x) e)
      | translate' ctx (M.Rec (x, e)) =
        DB.Rec ((), translate' (ctx++x) e)
      | translate' ctx (M.Let (e1, (x, e2))) =
        DB.Let (translate' ctx e1, ((), translate' (ctx++x) e2))
      | translate' ctx (M.LetType (t, (x, e))) =
        DB.LetType (trans_type ctx t, ((), translate' (ctx++x) e))
      | translate' ctx (M.Apply (e1, e2)) =
        DB.Apply (translate' ctx e1, translate' ctx e2)
      | translate' ctx (M.Var x) = DB.Var (lookup x ctx)
      
      | translate' ctx (M.Float r) = DB.Float r
      | translate' ctx (M.Unit) = DB.Unit 
      | translate' ctx (M.Inl e) = DB.Inl (translate' ctx e)
      | translate' ctx (M.Inr e) = DB.Inr(translate' ctx e)
      | translate' ctx (M.Roll e) = DB.Roll(translate' ctx e)
      | translate' ctx (M.Pack (t,e)) = DB.Pack(trans_type ctx t, translate' ctx e)
      | translate' ctx (M.Pair(e1,e2)) = DB.Pair(translate' ctx e1, translate' ctx e2)
      | translate' ctx (M.TFn(b,e)) = DB.TFn((), translate' (ctx++b) e)
      | translate' ctx (M.Case(e,(b1,e1),(b2,e2))) = DB.Case(translate' ctx e, 
                                                            ((),translate' (ctx++b1) e1), 
                                                            ((),translate' (ctx++b2) e2))
      | translate' ctx (M.Open(e,(b1,b2,e'))) = DB.Open(translate' ctx e, ((),(),translate' (ctx++b1++b2) e'))
      | translate' ctx (M.Inst(e,t)) = DB.Inst(translate' ctx e, trans_type ctx t)
      | translate' ctx (M.Unroll e) = DB.Unroll(translate' ctx e)
      | translate' ctx (M.Itof e) = DB.Itof(translate' ctx e)
      | translate' ctx (M.Fst e) = DB.Fst(translate' ctx e)
      | translate' ctx (M.Snd e) = DB.Snd(translate' ctx e)
      | translate' ctx (M.Abort e) = DB.Abort(translate' ctx e)
      | translate' ctx (M.Annotate (e,t)) = DB.Annotate(translate' ctx e, trans_type ctx t)

    fun translate (s : M.exp S.stream) : DB.exp S.stream = 
        S.map (translate' nil) s

end;  (* structure Translate *)
