(* Printing expressions *)
(* Supports both named and de Bruijn form *)

signature PRINT =
sig
  type exp
  type typ

  (* print a type *)
  val typToString : typ -> string

  (* print an expression *)
  val expToString : exp -> string
end

signature NAMEDPRINT = PRINT 
    where type exp = MinML.exp and type typ = MinML.T.typ

signature DBPRINT = PRINT 
    where type exp = DBMinML.exp and type typ = DBMinML.T.typ

(*--------------------------------------------------*)
structure Print :> NAMEDPRINT =
struct

    structure M = MinML
    type exp = M.exp
    type typ = T.typ

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"
      | po P.Mod = "%"
      | po P.Sqrt = "sqrt"
      | po P.FPlus = "+."
      | po P.FTimes = "*."

    fun typToString T.INT = "int"
      | typToString T.FLOAT = "float"
      | typToString T.BOOL = "bool"
      | typToString T.UNIT = "unit"
      | typToString T.VOID = "void"
      | typToString (T.VAR s) = s
      | typToString (T.MU(s,t)) = 
        "Mu "^ s ^ ". (" ^ typToString t ^ ")"
      | typToString (T.ALL(s,t)) = 
        "All "^ s ^ ". (" ^ typToString t ^ ")"
      | typToString (T.EXISTS(s,t)) = 
        "Exists "^ s ^ ". (" ^ typToString t ^ ")"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
      | typToString (T.SUM(d, r)) = 
	"(" ^ typToString d ^ ") + (" ^ typToString r ^ ")"
      | typToString (T.CROSS(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, [e])) = 
	"(" ^ po p ^ "(" ^ expToString e ^ "))"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Fn (x, e)) = 
	"fn " ^ x ^ " => " ^ expToString e
      | expToString (M.Rec (x, e)) = 
	"rec " ^ x ^ " => " ^ expToString e
      | expToString (M.Let (e1, (x, e2))) = 
	"let " ^ x ^ " = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.LetType (t, (x, e))) = 
	"let " ^ x ^ " = " ^ typToString t ^ " in " ^
	expToString e ^ " end"
      | expToString (M.Apply (f, e)) = 
	"((" ^ expToString f ^ ") (" ^ expToString e ^ "))"
      | expToString (M.Var v) = v
      | expToString (M.Float f) = Real.toString f
      | expToString (M.Unit) = "()"
      | expToString (M.Inl e) = "inl(" ^ expToString e ^ ")"
      | expToString (M.Inr e) = "inr(" ^ expToString e ^ ")"
      | expToString (M.Roll e) = "roll(" ^ expToString e ^ ")"
      | expToString (M.Pack (t,e)) = "pack(" ^ typToString t ^ "," ^ expToString e ^ ")"
      | expToString (M.Pair(e1,e2)) = "("^ expToString e1 ^ "," ^ expToString e2 ^ ")"
      | expToString (M.TFn(x,e)) = "Fn " ^ x ^ " => " ^ expToString e
      | expToString (M.Case(e,(b1,e1),(b2,e2))) = "case "^ expToString e ^ " of inl(" ^ b1 ^ ") => "^ expToString e1 ^ " | inr(" ^ b2 ^ ") => "^ expToString e2
      | expToString (M.Open(e,(b1,b2,e'))) = "open (" ^ b1 ^ "," ^ b2 ^ ") = " ^ expToString e ^ " in " ^ expToString e'
      | expToString (M.Inst(e,t)) = expToString e ^ "[" ^ typToString t ^ "]"
      | expToString (M.Unroll(e)) = "unroll(" ^ expToString e ^ ")"
      | expToString (M.Fst(e)) = "fst(" ^ expToString e ^ ")"
      | expToString (M.Itof(e)) = "itof(" ^ expToString e ^ ")"
      | expToString (M.Snd(e)) = "snd(" ^ expToString e ^ ")"
      | expToString (M.Abort(e)) = "abort(" ^ expToString e ^ ")"
      | expToString (M.Annotate(e,t)) = "(" ^ expToString e ^ " : " ^ typToString t ^ ")"

end;  (* structure Print *)

structure DBPrint :> DBPRINT =
struct

    structure M = DBMinML
    structure P = M.P
    structure T = M.T
    type exp = M.exp
    type typ = T.typ

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"
      | po P.Mod = "%"
      | po P.Sqrt = "sqrt"
      | po P.FPlus = "+."
      | po P.FTimes = "*."

    fun typToString T.INT = "int"
      | typToString T.FLOAT = "float"
      | typToString T.BOOL = "bool"
      | typToString T.UNIT = "unit"
      | typToString T.VOID = "void"
      | typToString (T.VAR x) = "DB[" ^ (Int.toString x) ^ "]"
      | typToString (T.MU(s,t)) = 
        "Mu _ . (" ^ typToString t ^ ")"
      | typToString (T.ALL(s,t)) = 
        "All _ . (" ^ typToString t ^ ")"
      | typToString (T.EXISTS(s,t)) = 
        "Exists _ . (" ^ typToString t ^ ")"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
      | typToString (T.SUM(d, r)) = 
	"(" ^ typToString d ^ ") + (" ^ typToString r ^ ")"
      | typToString (T.CROSS(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.Fn ((), e)) = 
	"fn _  => " ^ expToString e
      | expToString (M.Rec ((), e)) = 
	"rec _ => " ^ expToString e
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, [e])) = 
	"(" ^ po p ^ "(" ^ expToString e ^ "))"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Let (e1, ((), e2))) = 
	"let - = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.LetType (t, ((), e))) = 
	"let - = " ^ typToString t ^ " in " ^
	expToString e ^ " end"
      | expToString (M.Apply (e1, e2)) = 
	"((" ^ expToString e1 ^ ") (" ^ expToString e2 ^ "))"
      | expToString (M.Var x) = "DB[" ^ (Int.toString x) ^ "]"
      | expToString (M.Float f) = Real.toString f
      | expToString (M.Unit) = "()"
      | expToString (M.Inl e) = "inl(" ^ expToString e ^ ")"
      | expToString (M.Inr e) = "inr(" ^ expToString e ^ ")"
      | expToString (M.Roll e) = "roll(" ^ expToString e ^ ")"
      | expToString (M.Pack (t,e)) = "pack(" ^ typToString t ^ "," ^ expToString e ^ ")"
      | expToString (M.Pair(e1,e2)) = "("^ expToString e1 ^ "," ^ expToString e2 ^ ")"
      | expToString (M.TFn(x,e)) = "Fn _ => " ^ expToString e
      | expToString (M.Case(e,(b1,e1),(b2,e2))) = "case "^ expToString e ^ " of inl(_) => "^ expToString e1 ^ " | inr(_) => "^ expToString e2
      | expToString (M.Open(e,(b1,b2,e'))) = "open (_,_) = " ^ expToString e ^ " in " ^ expToString e'
      | expToString (M.Inst(e,t)) = expToString e ^ "[" ^ typToString t ^ "]"
      | expToString (M.Unroll(e)) = "unroll(" ^ expToString e ^ ")"
      | expToString (M.Fst(e)) = "fst(" ^ expToString e ^ ")"
      | expToString (M.Itof(e)) = "itof(" ^ expToString e ^ ")"
      | expToString (M.Snd(e)) = "snd(" ^ expToString e ^ ")"
      | expToString (M.Abort(e)) = "abort(" ^ expToString e ^ ")"
      | expToString (M.Annotate(e,t)) = "(" ^ expToString e ^ " : " ^ typToString t ^ ")"

end;  (* structure DBPrint *)
