(* Printing expressions *)
(* Supports both named and de Bruijn form *)

signature PRINT =
sig
  type exp

  (* print a type *)
  val typToString : T.typ -> string

  (* print an expression *)
  val expToString : exp -> string
end

signature NAMEDPRINT = PRINT 
    where type exp = MinML.exp

signature DBPRINT = PRINT 
    where type exp = DBMinML.exp

(* new asst4 code: *)
signature STATEPRINT =
sig
  type state

  (* print a state *)
  val stateToString : state -> string
end

signature STATEPRINT = STATEPRINT
    where type state = EMach.state
(* end asst4 code *)

(*--------------------------------------------------*)
structure Print :> NAMEDPRINT =
struct

    structure M = MinML
    type exp = M.exp

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
(* new asst4 code: *)
      | typToString (T.PAIR(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"
      | typToString T.UNIT = "unit"
      | typToString T.EXN = "exn"
(* end asst4 code *)

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Fn (t, (x, e))) = 
	"fn " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Rec (t, (x, e))) = 
	"rec " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Let (e1, (x, e2))) = 
	"let " ^ x ^ " = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.Apply (f, e)) = 
	"(" ^ expToString f ^ " " ^ expToString e ^ ")"
      | expToString (M.Var v) = v
(* new asst4 code: *)
      | expToString (M.Unit) = "()"
      | expToString (M.Pair (e1, e2)) =
	"(" ^ expToString e1 ^ ", " ^ expToString e2 ^ ")"
      | expToString (M.Fst e) = "(fst " ^ expToString e ^ ")"
      | expToString (M.Snd e) = "(snd " ^ expToString e ^ ")"

      | expToString (M.Exception(x,e)) =
	"exception " ^ x ^ " in " ^ expToString e ^ " end "
      | expToString (M.Try(e1,e2,e3)) =
	  "try " ^ expToString e1
	^ " catch " ^ expToString e2
	^ " with " ^ expToString e3
	^ " end "

      | expToString (M.Raise(t, e1)) =
	"(raise[" ^ typToString t ^ "] " ^ expToString e1 ^ ")"
(* end asst4 code *)

end;  (* structure Print *)

structure DBPrint :> DBPRINT =
struct

    structure M = DBMinML
    type exp = M.exp

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
(* new asst4 code: *)
      | typToString T.UNIT = "unit"
      | typToString (T.PAIR(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"
      | typToString T.EXN = "exn"
(* end asst4 code *)

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.Fn (t, ((), e))) = 
	"(fn _ : " ^ typToString t ^ " => " ^ expToString e ^ ")"
      | expToString (M.Rec (t, ((), e))) = 
	"(rec _ : " ^ typToString t ^ " => " ^ expToString e ^ ")"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::nil)) = "(" ^ po p ^ expToString e ^ ")"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => b ^ " " ^ po p ^ " " ^ 
                                  expToString a)
	(expToString e) es ^ ")"
      | expToString (M.Let (e1, ((), e2))) = 
	"let - = " ^ expToString e1 ^ " in " ^ expToString e2 ^ " end"
      | expToString (M.Apply (e1, e2)) = 
	"(" ^ expToString e1 ^ " " ^ expToString e2 ^ ")"
      | expToString (M.Var x) = "DB[" ^ (Int.toString x) ^ "]"
(* new asst4 code: *)
      | expToString (M.Unit) = "()"
      | expToString (M.Pair (e1, e2)) =
	"(" ^ expToString e1 ^ ", " ^ expToString e2 ^ ")"
      | expToString (M.Fst e) = "(fst " ^ expToString e ^ ")"
      | expToString (M.Snd e) = "(snd " ^ expToString e ^ ")"

      | expToString (M.Exception(x,e)) =
	"exception " ^ "_" ^ " in " ^ expToString e ^ " end "
      | expToString (M.Try(e1,e2,e3)) =
	  "try " ^ expToString e1
	^ " catch " ^ expToString e2
	^ " with " ^ expToString e3
	^ " end "

      | expToString (M.Raise(t, e1)) =
	"(raise[" ^ typToString t ^ "] " ^ expToString e1 ^ ")"
(* end asst4 code *)

end; (* structure DBPrint *)

(* new asst4 code: *)
structure StatePrint :> STATEPRINT =
struct
    structure M = EMach
    type state = M.state
    structure DB = DBPrint

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun envToString (E::Es) =
        foldl (fn (a, b) => valueToString a ^ ", " ^ b)
              (valueToString E) Es
        | envToString nil = "-"

    and valueToString (M.VInt (i)) = Int.toString i
      | valueToString (M.VBool (true)) = "true"
      | valueToString (M.VBool (false)) = "false"
      | valueToString (M.VUnit) = "()"
      | valueToString (M.VPair (v1, v2)) = "(" ^ valueToString v1 ^ ", " ^
                                       valueToString v2 ^ ")"
      | valueToString (M.VClosure (_)) = "<<closure>>"
      | valueToString (M.VSuspend (_)) = "*<<suspension>>"
      | valueToString (M.VExn (_)) = "EXN"

    fun frameToString (M.FPrimopN (p, v::vs, e::es)) =
        po p ^ "(" ^ foldl (fn (a, b) => b ^ ", " ^ valueToString a)
                           (valueToString v) vs ^ ", _, " 
        ^ foldl (fn (a, b) => b ^ ", " ^ DB.expToString a)
                (DB.expToString e) es ^ ")"
      | frameToString (M.FPrimopN (p, nil, e::es)) =
        po p ^ "(_, " ^ foldl (fn (a, b) => b ^ ", " ^ DB.expToString a)
                              (DB.expToString e) es ^ ")"
      | frameToString (M.FPrimopN (p, v::vs, nil)) =
        po p ^ "(" ^ foldl (fn (a, b) => b ^ ", " ^ valueToString a)
                           (valueToString v) vs ^ ", _)"
      | frameToString (M.FPrimopN (p, nil, nil)) = po p ^ "(_)"
      | frameToString (M.FIf (e1, e2)) = "if(_, " ^ DB.expToString e1 ^ ", " ^
                                         DB.expToString e2 ^ ")"

      | frameToString (M.FPair1 (e2)) = "pair(_, " ^ DB.expToString e2 ^ ")"
      | frameToString (M.FPair2 (v1)) = "pair(" ^ valueToString v1 ^ ", _)"
      | frameToString (M.FFst) = "fst(_)"
      | frameToString (M.FSnd) = "snd(_)"
      | frameToString (M.FApply1 (_)) = "apply1(...)"
      | frameToString (M.FApply2 (_)) = "apply2(...)"
      | frameToString (M.FLet (_)) = "let(...)"
      | frameToString (M.FRaise (_)) = "raise(...)"
      | frameToString (M.FTry1 (_)) = "try1(...)"
      | frameToString (M.FTry2 (_)) = "try2(...)"
      | frameToString (M.FEnv (E)) = "{" ^ envToString E ^ "}"

    fun stackToString (k::ks) =
        foldl (fn (a, b) => frameToString a ^ " |> " ^ b)
              (frameToString k) ks
      | stackToString nil = "-"

    fun stateToString (M.Eval (k, e)) =
        stackToString k ^ " > " ^ DB.expToString e
      | stateToString (M.Raise (k, v)) =
        stackToString k ^ " << " ^ valueToString v
      | stateToString (M.Return (k, v)) = 
        stackToString k ^ " < " ^ valueToString v


end;  (* structure StatePrint *)
(* end asst4 code *)

