#######################################################################
# MessyAPI.py
# MessyBoard Script API
#######################################################################

import sys
sys.argc = 1
sys.argv = ["messydev"]

import messyexe

#######################################################################
# Constant Class
#######################################################################

class MBConstant:
	def __init__(self, intID, stringRep, list):
		self._intID = intID
		self._stringRep = stringRep
		if (list != None):
			list.append(self)

	def __str__(self):
		return self._stringRep

	def __int__(self):
		return self._intID

	def __cmp__(self, other):
		if (type(other) == type(self)):
			return cmp(self._intID, other._intID)
		else:
			return cmp(self._intID, other)

	def __hash__(self):
		return hash(self._intID)

#######################################################################
# Constants
#######################################################################

FALSE = 0
TRUE = 1

LEFT = MBConstant(0, "Left", None)
RIGHT = MBConstant(1, "Right", None)

BROADCAST_NOTE_STRING = "!!!#B_R_O_A_D_C_A_S_T#!!!"

GUEST_ID = -1

#######################################################################
# Colors
#######################################################################

RED =		(255, 0, 0)
GREEN =		(0, 255, 0)
BLUE =		(0, 0, 255)
BLACK =		(0, 0, 0)
WHITE =		(255, 255, 255)
YELLOW =	(255, 255, 0)
CYAN =		(0, 255, 255)
MAGENTA =	(255, 0, 255)

#######################################################################
# Object Types
#######################################################################

MB_TYPE_LIST = []

NO_TYPE				= MBConstant(0, "No type", MB_TYPE_LIST)
EVERY_TYPE			= MBConstant(1, "Every type", MB_TYPE_LIST)
ARROW_TYPE			= MBConstant(2, "Arrow", MB_TYPE_LIST)
PICTURE_TYPE		= MBConstant(3, "Picture", MB_TYPE_LIST)
CAPSULE_TYPE		= MBConstant(4, "Capsule", MB_TYPE_LIST)
LINK_TYPE			= MBConstant(5, "Link", MB_TYPE_LIST)
NOTE_TYPE			= MBConstant(6, "Note", MB_TYPE_LIST)
SHORTCUT_TYPE		= MBConstant(7, "Shortcut", MB_TYPE_LIST)
THOUGHTBUBBLE_TYPE	= MBConstant(8, "Thought Bubble", MB_TYPE_LIST)
NAMETAG_TYPE		= MBConstant(9, "Name Tag", MB_TYPE_LIST)
WIDGET_TYPE			= MBConstant(10, "Widget", MB_TYPE_LIST)

#######################################################################
# Events
#######################################################################

MB_EVENT_LIST = []

CREATE_EVENT			= MBConstant(0, "Create", MB_EVENT_LIST)
DESTROY_EVENT			= MBConstant(1, "Destroy", MB_EVENT_LIST)
MOVE_EVENT				= MBConstant(2, "Move", MB_EVENT_LIST)
ZORDERFRONT_EVENT		= MBConstant(3, "Z Order Front", MB_EVENT_LIST)
ZORDERBACK_EVENT		= MBConstant(4, "Z Order Back", MB_EVENT_LIST)
TEXTUPDATE_EVENT		= MBConstant(5, "Text Update", MB_EVENT_LIST)
FONTUPDATE_EVENT		= MBConstant(6, "Font Update", MB_EVENT_LIST)
COLORUPDATE_EVENT		= MBConstant(7, "Color Update", MB_EVENT_LIST)
DIRECTIONUPDATE_EVENT	= MBConstant(8, "Direction Update", MB_EVENT_LIST)
URLUPDATE_EVENT			= MBConstant(9, "URL Update", MB_EVENT_LIST)
SETBGIMAGE_EVENT		= MBConstant(10, "Set Background Image", MB_EVENT_LIST)
SETBGCOLOR_EVENT		= MBConstant(11, "Set Background Color", MB_EVENT_LIST)
WIDGETCLICK_EVENT		= MBConstant(12, "Widget Click", MB_EVENT_LIST)
STRINGBROADCAST_EVENT	= MBConstant(13, "String Broadcast", MB_EVENT_LIST)

#######################################################################
# Module Variables
#######################################################################

# table of timer callback functions, indexed by timer ID
_timerCallbackDict = {}

#######################################################################
# Init and Cleanup
#######################################################################

def _timerCallback(timerID):
	apply(_timerCallbackDict[timerID])

def _cleanupPythonAPI():
	stopAllTimers()

def InitPythonAPI():
	messyexe.setTimerCallback(_timerCallback)
	messyexe.setCleanupCallback(_cleanupPythonAPI)

InitPythonAPI()

#######################################################################
# DesktopObject
#######################################################################

class DesktopObject:
	
	def __init__(self, desktop, pDesktopObject, type):
		self._desktop = desktop
		self._pDesktopObject = pDesktopObject
		self._type = type

	def __str__(self):
		left, top, right, bottom = self.getRect()
		return "DesktopObject (%s,%d,%d,%d,%d)"%(self._type, left, top, right, bottom)

	def getType(self):
		return self._type

	def getIDNumber(self):
		return messyexe.getDesktopObjectIDNumber(self._desktop._pDesktop, self._pDesktopObject)

	def getRect(self):
		return messyexe.getDesktopObjectRect(self._desktop._pDesktop, self._pDesktopObject)
	
	def setRect(self, left, top, right, bottom, callListeners=TRUE):
		return messyexe.setDesktopObjectRect(self._desktop._pDesktop, self._pDesktopObject, left, top, right, bottom, callListeners)		

	def setPosition(self, left, top, callListeners=TRUE):
		oldLeft, oldTop, oldRight, oldBottom = self.getRect()
		width = oldRight - oldLeft
		height = oldBottom - oldTop
		return self.setRect(left, top, left+width, top+height, callListeners)

	def setSize(self, width, height, callListeners=TRUE):
		left, top, right, bottom = self.getRect()
		return self.setRect(left, top, left + width, top + height, callListeners)

	def getText(self):
		return messyexe.getDesktopObjectText(self._desktop._pDesktop, self._pDesktopObject)

	def setText(self, text="", callListeners=TRUE):
		return messyexe.setDesktopObjectText(self._desktop._pDesktop, self._pDesktopObject, text, callListeners)

	def sendToBack(self, callListeners=TRUE):
		return self._desktop.sendToBack(self, callListeners)

	def bringToFront(self, callListeners=TRUE):
		return self._desktop.bringToFront(self, callListeners)

	def saveSnapshot(self, filename):
		return messyexe.saveDesktopObjectSnapshot(self._desktop._pDesktop, self._pDesktopObject, filename)

	def isNote(self):
		return self._type == NOTE_TYPE

	def isPicture(self):
		return self._type == PICTURE_TYPE

	def isThoughtBubble(self):
		return self._type == THOUGHTBUBBLE_TYPE

	def isArrow(self):
		return self._type == ARROW_TYPE

	def isCapsule(self):
		return self._type == CAPSULE_TYPE

	def isLink(self):
		return self._type == LINK_TYPE

	def isNameTag(self):
		return self._type == NAMETAG_TYPE

	def isWidget(self):
		return self._type == WIDGET_TYPE

	def isBroadcastNote(self):
		if (self.isNote() and (self.getText()[0:len(BROADCAST_NOTE_STRING)] == BROADCAST_NOTE_STRING)):
			return TRUE
		else:
			return FALSE

#######################################################################

class Arrow (DesktopObject):

	def __init__(self, pDesktop, pDesktopObject):
		DesktopObject.__init__(self, pDesktop, pDesktopObject, ARROW_TYPE)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.arrowSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)

	def getColors(self):
		return messyexe.arrowGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def getDirection(self):
		return messyexe.arrowGetDirection(self._desktop._pDesktop, self._pDesktopObject)

	def setDirection(self, direction, callListeners=TRUE):
		return messyexe.arrowSetDirection(self._desktop._pDesktop, self._pDesktopObject, direction, callListeners)

#######################################################################

class ThoughtBubble (DesktopObject):

	def __init__(self, pDesktop, pDesktopObject):
		DesktopObject.__init__(self, pDesktop, pDesktopObject, THOUGHTBUBBLE_TYPE)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.thoughtBubbleSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)

	def getColors(self):
		return messyexe.thoughtBubbleGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def getDirection(self):
		return messyexe.thoughtBubbleGetDirection(self._desktop._pDesktop, self._pDesktopObject)

	def setDirection(self, direction, callListeners=TRUE):
		return messyexe.thoughtBubbleSetDirection(self._desktop._pDesktop, self._pDesktopObject, direction, callListeners)

#######################################################################

class Capsule (DesktopObject):

	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, CAPSULE_TYPE)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.capsuleSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)

	def getColors(self):
		return messyexe.capsuleGetColors(self._desktop._pDesktop, self._pDesktopObject)

#######################################################################

class Link (DesktopObject):

	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, LINK_TYPE)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.linkSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)

	def getColors(self):
		return messyexe.linkGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def getURL(self):
		return messyexe.linkGetURL(self._desktop._pDesktop, self._pDesktopObject)

	def setURL(self, url, callListeners=TRUE):
		return messyexe.linkSetURL(self._desktop._pDesktop, self._pDesktopObject, url, callListeners)

#######################################################################

class Picture (DesktopObject):
	
	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, PICTURE_TYPE)

	def autoSize(self, callListeners=TRUE):
		return messyexe.pictureAutoSize(self._desktop._pDesktop, self._pDesktopObject, callListeners)

#######################################################################

class Note (DesktopObject):

	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, NOTE_TYPE)

	def getColors(self):
		return messyexe.noteGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.noteSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)

	def autoSize(self, callListeners=TRUE):
		return messyexe.noteAutoSize(self._desktop._pDesktop, self._pDesktopObject, callListeners)

	def expandIfNecessary(self, callListeners=TRUE):
		return messyexe.noteExpandIfNecessary(self._desktop._pDesktop, self._pDesktopObject, callListeners)

#######################################################################

class NameTag (DesktopObject):
	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, NAMETAG_TYPE)

	def getColors(self):
		return messyexe.nameTagGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.nameTagSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)		

#######################################################################

class Widget (DesktopObject):
	def __init__(self, desktop, pDesktopObject):
		DesktopObject.__init__(self, desktop, pDesktopObject, WIDGET_TYPE)

	def getColors(self):
		return messyexe.widgetGetColors(self._desktop._pDesktop, self._pDesktopObject)

	def setColors(self, bgColor, fgColor, callListeners=TRUE):
		return messyexe.widgetSetColors(self._desktop._pDesktop, self._pDesktopObject, bgColor, fgColor, callListeners)		

	def setText(self, text="", autoSize=TRUE, callListeners=TRUE):
		DesktopObject.setText(self, text, callListeners)
		if (autoSize):
			self.setSizeFromText(callListeners)
	
	def setPictureFromFile(self, filename, autoSize=TRUE, callListeners=TRUE):
		result = messyexe.widgetSetPictureFromFile(self._desktop._pDesktop, self._pDesktopObject, filename, callListeners)
		if (autoSize):
			self.setSizeFromPicture(callListeners)

	def setSizeFromText(self, callListeners=TRUE):
		return messyexe.widgetSetSizeFromText(self._desktop._pDesktop, self._pDesktopObject, callListeners)

	def setSizeFromPicture(self, callListeners=TRUE):
		return messyexe.widgetSetSizeFromPicture(self._desktop._pDesktop, self._pDesktopObject, callListeners)

	def setMoveable(self, moveable, callListeners=TRUE):
		return messyexe.widgetSetMoveable(self._desktop._pDesktop, self._pDesktopObject, moveable, callListeners)

	def setSelectable(self, selectable, callListeners=TRUE):
		return messyexe.widgetSetSelectable(self._desktop._pDesktop, self._pDesktopObject, selectable, callListeners)

#######################################################################

def _createDesktopObject(desktop, pDesktopObject):
	
	type = messyexe.getDesktopObjectType(desktop._pDesktop, pDesktopObject)
	# print "type: " + str(type)
	
	if (type == NOTE_TYPE):
		return Note(desktop, pDesktopObject)

	elif (type == PICTURE_TYPE):
		return Picture(desktop, pDesktopObject)

	elif (type == ARROW_TYPE):
		return Arrow(desktop, pDesktopObject)

	elif (type == CAPSULE_TYPE):
		return Capsule(desktop, pDesktopObject)

	elif (type == LINK_TYPE):
		return Link(desktop, pDesktopObject)

	elif (type == THOUGHTBUBBLE_TYPE):
		return ThoughtBubble(desktop, pDesktopObject)

	elif (type == NAMETAG_TYPE):
		return NameTag(desktop, pDesktopObject)

	elif (type == WIDGET_TYPE):
		return Widget(desktop, pDesktopObject)

	else:
		print "invalid object type passed to _createDesktopObject"
		return None

#######################################################################
# Desktop 
#######################################################################

class Desktop:

	def __init__(self, pDesktop):

		# diagnostics
		self._printEventInfo = FALSE

		# get the desktop pointer
		self._pDesktop = pDesktop

		# off-screen note for string broadcasting
		self._broadcastNote = None

		# initialize the object list
		self._objectTable = {}
		objectPointerList = messyexe.getDesktopObjectPointerList(self._pDesktop)
		for pObject in objectPointerList:
			newObject = _createDesktopObject(self, pObject)
			if (newObject.isBroadcastNote()):
				self._broadcastNote = newObject;
			else:
				self._objectTable[pObject] = newObject

		# if the broadcast note was not found, create a new one
		# this is before the event callback is set, so no listeners will be called
		if (self._broadcastNote == None):
			pBroadcastNote = messyexe.createNote(self._pDesktop, -1000, -1000, -900, -900, BLACK, WHITE, BROADCAST_NOTE_STRING + "0", FALSE, FALSE)
			self._broadcastNote = Note(self, pBroadcastNote)

		# initialize the event listener table
		self._eventListenerTable = {}
		
		for event in MB_EVENT_LIST:
			self._eventListenerTable[event] = []

		# set the event callback
		messyexe.setEventCallback(self._pDesktop, self._eventCallback)

	def _destroyBroadcastNote(self):
		objectPointerList = messyexe.getDesktopObjectPointerList(self._pDesktop)
		for pObject in objectPointerList:
			tempObject = _createDesktopObject(self, pObject)
			if (tempObject.isBroadcastNote()):
				assert(tempObject._pDesktopObject == self._broadcastNote._pDesktopObject)
				messyexe.destroyDesktopObject(self._pDesktop, tempObject._pDesktopObject, FALSE)
				self._broadcastNote = None

				# amf!!d
				print "broadcast note destroyed"

	def _findDesktopObject(self, pDesktopObject):
		if (pDesktopObject in self._objectTable.keys()):
			return self._objectTable[pDesktopObject]
		else:
			return None
		
	def _eventCallback(self, eventID, args):
		if (self._printEventInfo):
			print eventID
			print userID
			print args
			print "event: %s\t\tcallListeners: %d"%(MB_EVENT_LIST[eventID], args[1])

		# look for string broadcasts
		if ((args[0] == self._broadcastNote._pDesktopObject) and (eventID != STRINGBROADCAST_EVENT)):
			# amf!!d
			# print "hooked event for broadcast note"
			if (eventID == TEXTUPDATE_EVENT):
				self._eventCallback(STRINGBROADCAST_EVENT, args)
			return

		# amf!!d
		# print "generic event processing " + `eventID`

		if (eventID == CREATE_EVENT):
			apply(self._createCallback, args)
		elif (eventID == DESTROY_EVENT):
			apply(self._destroyCallback, args)
		elif (eventID == MOVE_EVENT):
			apply(self._moveCallback, args)
		elif (eventID == ZORDERFRONT_EVENT):
			apply(self._zOrderFrontCallback, args)
		elif (eventID == ZORDERBACK_EVENT):
			apply(self._zOrderBackCallback, args)
		elif (eventID == TEXTUPDATE_EVENT):
			apply(self._textUpdateCallback, args)
		elif (eventID == FONTUPDATE_EVENT):
			apply(self._fontUpdateCallback, args)
		elif (eventID == COLORUPDATE_EVENT):
			apply(self._colorUpdateCallback, args)
		elif (eventID == DIRECTIONUPDATE_EVENT):
			apply(self._directionUpdateCallback, args)
		elif (eventID == URLUPDATE_EVENT):
			apply(self._urlUpdateCallback, args)
		elif (eventID == SETBGIMAGE_EVENT):
			apply(self._setBgImageCallback, args)
		elif (eventID == SETBGCOLOR_EVENT):
			apply(self._setBgColorCallback, args)
		elif (eventID == WIDGETCLICK_EVENT):
			apply(self._widgetClickCallback, args)
		elif (eventID == STRINGBROADCAST_EVENT):
			apply(self._stringBroadcastCallback, args)

	def _createCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (not desktopObject):
			desktopObject = _createDesktopObject(self, pDesktopObject)
			self._objectTable[pDesktopObject] = desktopObject

		if (callListeners):
			listenerList = self._eventListenerTable[CREATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))

	def _destroyCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[DESTROY_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))

		del self._objectTable[pDesktopObject]

	def _moveCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[MOVE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))

	def _zOrderFrontCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[ZORDERFRONT_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))

	def _zOrderBackCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[ZORDERBACK_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _textUpdateCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[TEXTUPDATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _fontUpdateCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[FONTUPDATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _colorUpdateCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[COLORUPDATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _directionUpdateCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[DIRECTIONUPDATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _urlUpdateCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[URLUPDATE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _setBgImageCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[SETBGIMAGE_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))
		
	def _setBgColorCallback(self, userID, pDesktopObject, callListeners):
		desktopObject = self._findDesktopObject(pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[SETBGCOLOR_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, desktopObject))

	def _widgetClickCallback(self, userID, pDesktopObject, selectedObjectPointerList, callListeners):
		widget = self._findDesktopObject(pDesktopObject)
		objectList = []
		for p in selectedObjectPointerList:
			objectList.append(self._findDesktopObject(p))

		if (callListeners):
			listenerList = self._eventListenerTable[WIDGETCLICK_EVENT]
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, widget, objectList))

	def _stringBroadcastCallback(self, userID, pDesktopObject, callListeners):
		assert(pDesktopObject == self._broadcastNote._pDesktopObject)

		if (callListeners):
			listenerList = self._eventListenerTable[STRINGBROADCAST_EVENT]
			broadcastString = self._broadcastNote.getText()[len(BROADCAST_NOTE_STRING)+1:]
			
			# amf!!d
			# print "received string broadcast: " + broadcastString
			
			for listenerFunc in listenerList:
				apply(listenerFunc, (self, userID, broadcastString))

	# amf!!r
	#def setTitle(self, title):
	#	messyexe.setDesktopTitle(self._pDesktop, title)
		
	def getObjectList(self, type=EVERY_TYPE):
		objectList = []

		for pObject in self._objectTable.keys():
			object = self._objectTable[pObject]
			if ((type == EVERY_TYPE) or (object.getType() == type)):
				objectList.append(self._objectTable[pObject])

		return objectList

	def getName(self, userID):
		if (userID == GUEST_ID):
			return "Guest"

		for object in self.getObjectList(NAMETAG_TYPE):
			if (object.getIDNumber() == userID):
				return object.getText()

		raise RuntimeError("Invalid User ID Number: %d"%(userID,))

	def getNameTag(self, userID):
                if (userID == GUEST_ID):
                        return None

                for object in self.getObjectList(NAMETAG_TYPE):
                        if (object.getIDNumber() == userID):
                                return object

                raise RuntimeError("Invalid User ID Number: %d"%(userID,))

	def getObjectByID(self, ID):
                for object in self.getObjectList():
                        if (object.getIDNumber() == ID):
                                return object

                raise RuntimeError("Invalid ID Number: %d"%(ID,))

	def getRect(self):
		return messyexe.getDesktopRect(self._pDesktop)

	def getServerName(self):
		return messyexe.getServerName(self._pDesktop)

	def setBackgroundColor(self, color):
		return messyexe.setDesktopBackgroundColor(self._pDesktop, color)

	def getBackgroundColor(self):
		return messyexe.getDesktopBackgroundColor(self._pDesktop)

	def setBackgroundPictureFromFile(self, filename):
		messyexe.setBackgroundPictureFromFile(self._pDesktop, filename)

	def sendToBack(self, object, callListeners=TRUE):
		return messyexe.sendBackDesktopObject(self._pDesktop, object._pDesktopObject, callListeners)

	def bringToFront(self, object, callListeners=TRUE):
		return messyexe.bringFrontDesktopObject(self._pDesktop, object._pDesktopObject, callListeners)

	def saveSnapshot(self, filename):
		return messyexe.saveDesktopSnapshot(self._pDesktop, filename)

	def createNote(self, left=423, top=303, right=523, bottom=403, noteColor=BLACK, textColor=WHITE, text="", autoSize=None, callListeners=TRUE):
		if (autoSize == None):
			if (text == ""):
				autoSize = FALSE
			else:
				autoSize = TRUE
		pObject = messyexe.createNote(self._pDesktop, left, top, right, bottom, noteColor, textColor, text, autoSize, callListeners)
		return self._findDesktopObject(pObject)

	def createPictureFromFile(self, filename, left=423, top=303, right=523, bottom=403, autoSize=TRUE, callListeners=TRUE):
		pObject = messyexe.createPictureFromFile(self._pDesktop, left, top, right, bottom, filename, autoSize, callListeners)
		
		if (pObject == 0):
			raise RuntimeError("Could not find file to create picture: %s"%(filename,))
		
		return self._findDesktopObject(pObject)	

	def createArrow(self, left=423, top=303, direction=RIGHT, bgColor=BLACK, fgColor=WHITE, text="", callListeners=TRUE):
		pObject = messyexe.createArrow(self._pDesktop, left, top, direction, bgColor, fgColor, text, callListeners)
		return self._findDesktopObject(pObject)	

	def createThoughtBubble(self, left=423, top=303, direction=RIGHT, bgColor=BLACK, fgColor=WHITE, text="", callListeners=TRUE):
		pObject = messyexe.createThoughtBubble(self._pDesktop, left, top, direction, bgColor, fgColor, text, callListeners)
		return self._findDesktopObject(pObject)	

	def createCapsule(self, left=423, top=303, bgColor=BLACK, fgColor=WHITE, text="", callListeners=TRUE):
		pObject = messyexe.createCapsule(self._pDesktop, left, top, bgColor, fgColor, text, callListeners)
		return self._findDesktopObject(pObject)	

	def createLink(self, left=423, top=303, bgColor=BLACK, fgColor=WHITE, text="CMU Computer Science", url="http://www.cs.cmu.edu", callListeners=TRUE):
		pObject = messyexe.createLink(self._pDesktop, left, top, bgColor, fgColor, text, url, callListeners)
		return self._findDesktopObject(pObject)	

	def createNameTag(self, left=423, top=303, bgColor=BLACK, fgColor=WHITE, text="Type Your Name", callListeners=TRUE):
		pObject = messyexe.createNameTag(self._pDesktop, left, top, bgColor, fgColor, text, callListeners)
		return self._findDesktopObject(pObject)	

	def createWidget(self, left=423, top=303, right=523, bottom=403, bgColor=BLACK, fgColor=WHITE, text="", filename="", autoSizeText = None, autoSizeBitmap = None, selectable = FALSE, moveable = FALSE, callListeners=TRUE):
		if (autoSizeBitmap == None):
			if (filename == ""):
				autoSizeBitmap = FALSE
			elif (autoSizeText):
				autoSizeBitmap = FALSE
			else:
				autoSizeBitmap = TRUE
		
		if (autoSizeText == None):
			if (text == ""):
				autoSizeText = FALSE
			elif (autoSizeBitmap):
				autoSizeText = FALSE
			else:
				autoSizeText = TRUE

		pObject = messyexe.createWidget(self._pDesktop, left, top, right, bottom, bgColor, fgColor, text, filename, autoSizeText, autoSizeBitmap, selectable, moveable, callListeners)
		return self._findDesktopObject(pObject)

	def isObject(self, desktopObject):
		if (hasattr(desktopObject, "_pDesktopObject")):
			return self._objectTable.has_key(desktopObject._pDesktopObject)
		else:
			return FALSE

	def destroyObject(self, desktopObject, callListeners=TRUE):
		assert(self.isObject(desktopObject))
		messyexe.destroyDesktopObject(self._pDesktop, desktopObject._pDesktopObject, callListeners)

	def broadcastString(self, string):
		assert(self._broadcastNote != None)
		changebit = self._broadcastNote.getText()[len(BROADCAST_NOTE_STRING)]
		if (changebit == "0"):
			changebit = "1"
		else:
			changebit = "0"

		self._broadcastNote.setText(BROADCAST_NOTE_STRING + changebit + string, FALSE)

	def addListener(self, event, listenerFunc):
		self._eventListenerTable[event].append(listenerFunc)

	def removeListener(self, event, listenerFunc):
		self._eventListenerTable[event].remove(listenerFunc)

#######################################################################
# Global API functions
#######################################################################

def getCurrentDesktop():
	desktop = Desktop(messyexe.getDesktopPointer())
	return desktop

def startTimer(interval, callback):
	global _timerCallbackDict

	timerID = messyexe.startTimer(interval)
	_timerCallbackDict[timerID] = callback
	return timerID

def stopTimer(timerID):
	global _timerCallbackDict

	assert(_timerCallbackDict.has_key(timerID))
	messyexe.stopTimer(timerID)
	del _timerCallbackDict[timerID]

def stopAllTimers():
	global _timerCallbackDict

	for timerID in _timerCallbackDict.keys():
		stopTimer(timerID)

#######################################################################

import __builtin__
__builtin__.desktop = getCurrentDesktop()