/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* vvlprocs.c - volserver/vlserver RPC's */

#include "includes.h"
#include "hostafsd.h"


/* Generate a vldbentry from a VMC entry */
static void vmc2vldb(VMC *vmc_ent, vldbentry *vlent)
{
  memset(vlent, 0, sizeof(*vlent));
  strncpy(vlent->name, vmc_ent->name, VL_MAXNAMELEN - 1);
  vlent->name[VL_MAXNAMELEN-1] = 0;
  vlent->spares3/*volumeType*/         = 0;
  vlent->nServers           = 1;
  vlent->serverNumber[0]    = ntohl(my_serverid);
  vlent->serverPartition[0] = 0;
  if (vmc_ent->flags & VMCFLAG_RWEXPORT) {
    vlent->volumeId[RWVOL]  = vmc_ent->volid;
    vlent->serverFlags[0]   = VLSF_RWVOL;
    vlent->flags            = VLF_RWEXISTS;
#ifdef USE_DFS_SEMANTICS
    vlent->flags            |= VLF_DFSFILESET;
#endif
  } else {
    vlent->volumeId[RWVOL]  = vmc_ent->volid; /* to keep vos happy */
    vlent->volumeId[ROVOL]  = vmc_ent->volid;
    vlent->serverFlags[0]   = VLSF_ROVOL;
    vlent->flags            = VLF_ROEXISTS;
#ifdef USE_DFS_SEMANTICS
    vlent->flags            |= VLF_DFSFILESET;
#endif
  }
}


/* Look up a single volume by ID */
int SVL_GetEntryByID(struct rx_call *call, afs_int32 volid, 
			    afs_int32 voltype, vldbentry *vlent)
{
  VMC *vmc_ent;

  debug(DEBUG_RPC, "%I => VL_GetEntryByID(volid = %d)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))), volid);
  memset(vlent, 0, sizeof(*vlent));
  if (!(vmc_ent = vmc_byvolid(volid))) return VL_NOENT;
  vmc2vldb(vmc_ent, vlent);
  return 0;
}


/* Look up a single volume by name */
int SVL_GetEntryByNameO(struct rx_call *call, char *name, vldbentry *vlent)
{
  VMC *vmc_ent;
  afs_uint32 volid;

  debug(DEBUG_RPC, "%I => VL_GetEntryByName(name = %s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))), name);
  memset(vlent, 0, sizeof(*vlent));
  if (!(vmc_ent = vmc_byname(name))) {
    volid = atoi(name);
    if (!(vmc_ent = vmc_byvolid(volid))) return VL_NOENT;
  }
  vmc2vldb(vmc_ent, vlent);
  return 0;
}


/* List multiple volumes.  We are simplistic - SERVER and PARTITION
 * match either all volumes or none.  VOLUMEID matches at most one
 * volume.  Other bits are ignored.  So, we return 0, 1, or all volumes.
 * XXX: at some point, suppress VMCFLAG_NEW volumes
 * Also consider suppressing non-exported volumes.
 */
int SVL_ListAttributes(struct rx_call *call, 
			      VldbListByAttributes *attrs,
			      afs_int32 *nents, bulkentries *bulkents)
{
  VMC *ve;
  int i;

  *nents = 0;
  memset(bulkents, 0, sizeof(*bulkents));

  debug(DEBUG_RPC, "%I => VL_ListAttributes(0x%04x, %d)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        attrs->Mask, attrs->volumeid);
  if (((attrs->Mask & VLLIST_SERVER)    && attrs->server != my_serverid)
  ||  ((attrs->Mask & VLLIST_PARTITION) && attrs->partition != 1))
    return VL_NOENT;
  if ((attrs->Mask & VLLIST_VOLUMEID)) {  /* list one volume */
    if (!(ve = vmc_byvolid(attrs->volumeid))) return VL_NOENT;
    *nents = 1;
    bulkents->bulkentries_len = 1;
    bulkents->bulkentries_val = (vldbentry *)osi_Alloc(sizeof(vldbentry));
    vmc2vldb(ve, bulkents->bulkentries_val);
    return 0;
  } else {                                /* list every volume */
    i = 0;
    for (i = 0, ve = VMC_all_volumes; ve; ve = ve->anext, i++);
    bulkents->bulkentries_len = *nents = i;
    bulkents->bulkentries_val = (vldbentry *)osi_Alloc(i * sizeof(vldbentry));
    memset(bulkents->bulkentries_val, 0, i * sizeof(vldbentry));
    for (i = 0, ve = VMC_all_volumes; ve; ve = ve->anext, i++)
      vmc2vldb(ve, bulkents->bulkentries_val + i);
    return 0;
  }
}


/* Simple vlserver probe function */
int SVL_ProbeServer(struct rx_call *call)
{
  return 0;
}


/* Generate a volintInfo from a VMC entry */
static void vmc2volint(VMC *vmc_ent, volintInfo *volint)
{
  memset(volint, 0, sizeof(*volint));
  strncpy(volint->name, vmc_ent->name, VNAMESIZE - 1);
  volint->name[VNAMESIZE-1] = 0;
  volint->volid         = vmc_ent->volid;
  volint->parentID      = vmc_ent->volid;
  volint->type          = (vmc_ent->flags & VMCFLAG_RWEXPORT) ? RWVOL : ROVOL;
  volint->status        = VOK;
  if (vmc_ent->flags & VMCFLAG_EXPORT) volint->inUse = 1;
}


/* List all partitions */
int SAFSVolListPartitions(struct rx_call *call, struct pIDs *partIDs)
{
  int i;

  partIDs->partIds[0] = 0;
  for (i = 1; i < 26; i++) partIDs->partIds[i] = -1;
  return 0;
}


/* List a single volume by ID */
afs_int32 SAFSVolListOneVolume(struct rx_call *call, afs_int32 partid, 
			       afs_int32 volid, volEntries *ents)
{
  VMC *ve;

  debug(DEBUG_RPC, "%I => VOL_ListOneVolume(part %d, volid %d)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))), partid, volid);
  memset(ents, 0, sizeof(*ents));
  if (partid) return VL_NOENT;
  if (!(ve = vmc_byvolid(volid))) return VOLSERNOVOL;
  ents->volEntries_len = 1;
  ents->volEntries_val = (volintInfo *)osi_Alloc(sizeof(volintInfo));
  vmc2volint(ve, ents->volEntries_val);
  return 0;
}


/* List all volumes on a partition */
afs_int32 SAFSVolListVolumes(struct rx_call *call, afs_int32 partid, 
			     afs_int32 flags, volEntries *ents)
{
  VMC *ve;
  int i;

  debug(DEBUG_RPC, "%I => VOL_ListVolumes(part %d)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))), partid);
  memset(ents, 0, sizeof(*ents));
  if (partid) return 0;
  i = 0;
  for (i = 0, ve = VMC_all_volumes; ve; ve = ve->anext, i++);
  ents->volEntries_len = i;
  ents->volEntries_val = (volintInfo *)osi_Alloc(i * sizeof(volintInfo));
  memset(ents->volEntries_val, 0, i * sizeof(volintInfo));
  for (i = 0, ve = VMC_all_volumes; ve; ve = ve->anext, i++)
    vmc2volint(ve, ents->volEntries_val + i);
  return 0;
}

afs_int32 SVL_GetEntryByNameU(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ char * volumename,
			      /*OUT*/ uvldbentry * entry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetEntryByNameN(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ char * volumename,
			      /*OUT*/ nvldbentry * entry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetAddrsU(
			/*IN */ struct rx_call *z_call,
			/*IN */ ListAddrByAttributes * inaddr,
			/*OUT*/ afsUUID * uuidp1,
			/*OUT*/ afs_int32 * uniquifier,
			/*OUT*/ afs_int32 * nentries,
			/*OUT*/ bulkaddrs * blkaddrs)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_CreateEntry(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ vldbentry * newentry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_DeleteEntry(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 Volid,
			  /*IN */ afs_int32 voltype)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetNewVolumeId(
			     /*IN */ struct rx_call *z_call,
			     /*IN */ afs_int32 bumpcount,
			     /*OUT*/ afs_int32 * newvolumid)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ReplaceEntry(
			   /*IN */ struct rx_call *z_call,
			   /*IN */ afs_int32 Volid,
			   /*IN */ afs_int32 voltype,
			   /*IN */ vldbentry * newentry,
			   /*IN */ afs_int32 ReleaseType)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_UpdateEntry(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 Volid,
			  /*IN */ afs_int32 voltype,
			  /*IN */ VldbUpdateEntry * UpdateEntry,
			  /*IN */ afs_int32 ReleaseType)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_SetLock(
		      /*IN */ struct rx_call *z_call,
		      /*IN */ afs_int32 Volid,
		      /*IN */ afs_int32 voltype,
		      /*IN */ afs_int32 voloper)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ReleaseLock(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 Volid,
			  /*IN */ afs_int32 voltype,
			  /*IN */ afs_int32 ReleaseType)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ListEntry(
			/*IN */ struct rx_call *z_call,
			/*IN */ afs_int32 previous_index,
			/*OUT*/ afs_int32 * count,
			/*OUT*/ afs_int32 * next_index,
			/*OUT*/ vldbentry * entry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_LinkedList(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ VldbListByAttributes * attributes,
			 /*OUT*/ afs_int32 * nentries,
			 /*OUT*/ vldb_list * linkedentries)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetStats(
		       /*IN */ struct rx_call *z_call,
		       /*OUT*/ vldstats * stats,
		       /*OUT*/ vital_vlheader * vital_header)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetAddrs(
		       /*IN */ struct rx_call *z_call,
		       /*IN */ afs_int32 Handle,
		       /*IN */ afs_int32 spare2,
		       /*OUT*/ VLCallBack * spare3,
		       /*OUT*/ afs_int32 * nentries,
		       /*OUT*/ bulkaddrs * blkaddrs)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ChangeAddr(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 ip1,
			 /*IN */ afs_int32 ip2)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_CreateEntryN(
			   /*IN */ struct rx_call *z_call,
			   /*IN */ nvldbentry * newentry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_GetEntryByIDN(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ afs_int32 Volid,
			    /*IN */ afs_int32 voltype,
			    /*OUT*/ nvldbentry * entry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ReplaceEntryN(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ afs_int32 Volid,
			    /*IN */ afs_int32 voltype,
			    /*IN */ nvldbentry * newentry,
			    /*IN */ afs_int32 ReleaseType)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ListEntryN(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 previous_index,
			 /*OUT*/ afs_int32 * count,
			 /*OUT*/ afs_int32 * next_index,
			 /*OUT*/ nvldbentry * entry)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ListAttributesN(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ VldbListByAttributes * attributes,
			      /*OUT*/ afs_int32 * nentries,
			      /*OUT*/ nbulkentries * blkentries)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_LinkedListN(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ VldbListByAttributes * attributes,
			  /*OUT*/ afs_int32 * nentries,
			  /*OUT*/ nvldb_list * linkedentries)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_UpdateEntryByName(
				/*IN */ struct rx_call *z_call,
				/*IN */ char * volumename,
				/*IN */ VldbUpdateEntry * UpdateEntry,
				/*IN */ afs_int32 ReleaseType)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_RegisterAddrs(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ afsUUID * uuidp,
			    /*IN */ afs_int32 spare1,
			    /*IN */ bulkaddrs * ipaddr)
{
  return RXGEN_OPCODE;
}

afs_int32 SVL_ListAttributesN2(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ VldbListByAttributes * attributes,
			       /*IN */ char * volumename,
			       /*IN */ afs_int32 startindex,
			       /*OUT*/ afs_int32 * nentries,
			       /*OUT*/ nbulkentries * blkentries,
			       /*OUT*/ afs_int32 * nextstartindex)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolCreateVolume(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ afs_int32 partition,
			      /*IN */ char * name,
			      /*IN */ afs_int32 type,
			      /*IN */ afs_int32 parent,
			      /*I/O*/ afs_int32 * volid,
			      /*OUT*/ afs_int32 * trans)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolDeleteVolume(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ afs_int32 trans)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolRestore(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 toTrans,
			 /*IN */ afs_int32 flags,
			 /*IN */ struct restoreCookie * cookie)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolForward(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 fromTrans,
			 /*IN */ afs_int32 fromDate,
			 /*IN */ struct destServer * destination,
			 /*IN */ afs_int32 destTrans,
			 /*IN */ struct restoreCookie * cookie)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolEndTrans(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 trans,
			  /*OUT*/ afs_int32 * rcode)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolClone(
		       /*IN */ struct rx_call *z_call,
		       /*IN */ afs_int32 trans,
		       /*IN */ afs_int32 purgeVol,
		       /*IN */ afs_int32 newType,
		       /*IN */ char * newName,
		       /*I/O*/ afs_int32 * newVol)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSetFlags(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 trans,
			  /*IN */ afs_int32 flags)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolGetFlags(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ afs_int32 trans,
			  /*OUT*/ afs_int32 * flags)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolTransCreate(
			     /*IN */ struct rx_call *z_call,
			     /*IN */ afs_int32 volume,
			     /*IN */ afs_int32 partition,
			     /*IN */ afs_int32 flags,
			     /*OUT*/ afs_int32 * trans)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolDump(
		      /*IN */ struct rx_call *z_call,
		      /*IN */ afs_int32 fromTrans,
		      /*IN */ afs_int32 fromDate)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolGetNthVolume(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ afs_int32 index,
			      /*OUT*/ afs_int32 * volume,
			      /*OUT*/ afs_int32 * partition)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSetForwarding(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ afs_int32 tid,
			       /*IN */ afs_int32 newsite)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolGetName(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 tid,
			 /*OUT*/ char * *tname)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolGetStatus(
			   /*IN */ struct rx_call *z_call,
			   /*IN */ afs_int32 tid,
			   /*OUT*/ struct volser_status * status)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSignalRestore(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ char * name,
			       /*IN */ int type,
			       /*IN */ afs_int32 pid,
			       /*IN */ afs_int32 cloneid)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSetIdsTypes(
			     /*IN */ struct rx_call *z_call,
			     /*IN */ afs_int32 tId,
			     /*IN */ char * name,
			     /*IN */ afs_int32 type,
			     /*IN */ afs_int32 pId,
			     /*IN */ afs_int32 cloneId,
			     /*IN */ afs_int32 backupId)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolMonitor(
			 /*IN */ struct rx_call *z_call,
			 /*OUT*/ transDebugEntries * result)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolPartitionInfo(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ char * name,
			       /*OUT*/ struct diskPartition * partition)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolReClone(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 tid,
			 /*IN */ afs_int32 cloneID)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolNukeVolume(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ afs_int32 partID,
			    /*IN */ afs_int32 volID)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSetDate(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 tid,
			 /*IN */ afs_int32 newDate)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolXListVolumes(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ afs_int32 partID,
			      /*IN */ afs_int32 flags,
			      /*OUT*/ volXEntries * resultXEntriesP)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolXListOneVolume(
				/*IN */ struct rx_call *z_call,
				/*IN */ afs_int32 partID,
				/*IN */ afs_int32 volid,
				/*OUT*/ volXEntries * resultXEntries)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolSetInfo(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 tid,
			 /*IN */ struct volintInfo * status)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolXListPartitions(
				 /*IN */ struct rx_call *z_call,
				 /*OUT*/ struct partEntries * partIDs)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolForwardMultiple(
				 /*IN */ struct rx_call *z_call,
				 /*IN */ afs_int32 fromTrans,
				 /*IN */ afs_int32 fromDate,
				 /*IN */ manyDests * destinations,
				 /*IN */ afs_int32 spare,
				 /*IN */ struct restoreCookie * cookie,
				 /*OUT*/ manyResults * results)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolConvertROtoRWvolume(
				     /*IN */ struct rx_call *z_call,
				     /*IN */ afs_int32 partid,
				     /*IN */ afs_int32 volid)
{
  return RXGEN_OPCODE;
}

afs_int32 SAFSVolGetSize(
			 /*IN */ struct rx_call *z_call,
			 /*IN */ afs_int32 fromTrans,
			 /*IN */ afs_int32 fromDate,
			 /*OUT*/ struct volintSize * size)
{
  return RXGEN_OPCODE;
}

