/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/vmc.c - volume mount cache management */

#include "includes.h"
#include "hostafsd.h"


/* Hash parameters and function */
#define VMC_MASK ((2 << VMC_BITS) - 1)
#define VMC_SIZE (2 << VMC_BITS)
#define VMC_HASH(x) ((x) & VMC_MASK)


/* XXX: allow runtime configuration of default types */
static char *default_fstypes[] = { DEF_FSTYPES , 0 };
static struct vol_mnt_cache *VMC_dev[VMC_SIZE];  /* Hash table by device */
static struct vol_mnt_cache *VMC_vol[VMC_SIZE];  /* Hash table by volid */
VMC *VMC_all_volumes = 0;


static char *mkvolname(VMC *ent)
{
  /* FUTURE: better volume names? */
  return ent->path;
}


#if 1
#define MTABENT struct fstab
#define MTABENTPATH(X) (X)->fs_file
#define MTABENTTYPE(X) (X)->fs_vfstype
#define MTABENTOPTS(X) (X)->fs_mntops
#else
#define MTABENT struct mntent
#define MTABENTPATH(X) (X)->mnt_dir
#define MTABENTTYPE(X) (X)->mnt_type
#define MTABENTOPTS(X) (X)->mnt_opts
#endif

static afs_uint32 mkvolcookie(MTABENT *ep)
{
#ifdef HAVE_STATVFS
  struct statvfs fss;

  if (statvfs(MTABENTPATH(ep), &fss)) return 0;
  return fss.f_blocks ^ fss.f_files;
#else
#ifdef HAVE_STATFS
  struct statfs fss;
  if (statfs(MTABENTPATH(ep), &fss)) return 0;
  return fss.f_type ^ fss.f_blocks ^ fss.f_files;
  /* it'd be nice to include the filesystem ID */
#endif /* HAVE_STATFS */
#endif /* HAVE_STATVFS */
}


/* Lookup a VMC entry by device */
VMC *vmc_bydev(dev_t dev)
{
  VMC *ent;

  for (ent = VMC_dev[VMC_HASH(dev)]; ent; ent = ent->dnext) {
    if (ent->flags & VMCFLAG_NEW) continue;
    if (ent->dev == dev) return ent;
  }
  return 0;
}


/* Look up a VMC entry by volume ID */
VMC *vmc_byvolid(afs_uint32 volid)
{
  VMC *ent;

  for (ent = VMC_vol[VMC_HASH(volid)]; ent; ent = ent->vnext) {
    if (ent->flags & VMCFLAG_NEW) continue;
    if (ent->volid == volid) return ent;
  }
  return 0;
}


/* Look up a VMC entry by name - expensive! */
VMC *vmc_byname(char *name)
{
  VMC *ent;
  afs_uint32 i;

  for (i = 0; i < VMC_SIZE; i++) {
    for (ent = VMC_dev[i]; ent; ent = ent->dnext) {
      if (ent->flags & VMCFLAG_NEW) continue;
      if (!strcmp(ent->name, name)) return ent;
    }
  }
  return 0;
}


static void vmc_add(MTABENT *ep)
{
  FC *fcent;
  VMC *ent, *xent;
  AFSFid fid;
  int i, newflags, level, maxlevel;
  afs_uint32 cookie;
  struct stat sbuf;
  char *name, *x, *y;

  debug(DEBUG_VMC, "vmc_add(%s)", MTABENTPATH(ep));
  debug(DEBUG_VMC, "vmc_add: fstype /%s/, options /%s/",
        MTABENTTYPE(ep), MTABENTOPTS(ep));
  if (stat(MTABENTPATH(ep), &sbuf)) {
    my_syslog(LOG_ERR, "stat %s: %s", MTABENTPATH(ep), strerror(errno));
    return;
  }
  debug(DEBUG_VMC, "vmc_add: device 0x%08x", sbuf.st_dev);

  ent = vmc_bydev(sbuf.st_dev);
  cookie = mkvolcookie(ep);
  if (ent && ent->cookie != cookie) {
    debug(DEBUG_VMC, "vmc_add: cookie mismatch (old 0x%08x != new 0x%08x)",
          ent->cookie, cookie);
    ent = 0;
  }

  if (!ent) {
    debug(DEBUG_VMC, "vmc_add: new entry required");
    ent = (VMC *)osi_Alloc(sizeof(VMC));
    memset(ent, 0, sizeof(VMC));

    /* FUTURE: support persistent volume ID's? */
    ent->flags = VMCFLAG_NEW;
    ent->dev = sbuf.st_dev;
    ent->root_fd = -1;
    ent->max_uniq = 1;
    ent->cookie = cookie;
    if (!strcmp(MTABENTPATH(ep), "/")) {
      /* Force the root to always have a volid of 1, so that clients are
       * able to find it after a server restart without needing a checkv.
       */
      ent->volid = 1;
    } else {
      do { ent->volid = getrandomid(); }
      while (vmc_byvolid(ent->volid));
    }

    if (VMC_all_volumes) VMC_all_volumes->aprev = ent;
    ent->anext = VMC_all_volumes;
    VMC_all_volumes = ent;

    i = VMC_HASH(ent->dev);
    if (VMC_dev[i]) VMC_dev[i]->dprev = ent;
    ent->dnext = VMC_dev[i];
    VMC_dev[i] = ent;

    i = VMC_HASH(ent->volid);
    if (VMC_vol[i]) VMC_vol[i]->vprev = ent;
    ent->vnext = VMC_vol[i];
    VMC_vol[i] = ent;
  }

  if (!ent->path || strcmp(ent->path, MTABENTPATH(ep))) {
    debug(DEBUG_VMC, "vmc_add: path change (or new entry)");
    if (ent->path) osi_Free(ent->path, strlen(ent->path) + 1);
    ent->path = osi_Alloc(strlen(MTABENTPATH(ep)) + 1);
    strcpy(ent->path, MTABENTPATH(ep));

    name = mkvolname(ent);
    if (ent->name) osi_Free(ent->name, strlen(ent->name) + 1);
    ent->name = osi_Alloc(strlen(name) + 1);
    strcpy(ent->name, name);

    fc_inval(ent->volid, 0);
  }

  newflags = VMCFLAG_EXIST;
  try_iopen(ent);
  if (hasmntopt(ep, MNTOPT_RO)) newflags |= VMCFLAG_ROFS;
  for (i = 0; default_fstypes[i]; i++)
    if (!strcmp(MTABENTTYPE(ep), default_fstypes[i]))
      newflags |= VMCFLAG_DEFAULT;

  /* Determine maximum export level */
  if ((newflags & VMCFLAG_IOPEN) && !(newflags & VMCFLAG_ROFS)) maxlevel = 2;
  else if (newflags & VMCFLAG_ROFS) maxlevel = 1;
  else if (enable_dangerous_export) maxlevel = enable_dangerous_export;
  else maxlevel = 0;

  /* Determine actual export level */
  /* XXX: allow runtime configuration of export levels */
  level = (newflags & VMCFLAG_DEFAULT) ? maxlevel : 0;
  if (level) newflags |= VMCFLAG_EXPORT;
  if (level > 1) newflags |= VMCFLAG_RWEXPORT;
  debug(DEBUG_VMC, "vmc_add: flags 0x%04x, max level %d, actual level %d",
        newflags, maxlevel, level);

  /* Determine access modes */
  /* XXX: allow runtime configuration of access modes */
  ent->access_unauth.mode  = ACMODE_WORLD;
  ent->access_foreign.mode = ACMODE_WORLD;
  ent->access_local.mode   = ACMODE_WORLD;
  ent->access_auth.mode    = ACMODE_WORLD | ACMODE_GROUP | ACMODE_USER
                           | ACMODE_OWN   | ACMODE_RW;

  ent->flags = newflags;
  if (newflags & VMCFLAG_EXPORT) {
    debug(DEBUG_VMC, "vmc_add: exported filesystem; updating FID cache");
    memset(&fid, 0, sizeof(fid));
    fid.Volume = ent->volid;
    fid.Vnode = fid.Unique = 1;
    if (!(fcent = fc_byfid(&fid))) {
      fcent = (FC *)osi_Alloc(sizeof(FC));
      memset(fcent, 0, sizeof(FC));

      fcent->kind = FIDKIND_DIR;
      fcent->Volume = ent->volid;
      fcent->Vnode = 1;
      fcent->Unique = 1;
      fcent->path = osi_Alloc(2); strcpy(fcent->path, "/");
      fcent->dev = sbuf.st_dev;
      fcent->ino = sbuf.st_ino;
      fcent->min_dv = FT_ApproxTime();
      fc_add(fcent);
    }
  } else {
    fc_inval(ent->volid, 1);
  }
}


/* Remove a VMC entry */
static void vmc_remove(VMC *ent)
{
  int i;

  debug(DEBUG_VMC, "vmc_remove(%d [%s])", ent->volid, ent->path);
  if (ent->root_fd >= 0) close(ent->root_fd);

  if (ent->anext) ent->anext->aprev = ent->aprev;
  if (ent->aprev) ent->aprev->anext = ent->anext;
  else VMC_all_volumes = ent->anext;

  if (ent->dnext) ent->dnext->dprev = ent->dprev;
  if (ent->dprev) ent->dprev->dnext = ent->dnext;
  else {
    i = VMC_HASH(ent->dev);
    VMC_dev[i] = ent->dnext;
  }

  if (ent->vnext) ent->vnext->vprev = ent->vprev;
  if (ent->vprev) ent->vprev->vnext = ent->vnext;
  else {
    i = VMC_HASH(ent->volid);
    VMC_vol[i] = ent->vnext;
  }

  osi_Free(ent->path, strlen(ent->path) + 1);
  osi_Free(ent->name, strlen(ent->name) + 1);
  fc_inval(ent->volid, 1);
  osi_Free(ent, sizeof(VMC));
}


/* Update the VMC - expensive! */
void vmc_update(void)
{
  int i;
  VMC *ent, *nxtent;
  MTABENT *ep;
  MTABENT e;
  FILE *F;

  debug(DEBUG_VMC, "vmc_update: BEGIN PASS 1");
#if defined(HAVE_GETMNTENT)
  if (!(F = setmntent(MOUNTED, "r"))) {
    my_syslog(LOG_ERR, "setmntent(%s): %s", MOUNTED, strerror(errno));
    return;
  }
  while (ep = getmntent_2(F, &e)) vmc_add(ep);
  endmntent(F);


#elif defined(HAVE_GETFSENT)
  setfsent();
  while (ep = getfsent()) vmc_add(MTABENTPATH(ep));
  endfsent();
#endif

  /* Now free the ones that are no longer good */
  debug(DEBUG_VMC, "vmc_update: BEGIN PASS 2");
  for (i = 0; i < VMC_SIZE; i++) {
    for (ent = VMC_dev[i]; ent; ent = nxtent) {
      nxtent = ent->dnext;
      if (!(ent->flags & VMCFLAG_EXIST)) vmc_remove(ent);
      else ent->flags &=~ VMCFLAG_EXIST;
    }
  }
  debug(DEBUG_VMC, "vmc_update: DONE");
}


/* VMC initialization */
void vmc_init(void)
{
  memset(VMC_dev, 0, sizeof(VMC_dev));
  memset(VMC_vol, 0, sizeof(VMC_vol));
  VMC_all_volumes = 0;
  vmc_update();
}


/* VMC shutdown */
void vmc_shutdown(void)
{
  int i, count = 0;
  VMC *ent;

  for (i = 0; i < VMC_SIZE; i++) {
    while (ent = VMC_dev[i]) {
      count++;
      VMC_dev[i] = ent->dnext;
      osi_Free(ent->path, strlen(ent->path) + 1);
      osi_Free(ent->name, strlen(ent->name) + 1);
      osi_Free(ent, sizeof(VMC));
    }
  }
  memset(VMC_dev, 0, sizeof(VMC_dev));
  memset(VMC_vol, 0, sizeof(VMC_vol));
  debug(DEBUG_VMC, "vmc_shutdown: %d volumes were present", count);
}
