/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* util.h - Miscellaneous utilites */

#include "includes.h"
#include "hostafsd.h"


static int got_sigsys;


/* Generate a random 32-bit ID, guaranteeing that it is not zero. */
afs_uint32 getrandomid(void)
{
  afs_uint32 result;
  des_cblock block;

  do {
    des_generate_random_block(block);
    result = *(afs_uint32 *)block;
    result &= 0x7fffffff;
  } while (!(result & 0xfffffffe));
  return result;
}


/* Find out our IP address (or at least try).
 * The following method was borrowed from Zephyr 2.0.2
 */
afs_uint32 getmyipaddr(void)
{
  char localrealm[REALM_SZ + 1], hostname[MAXHOSTNAMELEN];
  struct hostent *hostent;
  struct sockaddr_in sin;
  struct in_addr hostaddr;
  int s, sinsize = sizeof(sin);
  afs_uint32 myipaddr = 0;

  /* Attempt to find the address of the local Kerberos server */
  if (!krb_get_lrealm(localrealm, 1)
  &&  !krb_get_krbhst(hostname, localrealm, 1)
  &&  (hostent = gethostbyname(hostname))
  &&  hostent->h_addrtype == AF_INET)
    memcpy(&hostaddr, hostent->h_addr, sizeof(hostaddr));

  if (hostaddr.s_addr != INADDR_NONE) {
    /* Try to get the local interface address by connecting a UDP
     * socket to the server address and getting the local address.
     * Some broken operating systems (e.g. Solaris 2.0-2.5) yield
     * INADDR_ANY (zero), so we have to check for that. */
    s = socket(AF_INET, SOCK_DGRAM, 0);
    if (s != -1) {
      memset(&sin, 0, sizeof(sin));
      sin.sin_family = AF_INET;
      memcpy(&sin.sin_addr, &hostaddr, sizeof(hostaddr));
      sin.sin_port = HAFS_PORT;
      if (connect(s, (struct sockaddr *) &sin, sizeof(sin)) == 0
      && getsockname(s, (struct sockaddr *) &sin, &sinsize) == 0
      && sin.sin_addr.s_addr != 0)
        myipaddr = sin.sin_addr.s_addr;
      close(s);
    }
  }

  if (myipaddr == INADDR_NONE || myipaddr == INADDR_LOOPBACK) {
    /* We couldn't figure out the local interface address by the
     * above method.  Try rxi_getaddr, which works on some systems. */
    myipaddr = rxi_getaddr();
  }

  if (myipaddr == INADDR_NONE || myipaddr == INADDR_LOOPBACK) {
    /* We couldn't figure out the local interface address by the
     * above methods.  Try by resolving the local hostname.
     * (This is a pretty broken thing to do.) */
    if (!gethostname(hostname, sizeof(hostname))
    &&  (hostent = gethostbyname(hostname))
    &&  hostent->h_addrtype == AF_INET) {
      memcpy(&hostaddr, hostent->h_addr, sizeof(hostaddr));
      myipaddr = hostaddr.s_addr;
    }
  }

  if (myipaddr == INADDR_NONE) {
    /* Feh; nothing worked.  Default to the loopback address, so
     * at least it works for _someone_. */
    myipaddr = INADDR_LOOPBACK;
  }
  return myipaddr;
}


/* Generate a full pathname from two components */
char *genpath(char *base, char *rel)
{
  int bl, rl;
  char *path;

  bl = strlen(base);
  while (bl && base[bl-1] == '/') bl--;

  while (*rel == '/') rel++;
  rl = strlen(rel);
  while (rl && rel[rl-1] == '/') rl--;

  if (rl) {
    path = osi_Alloc(rl + bl + 2);
    strncpy(path, base, bl);
    path[bl] = '/';
    strncpy(path + bl + 1, rel, rl);
    path[bl + rl + 1] = 0;
  } else if (bl) {
    path = osi_Alloc(bl + 1);
    strncpy(path, base, bl);
    path[bl] = 0;
  } else {
    path = osi_Alloc(2);
    strcpy(path, "/");
  }
  return path;
}


/* Hash function for filenames and paths.  Don't change this!
 * It is the same as the hash function used in AFS directories.
 */
int namehash(char *name, int buckets, int seed)
{
  int hval = seed, tval;

  while (*name) hval = (hval * 173) + *name++;
  tval = hval & (buckets - 1);
  return tval ? hval < 0 ? buckets - tval : tval : 0;
}


/* Open a file by inode number */
int my_iopen(VMC *vmc, ino_t ino, int oflag)
{
#if defined(__linux__) && defined(IMAGIC_IOC_OPEN)
  struct imagic_openargs openargs;
  int fd;
#endif

  if (!(vmc->flags & VMCFLAG_IOPEN)) {
    errno = ENOENT;
    return -1;
  }

#ifdef __linux__
#ifdef IMAGIC_IOC_OPEN
  if (vmc->root_fd) {
    openargs.oa_ino = ino;
    openargs.oa_mode = oflag;
    fd = ioctl(vmc->root_fd, IMAGIC_IOC_OPEN, &openargs);
    if (fd < 0 && errno != ENOTTY) return fd;
  }
#endif
#endif /* __linux__ */

#ifdef HAVE_IOPEN
  return iopen(vmc->dev, ino, oflag);
#endif

  errno = ENOENT;
  return -1;
}


static void handle_sigsys()
{
  got_sigsys = 1;
}


/* Determine whether iopen is supported for a filesystem */
void try_iopen(VMC *vmc)
{
#ifdef SIGSYS
  void (*handler)();
#endif
#if defined(__linux__) && defined(IMAGIC_IOC_OPEN)
  struct imagic_openargs openargs;
#endif
  struct stat sbuf;
  int fd;

#ifdef HAVE_IOPEN
#ifdef SIGSYS
  got_sigsys = 0;
  handler = signal(SIGSYS, got_sigsys);
#endif
  if (stat(vmc->path, &sbuf)) goto not_iopen;
  fd = iopen(vmc->dev, sbuf.st_ino, O_RDONLY);
#ifdef SIGSYS
  signal(SIGSYS, handler);
  if (got_sigsys) goto not_iopen;
#endif
  if (fd >= 0) {  /* Hey! It should not allow opening directories */
    close(fd);
    goto not_iopen;
  }
  if (errno != ENOENT) goto not_iopen;
  vmc->flags |= VMCFLAG_IOPEN;
  return;

not_iopen:
#endif /* HAVE_IOPEN */
  
#ifdef __linux__
#ifdef IMAGIC_IOC_OPEN
  /* XXX: implement test for imagic */
#endif /* IMAGIC_IOC_OPEN */
#endif /* __linux__ */
  return;
}
