/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/safety.c - Safe operations on local files/directories */


#include "includes.h"
#include "hostafsd.h"


/* Safely stat the file corresponding to an FC entry */
afs_uint32 safe_stat(FC *ent, struct stat *sbuf)
{
  int code;

  debug(DEBUG_SAFETY, "safe_stat(%u/%u/%u, [%s])",
        ent->Volume, ent->Vnode, ent->Unique, ent->fullpath);
  debug(DEBUG_SAFETY, "safe_stat: kind %d, dev 0x%08x, ino %u",
        ent->kind, (afs_uint32)ent->dev, (afs_uint32)ent->ino);
  if (lstat(ent->fullpath, sbuf)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_stat: lstat failed (%d)", code);
    return code;
  }
  debug(DEBUG_SAFETY, "safe_stat: found dev 0x%08x, ino %d, mode 0%03o",
        (afs_uint32)sbuf->st_dev, (afs_uint32)sbuf->st_ino, (afs_uint32)sbuf->st_mode);
  if (ent->dev != sbuf->st_dev || ent->ino != sbuf->st_ino) {
    debug(DEBUG_SAFETY, "safe_stat: wrong dev/inode");
    return ENOENT;
  }
  if ((ent->kind == FIDKIND_FILE    && !S_ISREG(sbuf->st_mode))
  ||  (ent->kind == FIDKIND_DIR     && !S_ISDIR(sbuf->st_mode))
  ||  (ent->kind == FIDKIND_LINK    && !S_ISLNK(sbuf->st_mode))) {
    debug(DEBUG_SAFETY, "safe_stat: wrong type");
    return ENOENT;
  }
  return 0;
}


/* Safely open the file corresponding to an FC entry */
afs_uint32 safe_open(FC *ent, int oflag, int *fd)
{
  int code;

  debug(DEBUG_SAFETY, "safe_open(%u/%u/%u, [%s], %d)",
        ent->Volume, ent->Vnode, ent->Unique, ent->fullpath, oflag);
  debug(DEBUG_SAFETY, "safe_open: kind %d, dev 0x%08x, ino %u",
        ent->kind, (afs_uint32)ent->dev, (afs_uint32)ent->ino);
  *fd = -1;
  if (!ent->vmc) return VNOVOL;
  if (ent->vmc->flags & VMCFLAG_IOPEN) 
    *fd = my_iopen(ent->vmc, ent->ino, oflag);
  else
    *fd = open(ent->fullpath, oflag);
  if (*fd < 0) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_open: open/iopen failed (%d)", code);
    return code;
  }
  return 0;
}


/* Safely open the directory corresponding to an FC entry */
afs_uint32 safe_opendir(FC *ent, DIR **D, int *dfd)
{
  struct stat sbuf;
  int code;

  debug(DEBUG_SAFETY, "safe_opendir(%u/%u/%u, [%s])",
        ent->Volume, ent->Vnode, ent->Unique, ent->fullpath);
  debug(DEBUG_SAFETY, "safe_opendir: kind %d, dev 0x%08x, ino %u",
        ent->kind, (afs_uint32)ent->dev, (afs_uint32)ent->ino);
  if (D) *D = 0;
  *dfd = -1;
  if (chdir(ent->fullpath)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_opendir: chdir failed (%d)", code);
    return code;
  }

  if (lstat(".", &sbuf)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_opendir: lstat failed (%d)", code);
    chdir("/");
    return code;
  }
  debug(DEBUG_SAFETY, "safe_opendir: found dev 0x%08x, ino %d, mode 0%03o",
        (afs_uint32)sbuf.st_dev, (afs_uint32)sbuf.st_ino, (afs_uint32)sbuf.st_mode);
  if (sbuf.st_dev != ent->dev || sbuf.st_ino != ent->ino) {
    debug(DEBUG_SAFETY, "safe_opendir: wrong dev/inode");
    chdir("/");
    return ENOENT;
  }
  if (D) {
    if (!(*D = opendir("."))) {
      code = errno;
      debug(DEBUG_SAFETY, "safe_opendir: opendir failed (%d)", code);
      chdir("/");
      return code;
    }
  }
  if ((*dfd = open(".", O_RDONLY, 0)) < 0) {
    code = errno;
    if (D) {
      closedir(*D);
      *D = 0;
    }
    debug(DEBUG_SAFETY, "safe_opendir: open failed (%d)", code);
    chdir("/");
    return code;
  }
  chdir("/");
  return 0;
}


/* Safely stat a filename in a given directory */
afs_uint32 safe_dirstat(int dfd, char *name, struct stat *sbuf)
{
  int code;

  debug(DEBUG_SAFETY, "safe_dirstat(%d, %s)", dfd, name);
  if (fchdir(dfd)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_dirstat: fchdir failed (%d)", code);
    return code;
  }
  if (lstat(name, sbuf)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_dirstat: lstat failed (%d)", code);
    chdir("/");
    return code;
  }
  debug(DEBUG_SAFETY, "safe_dirstat: found dev 0x%08x, ino %d, mode 0%03o",
        (afs_uint32)sbuf->st_dev, (afs_uint32)sbuf->st_ino, (afs_uint32)sbuf->st_mode);
  chdir("/");
  return 0;
}


/* Safely read the symbolic link corresponding to an FC entry */
afs_uint32 safe_readlink(FC *ent, char **bp)
{
  VMC *vmcent;
  struct stat sbuf;
  char *b = 0, *b2;
  afs_uint32 code;
  int count, r;

  debug(DEBUG_SAFETY, "safe_readlink(%u/%u/%u, [%s])",
        ent->Volume, ent->Vnode, ent->Unique, ent->fullpath);
  debug(DEBUG_SAFETY, "safe_readlink: kind %d, dev 0x%08x, ino %u",
        ent->kind, ent->dev, ent->ino);
  *bp = 0;
  if (lstat(ent->fullpath, &sbuf)) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_readlink: lstat failed (%d)", code);
    goto fail;
  }

  debug(DEBUG_SAFETY, "safe_readlink: found dev 0x%08x, ino %d, mode %d",
        sbuf.st_dev, sbuf.st_ino, sbuf.st_mode);

  if (sbuf.st_dev != ent->dev || sbuf.st_ino != ent->ino
  ||  !S_ISLNK(sbuf.st_mode)) {
    code = ENOENT;
    debug(DEBUG_SAFETY, "safe_readlink: wrong dev/inode or mode");
    goto fail;
  }
  count = sbuf.st_size;

  for (;;) {
    debug(DEBUG_SAFETY, "safe_readlink: count = %d", count);
    b = osi_Alloc(count + 1);
    r = readlink(ent->fullpath, b, count);
    if (r < 0) {
      code = errno;
      debug(DEBUG_SAFETY, "safe_readlink: readlink failed (%d)", code);
      goto fail;
    }
    if (r < count) {
      b2 = osi_Alloc(r + 1);
      strncpy(b2, b, r);
      osi_Free(b, count + 1);
      count = r;
      b = b2;
      break;
    }
    if (r == count) break;
    osi_Free(b, count + 1);
    count = r;
  }

  b[count] = 0;
  *bp = b;
  return 0;

fail:
  if (b) osi_Free(b, count + 1);
  return code;
}


afs_uint32 safe_tmpfile(int *fd)
{
  char *n;
  int code;

  n = osi_Alloc(strlen(GSAFE_DIR) + 40);
  sprintf(n, "%s/tf.%d.%d", GSAFE_DIR, LWP_Index(), FT_ApproxTime());
  *fd = open(n, O_RDWR | O_CREAT | O_TRUNC, 0600);
  if (*fd < 0) {
    code = errno;
    debug(DEBUG_SAFETY, "safe_tmpfile: open %s failed (%d)", n, code);
    return code;
  }
/*  unlink(n); */
  osi_Free(n, strlen(GSAFE_DIR) + 40);
  return 0;
}
