/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* logging.c - Generate logging messages */

#include "includes.h"
#include "hostafsd.h"

#define MY_LOGLEN 256

static void numfill(afs_uint32 val, int base, char *buf)
{
  int len = 0, dig, i;

  while (val) {
    dig = val % base;
    val = (val - dig) / base;
    if (dig < 10) dig = dig + '0';
    else          dig = dig + 'a' - 10;
    buf[len++] = dig;
  }
  if (!len) buf[len++] = '0';
  for (i = 0; i < (len + 1) / 2; i++) {
    dig = buf[i];
    buf[i] = buf[len - i - 1];
    buf[len - i - 1] = dig;
  }
  buf[len] = 0;
}

/* This function handles the actual work of formatting things
 * to be given to syslog().  We implement our own snprintf-like
 * mechanism internally, to insure we don't produce too long a
 * message.  Our version provides %d, %u, %o, %x, and %s, as well
 * as some more unusual ones (see below).  We support min-width,
 * zero-fill, and left justification.
 *
 *   %I  formats a network-order IP address as a dotted-quad
 *   %R  formats AFS rights in human-readable form
 */
static void my_vsyslog(int level, char *fmt, va_list ap)
{
  char buf[MY_LOGLEN + 1], numbuf[20], numbuf2[20], *p, *x;
  int avail = MY_LOGLEN;
  int ljust, zfill, width, l, j;
  afs_uint32 val;
  afs_int32 sval;

  p = buf;
  while (*fmt && avail) {
    if (*fmt != '%') {
      *p++ = *fmt++;
      avail--;
      continue;
    } else {
      ljust = zfill = width = 0;
      while (*++fmt) {
        switch (*fmt) {
          case '-': ljust = 1; continue;
          case '%': ljust = zfill = width = 0; x = "%"; break;

          case 's':
            x = va_arg(ap, char *);
            if (!x) x = "(null)";
            break;

          case 'I':
            x = numbuf;
            val = va_arg(ap, afs_uint32);
            val = ntohl(val);
            sprintf(numbuf, "%u.%u.%u.%u",
                    (val & 0xff000000) >> 24, (val & 0x00ff0000) >> 16,
                    (val & 0x0000ff00) >> 8,  (val & 0x000000ff));
            break;

          case 'R':
            val = va_arg(ap, afs_uint32);
            x = numbuf;
            if (val & PRSFS_READ)   *x++ = 'r';
            if (val & PRSFS_LOOKUP) *x++ = 'l';
            if (val & PRSFS_INSERT) *x++ = 'i';
            if (val & PRSFS_DELETE) *x++ = 'd';
            if (val & PRSFS_WRITE)  *x++ = 'w';
            if (val & PRSFS_LOCK)   *x++ = 'k';
            if (val & PRSFS_ADMINISTER)  *x++ = 'a';
            *x = 0;
            x = numbuf;
            break;

          case 'd':
            x = numbuf;
            sval = va_arg(ap, afs_int32);
            if (sval < 0) {
              numbuf[0] = '-';
              numfill(- sval, 10, numbuf + 1);
            } else {
              numfill(sval, 10, numbuf);
            }
            break;

          case 'u':
            x = numbuf;
            val = va_arg(ap, afs_uint32);
            numfill(val, 10, numbuf);
            break;

          case 'o':
            x = numbuf;
            val = va_arg(ap, afs_uint32);
            numfill(val, 8, numbuf);
            break;

          case 'x':
            x = numbuf;
            val = va_arg(ap, afs_uint32);
            numfill(val, 16, numbuf);
            break;

          case '0':
            if (!width) {
              zfill = 1;
              continue;
            }
            /* else fall through to other digits */

          default:
            if (*fmt >= '0' && *fmt <= '9') {
              width = (width * 10) + (*fmt - '0');
              continue;
            } else {
              ljust = zfill = width = 0;
              x = "";
              break;
            }
        }
        break;
      }
      if (!*fmt) break;
      fmt++;
      l = strlen(x);
      if (l > avail) l = avail;
      if (!width) width = l;
      if (width > avail) width = avail;
      if (l < width) j = width - l;
      else j = 0;
      if (!ljust) while (j--) {
        *p++ = zfill ? '0' : ' ';
        avail--;
      }
      strncpy(p, x, l);
      p += l;
      x -= l;
      if (ljust) while (j--) {
        *p++ = zfill ? '0' : ' ';
        avail--;
      }
    }
  }
  *p = 0;

  syslog(level, "%s", buf);
}


/* Put something in the system log (safely) */
void my_syslog(int level, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  my_vsyslog(level, fmt, ap);
  va_end(ap);
}


/* Log debugging messages */
void debug(int kind, char *fmt, ...)
{
  va_list ap;

  if (kind & hafs_debug) {
    va_start(ap, fmt);
    my_vsyslog(LOG_DEBUG, fmt, ap);
    va_end(ap);
  }
}


/* Convert a kerberos error code to its com_err equivalent */
afs_uint32 kerberize_error(afs_uint32 kerr)
{
  if (!kerr)             return 0;
  if (kerr < 0)          return errno;
  if (kerr <= KNAME_FMT) return kerr + ERROR_TABLE_BASE_krb;
  return kerr;
}
