/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/hostafsd.h - Common declarations, definitions, and macros */


/* Various global parameters - these are not intended to be
 * configurable, but can be changed if needed.
 */
#define HAFS_VERSION  "0.11"       /* Version number */
#define HAFS_VERCODE  0x001100     /* Version code */

#define USE_DFS_SEMANTICS          /* pretend to be a DFS translator */
#define VMC_BITS      3            /* # bits of volume ID in VMC hash */
#define FC_VUBITS     6            /* # bits of vnode/uniq in FC hash */
#define FC_VOLBITS    3            /* # bits of volume ID in FC hash */
#define FC_STALE_TIME (3 * 3600)   /* time before FC entries become stale */
#define CI_STALE_TIME (30 * 60)    /* time before CI entries become stale */
#define DAEMON_TIME 300            /* How often to run periodic functions */
#define HAFS_CB_TIME  10           /* Length of callbacks, in seconds */

#define GSAFE_DIR "/var/spool/hostafs" /* Global safe temp directory */

/* Debugging flags */
#define DEBUG_MAIN       0x00000001    /* main program                */
#define DEBUG_CONFIG     0x00000002    /* run-time configuration      */
#define DEBUG_RPC        0x00000004    /* calls to RPC's              */
#define DEBUG_VMC        0x00000010    /* vol/mount cache changes     */
#define DEBUG_FC_UPDATE  0x00000020    /* FID cache changes           */
#define DEBUG_FC_LOOKUP  0x00000040    /* FID cache lookups           */
#define DEBUG_CI         0x00000080    /* client info                 */
#define DEBUG_DIRGEN     0x00000100    /* directory generation        */
#define DEBUG_SAFETY     0x00000200    /* safe file access            */
#define DEBUG_ACCESS     0x00000400    /* access control              */
#define DEBUG_AUTH       0x00000800    /* authorization checks        */

#define ACMODE_WORLD     0x0001 /* May use world bits */
#define ACMODE_GROUP     0x0002 /* May use group bits, if gid matches */
#define ACMODE_USER      0x0004 /* May use owner bits, if uid matches */
#define ACMODE_OWN       0x0008 /* May be treated as owner */
#define ACMODE_RW        0x0010 /* May use write bits (subject to above) */
#define ACMODE_SRCH      0x0020 /* May always use 0001 on directories */
typedef struct access_mode {
  int mode;                     /* Access mode flags (ACMODE_*) */
  uid_t uid;                    /* required UID */
  gid_t gid;                    /* required GID */
} ACMODE;

/* Volume mount cache entry */
#define VMCFLAG_IOPEN    0x0001 /* Filesystem supports iopen() */
#define VMCFLAG_ROFS     0x0002 /* Filesystem is mounted read-only */
#define VMCFLAG_DEFAULT  0x0004 /* Export by default */
#define VMCFLAG_EXPORT   0x0010 /* Volume is exported */
#define VMCFLAG_RWEXPORT 0x0020 /* Volume may be exported read-write */
#define VMCFLAG_NEW      0x4000 /* Marks entries not yet ready for use */
#define VMCFLAG_EXIST    0x8000 /* Tag used in updating from mnttab */
typedef struct vol_mnt_cache {
  struct vol_mnt_cache *anext;  /* Next     entry in all-volumes chain */
  struct vol_mnt_cache *aprev;  /* Previous entry in all-volumes chain */
  struct vol_mnt_cache *dnext;  /* Next     entry in device table */
  struct vol_mnt_cache *dprev;  /* Previous entry in device table */
  struct vol_mnt_cache *vnext;  /* Next     entry in volid table */
  struct vol_mnt_cache *vprev;  /* Previous entry in volid table */

  char *path;                   /* Path where this is mounted */
  char *name;                   /* Volume name */
  afs_uint32 volid;                /* Volume ID; 0 if bogus volume */
  dev_t dev;                    /* Device of this filesystem */
  int flags;                    /* Flags */
  int root_fd;                  /* Cached root file descriptor (for iopen) */
  afs_uint32 max_uniq;             /* Last uniqifier used */
  afs_uint32 cookie;               /* cookie to identify this volume */

  ACMODE access_unauth;         /* Access mode for unauthenticated users */
  ACMODE access_foreign;        /* Access mode for foreign users */
  ACMODE access_local;          /* Access mode for local/anonymous users */
  ACMODE access_auth;           /* Access mode for local auth'd users */
} VMC;


/* FID cache entry */
#define FIDKIND_FILE      0     /* Regular file */
#define FIDKIND_DIR       1     /* Directory */
#define FIDKIND_LINK      2     /* Symbolic link */
#define FIDKIND_MOUNT     3     /* Filesystem mount point */
#define FIDKIND_SPECIAL   4     /* Special inode (do not serve) */
typedef struct fid_cache {
  struct fid_cache *next;       /* Next entry in by-fid bucket */
  struct fid_cache *prev;       /* Previous entry in by-fid bucket */
  struct fid_cache *nextpath;   /* Next entry in by-path bucket */
  struct fid_cache *prevpath;   /* Previous entry in by-path bucket */
  struct fid_cache *nextsib;    /* Next sibling */
  struct fid_cache *prevsib;    /* Previous sibling */
  struct fid_cache *kids;       /* Children (if a directory) */
  struct fid_cache *parent;     /* Pointer to parent, if present */

  afs_uint32 Volume;               /* Volume this entry is in */
  afs_uint32 Vnode;                /* Vnode number */
  afs_uint32 Unique;               /* Uniqifier */

  int kind;                     /* Type of this entry */
  afs_uint32 mount;                /* Mounted volume if FIDKIND_MOUNT */
  char *path;                   /* path to file within vol if !FIDKIND_MOUNT */
  char *fullpath;               /* full path to file (cached) */

  dev_t dev;                    /* Device where this lives */
  ino_t ino;                    /* inode number in filesystem */

  VMC *vmc;                     /* VMC entry for volume containing this FID */
  afs_uint32 min_dv;               /* Minimum data version to report */
  afs_uint32 lastuse;              /* Last time this entry was used */
} FC;


/* Client host information */
typedef struct client_host {
  struct client_host *next;     /* Next in hash bucket */
  struct client_host *prev;     /* Previous in hash bucket */
  afs_uint32 hostaddr;             /* IP address of client host */
  short port;                   /* Port this client came from */

  afs_uint32 lastheard;            /* last time we heard from this host */
  int refcount;                 /* References to this structure */

  struct rx_connection *conn;   /* Rx connection to this client */
  des_cblock mungekey;          /* Key for obfuscating vnode numbers */
  des_key_schedule mungeks;     /* Key sched for obfuscating vnode numbers */
} CH;


/* Client information */
#define CIFLAG_REQUEST   0x0001 /* Set if nonstandard access requested */
typedef struct client_info {
  struct client_info *next;     /* Next in hash bucket */
  struct client_info *prev;     /* Previous in hash bucket */
  afs_uint32 hostaddr;             /* IP address of client host */
  char *princ;                  /* Kerberos principal (0 if unauthenticated) */

  int mode;                     /* Access mode */
  char *lcluser;                /* Name of local user, 0 if anonymous */
  uid_t uid;                    /* uid of local user, from passwd */
  gid_t gid;                    /* gid of local user, from passwd */
  int ngroups;                  /* number of groups in group list */
  gid_t *groups;                /* list of groups */

  int flags;                    /* Flags */
  afs_uint32 lastuse;              /* last use of this entry */
  afs_uint32 expires;              /* expiration date of tickets */
} CI;


/* Access modes */
#define AMODE_STAT       0      /* Get status of file */
#define AMODE_READ       1      /* Read file/directory contents */
#define AMODE_WRITE      2      /* Write file contents */
#define AMODE_LOOKUP     3      /* Look up directory entry */


typedef struct dir_state DS;


/* Global variables */
extern int hafs_debug;                   /* debugging level */
extern int enable_dangerous_export;      /* max level for "dangerous" FS */
extern afs_uint32 my_serverid;              /* our fileserver IP address */
extern VMC *VMC_all_volumes;             /* head of all-volumes chain */


/* client.c - client management */
void ci_init(void);                       /* initialize CI subsystem */
void ci_shutdown(void);                   /* shut down CI subsystem */
void ci_clean(void);                      /* clean stale CI entries */
afs_uint32 ci_setauth(CI *, int, char *);    /* set authorization mode */
CI *ci_getinfo(struct rx_call *);         /* get client info from connection */
CH *ci_gethost(struct rx_call *);         /* get host info from connection */
int ci_userok(CI *, char *);              /* check if client may act as user */
int ci_rootok(CI *);                      /* check if client may act as root */
void ci_munge_vnode(CH *, afs_uint32 *, afs_uint32 *);
void ci_demunge_vnode(CH *, afs_uint32, afs_uint32 *, afs_uint32 *);


/* access.c - Access control */
afs_uint32 access_rights(FC *, AFSFetchStatus *, CI *);     /* compute rights */
afs_uint32 access_ok(FC *, AFSFetchStatus *, CI *, int);    /* check access */
afs_uint32 access_acl(FC *, AFSFetchStatus *, AFSOpaque *); /* generate ACL */


/* vmc.c - volume moount cache management */
void vmc_init(void);                      /* VMC initialization */
void vmc_update(void);                    /* Update the VMC - expensive! */
void vmc_shutdown(void);                  /* VMC shutdown */
VMC *vmc_bydev(dev_t);                    /* Lookup a VMC entry by device */
VMC *vmc_byvolid(afs_uint32);                /* Lookup a VMC entry by volume ID */
VMC *vmc_byname(char *);                  /* Lookup a VMC entry by name */


/* fidcache.c - FID cache management */
void fc_init(void);                       /* FID cache initialization */
void fc_shutdown(void);                   /* FID cache shutdown */
FC *fc_byfid(AFSFid *);                   /* Get FC entry by FID */
FC *fc_bypath(char *);                    /* Get FC entry by pathname */
FC *fc_findmount(afs_uint32, afs_uint32);       /* Get FC entry for a mount point */
void fc_add(FC *);                        /* Add a FID cache entry */
void fc_remove(FC *);                     /* Remove a FID cache entry */
void fc_inval(afs_uint32, int);              /* Invalidate volume's FC entries */
int fc_isstale(FC *, afs_uint32);            /* Test whether FC entry is stale */
void fc_cleanup(void);                    /* periodic FID cache cleanup */


/* safety.c - Safe operations on local files/directories */
afs_uint32 safe_stat(FC *, struct stat *);           /* stat file */
afs_uint32 safe_open(FC *, int, int *);              /* open file */
afs_uint32 safe_readlink(FC *, char **);             /* read symlink */
afs_uint32 safe_opendir(FC *, DIR **, int *);        /* open directory */
afs_uint32 safe_dirstat(int, char *, struct stat *); /* stat file in directory */
afs_uint32 safe_tmpfile(int *);                      /* get temporary file FD */


/* dirgen.c - Directory generation */
afs_uint32 dirgen(CH *, FC *, int *);                /* Generate directory */
afs_uint32 gen_fid(FC *, int, char *, afs_uint32 *, afs_uint32 *); /* Generate FID */
afs_uint32 gen_direntry(FC *, int, char *, DS *);    /* Generate dir entry */


/* ds_*.c - Directory generation support */
afs_uint32 ds_init(DS **, CH *, FC *, int, int);     /* Prepare directory */
afs_uint32 ds_add(DS *, char *, afs_uint32, afs_uint32);   /* Add directory entry */
afs_uint32 ds_finalize(DS *);                        /* Finish directory */
void ds_free(DS *);                               /* Free directory */


/* bulk.c - Rx bulk data transfer */
afs_uint32 bulk_xmit_fd(struct rx_call *, int, afs_uint32);     /* xmit from file */
afs_uint32 bulk_xmit_str(struct rx_call *, char *, afs_uint32); /* xmit string */


/* logging.c - generate logging and error messages */
void my_syslog(int, char *, ...);         /* Safe syslog */
void debug(int, char *, ...);             /* Safe debugging log */
afs_uint32 kerberize_error(afs_uint32);         /* Convert Kerberos error codes */


/* util.c - Miscellaneous utilities */
afs_uint32 getrandomid(void);                /* Return a random 32-bit ID */
afs_uint32 getmyipaddr(void);                /* Try to find our IP address */
char *genpath(char *, char *);            /* Generate path to a vnode */
int namehash(char *, int, int);           /* filename/path hash function */
int my_iopen(VMC *, ino_t, int);          /* open file by inode */
void try_iopen(VMC *);                    /* test iopen ability */
