/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/hostafsd.c - Main program */

#include "includes.h"
#include "hostafsd.h"

extern int optind, opterr, getopt();
extern char *optarg;

int hafs_debug;
int enable_dangerous_export;
afs_uint32 my_serverid;
char *hrealm, *hinst, *hprinc, lrealm[REALM_SZ];

static char *argv0;

static void usage(int status, char *msg)
{
  if (msg) fprintf(stderr, "%s: %s\n", argv0, msg);
  fprintf(stderr, "Usage: %s [options]\n", argv0);
  fprintf(stderr, "  -d xxx    Set debugging flags\n");
  fprintf(stderr, "  -h        Display this help message\n");
  fprintf(stderr, "  -n        Don't run in the background\n");
  fprintf(stderr, "  -U        Allow unsafe exports (do not use!)\n");
  fprintf(stderr, "  -p princ  Use princ instead of rcmd\n");
  fprintf(stderr, "  -r rlm    Run in kerberos realm rlm\n");
  fprintf(stderr, "  -i inst   Run as instance instance\n");
  exit(status);
}


static unsigned short lookup_port(char *svc, unsigned short def)
{
  struct servent *ent;

  ent = getservbyname(svc, "udp");
  if (ent) return ent->s_port;
  else return htons(def);
}


static afs_int32 getkey(void *refcon, afs_int32 kvno, struct ktc_encryptionKey *key)
{
  afs_int32 code;
  char instance[INST_SZ + 1];
  char aname[ANAME_SZ + 1];

  if (hinst)
    strcpy(instance, hinst);
  else
    strcpy(instance, "*");
  if (hprinc)
    strcpy(aname, hprinc);
  else
    strcpy(aname, "rcmd");

  code = read_service_key(aname, instance, hrealm, 0, "/etc/srvtab", key->data);
  return kerberize_error(code);
}


/* Process a SIGHUP */
static void handle_HUP(void)
{
  debug(DEBUG_MAIN, "got SIGHUP");
  vmc_update();
}

static void signal_HUP()
{
  IOMGR_SoftSig(handle_HUP, 0);
}


/* Periodic cleanup */
void cleanup_lwp(void)
{
  afs_uint32 stale;
  int i;
  FC *ent, *nxtent;

  while (1) {
    IOMGR_Sleep(DAEMON_TIME);
    fc_cleanup();
    vmc_update();
    ci_clean();
  }
}

int main(int argc, char **argv)
{
  afs_int32 code;
  int c, fd, pid, nofork = 0;
  PROCESS fcc_pid;
  struct rx_securityClass *sc[3];
  struct rx_service *fs_svc;
  struct rx_service *vol_svc;
  struct rx_service *vl_svc;
  struct rx_service *hafs_svc;
  struct ktc_encryptionKey key;

  argv0 = strrchr(argv[0], '/');

  /* Cache local realm in case none provided */
  krb_get_lrealm(lrealm, 1);
  hrealm = lrealm;
  hprinc = hinst = NULL;

  hafs_debug = 0;
  if (!argv0) argv0 = argv[0];
  enable_dangerous_export = 0;
  while ((c = getopt(argc, argv, "hd:nUp:i:r:")) != EOF) {
    switch (c) {
      case 'd': hafs_debug = strtoul(optarg, 0, 0); continue;
      case 'h': usage(0, 0);
      case 'n': nofork = 1; continue;
      case 'U': enable_dangerous_export = 1; continue;
      case 'r': hrealm = optarg; continue;
      case 'p': hprinc = optarg; continue;
      case 'i': hinst = optarg; continue;
      default:  usage(1, "Invalid option!");
    }
  }
  if (optind < argc) usage(1, "Too many arguments!");

  if (!nofork) {
    pid = fork();
    if (pid < 0) {
      fprintf(stderr, "%s: fork: %s\n", argv0, strerror(errno));
      exit(-1);
    } else if (pid) {
      exit(0);
    } else {
#if defined(__sun) && defined(__SVR4)
      setsid();
#endif

      fd = open("/dev/null", O_RDONLY);
      dup2(fd, 0);
      dup2(fd, 1);
      dup2(fd, 2);
      if (fd > 2) close(fd);

#if defined(SYSV) || defined(hpux)
      setpgrp();
#endif

#if defined(TIOCNOTTY) && !(defined(__sun) && defined(__SYSV))
      fd = open("/dev/tty", O_RDWR);
      if (fd >= 0) {
        ioctl(fd, TIOCNOTTY, 0);
        close(fd);
      }
#endif /* TIOCNOTTY && !(__sun && __SYSV) */
    }
  }

  openlog("hostafsd", LOG_PID, LOG_USER);
  my_syslog(LOG_NOTICE, "hostafsd %s startup", HAFS_VERSION);

  if (code = getkey(0, 0, &key)) {
    my_syslog(LOG_ERR, "failed to get service key: %d", code);
    exit(0);
  }
  des_init_random_number_generator(key.data);

  rx_Init(0);
  my_serverid = getmyipaddr();
  sc[0] = rxnull_NewServerSecurityObject();
  sc[1] = 0;
  sc[2] = rxkad_NewServerSecurityObject(rxkad_clear, 0, getkey, 0);

  fc_init();
  vmc_init();

  fs_svc   = rx_NewService(lookup_port(AFS_NAME, AFS_PORT), AFS_SVCID,
                           "Fileserver", sc, 3, RXAFS_ExecuteRequest);

  vol_svc  = rx_NewService(lookup_port(VOL_NAME, VOL_PORT), VOL_SVCID,
                           "Volume server", sc, 3, AFSVolExecuteRequest);

  vl_svc   = rx_NewService(lookup_port(VL_NAME, VL_PORT), VL_SVCID,
                           "Volume location server", sc, 3, VL_ExecuteRequest);

  hafs_svc = rx_NewService(lookup_port(HAFS_NAME, HAFS_PORT), HAFS_SVCID,
                           "hostafs control", sc, 3, HAFS_ExecuteRequest);

  LWP_CreateProcess(cleanup_lwp, AFS_LWP_MINSTACKSIZE,
                    LWP_NORMAL_PRIORITY, "Periodic cleanup", 0, &fcc_pid);

  signal(SIGHUP, signal_HUP);
  rx_StartServer(1);
}

void
add_to_error_table(struct et_list *new_table)
{

}
