/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* fsprocs.c - fileserver RPC implementations */

#include "includes.h"
#include "hostafsd.h"


/** Get everything we need to know about a file **/
static afs_uint32 getfinfo(CH *chost, AFSFid *fid, FC **fcp, AFSFetchStatus *st)
{
  struct stat sbuf;
  afs_uint32 code;
  VMC *vmc_ent;

  memset(st, 0, sizeof(*st));
  *fcp = fc_byfid(fid);
  if (!*fcp) return ENOENT;
  switch ((*fcp)->kind) {
    case FIDKIND_FILE:
      if (code = safe_stat(*fcp, &sbuf)) return code;
      st->FileType = File;
      break;

    case FIDKIND_DIR:
      if (code = safe_stat(*fcp, &sbuf)) return code;
      st->FileType = Directory;
      break;

    case FIDKIND_LINK:
      if (code = safe_stat(*fcp, &sbuf)) return code;
      st->FileType = SymbolicLink;
      break;

    case FIDKIND_SPECIAL:
      if (code = safe_stat(*fcp, &sbuf)) return code;
      st->FileType = Invalid;
      break;

    case FIDKIND_MOUNT:
      /* Fake up something appropriate */
      if (!(vmc_ent = vmc_byvolid((*fcp)->mount))) return ENOENT;
      st->FileType = SymbolicLink;
      sbuf.st_size  = strlen(vmc_ent->name) + 2;
      sbuf.st_nlink = 1;
      sbuf.st_ctime = 0;
      sbuf.st_mtime = 0;
      sbuf.st_uid   = 0;
      sbuf.st_gid   = 0;
      sbuf.st_mode  = 0666; /* mount points must not be executable! */
      break;

    default:
      return EPERM;
  }

  st->InterfaceVersion = 1;
  st->LinkCount        = sbuf.st_nlink;
  st->Length           = sbuf.st_size;
  st->DataVersion      = sbuf.st_mtime;
  st->Author           = sbuf.st_uid;
  st->Owner            = sbuf.st_uid;
  st->UnixModeBits     = sbuf.st_mode;
  st->ClientModTime    = sbuf.st_ctime;
  st->ServerModTime    = sbuf.st_ctime;
  st->Group            = sbuf.st_gid;
  if (st->DataVersion < (*fcp)->min_dv)
    st->DataVersion = (*fcp)->min_dv;

  if ((*fcp)->parent) {
    st->ParentVnode  = (*fcp)->parent->Vnode;
    st->ParentUnique = (*fcp)->parent->Unique;
    ci_munge_vnode(chost, &st->ParentVnode, &st->ParentUnique);
  } else if ((*fcp)->Vnode == 1) {
    st->ParentVnode  = 1;
    st->ParentUnique = 1;
  }
  return 0;
}


/** Read data from a file **/
afs_int32 
SRXAFS_FetchData(struct rx_call *call, AFSFid *fid, afs_int32 offset, 
		 afs_int32 length, AFSFetchStatus *st, AFSCallBack *cb, 
		 AFSVolSync *vsync)
{
  AFSFetchStatus myst;
  char volidbuf[20];
  afs_uint32 code;
  VMC *vmc_ent;
  CI *client;
  CH *chost;
  FC *fc;
  char *x;
  int fd;

  debug(DEBUG_RPC, "%I => AFS_FetchData(%u:%u:%u, offset %u, length %u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique, offset, length);
  memset(st, 0, sizeof(*st));
  memset(cb, 0, sizeof(*cb));
  memset(vsync, 0, sizeof(*vsync));

  client = ci_getinfo(call);
  chost = ci_gethost(call);
  ci_demunge_vnode(chost, fid->Volume, &fid->Vnode, &fid->Unique);
  if ((code = getfinfo(chost, fid, &fc, &myst))
  ||  (code = access_rights(fc, &myst, client))
  ||  (code = access_ok(fc, &myst, client, AMODE_READ)))
    return code;

  switch (fc->kind) {
    case FIDKIND_FILE:
      if (code = safe_open(fc, O_RDONLY, &fd)) return code;
      if (lseek(fd, offset, SEEK_SET) == -1) {
        code = errno;
        close(fd);
        return code;
      }
      code = bulk_xmit_fd(call, fd, length);
      close(fd);
      if (code) return code;
      break;

    case FIDKIND_DIR:
      if (code = dirgen(chost, fc, &fd)) return code;
      if (lseek(fd, offset, SEEK_SET) == -1) {
        code = errno;
        close(fd);
        return code;
      }
      code = bulk_xmit_fd(call, fd, length);
      close(fd);
      if (code) return code;
      break;

    case FIDKIND_LINK:
      if (code = safe_readlink(fc, &x)) return code;
      if (offset > strlen(x)) code = EINVAL;
      if (!code) code = bulk_xmit_str(call, x + offset, length);
      osi_Free(x, strlen(x) + 1);
      if (code) return code;
      break;

    case FIDKIND_MOUNT:
      if (vmc_ent = vmc_byvolid(fc->mount)) {
        x = osi_Alloc(strlen(vmc_ent->name) + 3);
        sprintf(x, "#%s.", vmc_ent->name);
      } else {
        sprintf(volidbuf, "#%d.", fc->mount);
        x = osi_Alloc(strlen(volidbuf) + 1);
        strcpy(x, volidbuf);
      }
      if (offset > strlen(x)) code = EINVAL;
      if (!code) code = bulk_xmit_str(call, x + offset, length);
      osi_Free(x, strlen(x) + 1);
      if (code) return code;
      break;

    case FIDKIND_SPECIAL:
    default:
      return EPERM;
  }

  cb->CallBackVersion = CALLBACK_VERSION;
  cb->ExpirationTime  = FT_ApproxTime() + HAFS_CB_TIME;
  cb->CallBackType    = CB_SHARED;
  memcpy(st, &myst, sizeof(AFSFetchStatus));
  return 0;
}


/** Retrieve the ACL of a file **/
afs_int32 
SRXAFS_FetchACL(struct rx_call *call, AFSFid *fid,
		AFSOpaque *acl, AFSFetchStatus *st, AFSVolSync *vsync)
{
  AFSFetchStatus myst;
  afs_uint32 code;
  CI *client;
  CH *chost;
  FC *fc;

  debug(DEBUG_RPC, "%I => AFS_FetchACL(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  memset(acl, 0, sizeof(*acl));
  memset(st, 0, sizeof(*st));
  memset(vsync, 0, sizeof(*vsync));

  client = ci_getinfo(call);
  chost = ci_gethost(call);
  ci_demunge_vnode(chost, fid->Volume, &fid->Vnode, &fid->Unique);
  if ((code = getfinfo(chost, fid, &fc, &myst))
  ||  (code = access_rights(fc, &myst, client))
  ||  (code = access_ok(fc, &myst, client, AMODE_STAT))
  ||  (code = access_acl(fc, &myst, acl)))
    return code;

  memcpy(st, &myst, sizeof(AFSFetchStatus));
  return 0;
}


/** Retrieve the status of a file **/
afs_int32 
SRXAFS_FetchStatus(struct rx_call *call, AFSFid *fid,
		   AFSFetchStatus *st, AFSCallBack *cb, AFSVolSync *vsync)
{
  AFSFetchStatus myst;
  afs_uint32 code;
  CI *client;
  CH *chost;
  FC *fc;

  debug(DEBUG_RPC, "%I => AFS_FetchStatus(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  memset(st, 0, sizeof(*st));
  memset(cb, 0, sizeof(*cb));
  memset(vsync, 0, sizeof(*vsync));

  client = ci_getinfo(call);
  chost = ci_gethost(call);
  ci_demunge_vnode(chost, fid->Volume, &fid->Vnode, &fid->Unique);
  if ((code = getfinfo(chost, fid, &fc, &myst))
  ||  (code = access_rights(fc, &myst, client))
  ||  (code = access_ok(fc, &myst, client, AMODE_STAT)))
    return code;

  cb->CallBackVersion = CALLBACK_VERSION;
  cb->ExpirationTime  = FT_ApproxTime() + HAFS_CB_TIME;
  cb->CallBackType    = CB_SHARED;
  memcpy(st, &myst, sizeof(AFSFetchStatus));
  return 0;
}


/** Write data to a file (not implemented) **/
afs_int32 
SRXAFS_StoreData(struct rx_call *call, AFSFid *fid, AFSStoreStatus *sst,
		 afs_uint32 offset, afs_uint32 length, afs_uint32 file_length,
		 AFSFetchStatus *st, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_StoreData(%u:%u:%u, offset = %u, length = %u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique, offset, length);
  return RXGEN_OPCODE;
}


/** Modify the ACL of a file (not implemented) **/
afs_int32 
SRXAFS_StoreACL(struct rx_call *call, AFSFid *fid, AFSOpaque *sacl, 
		AFSFetchStatus *st, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_StoreACL(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  return RXGEN_OPCODE;
}


/** Modify the status of a file (not implmenented) **/
afs_int32 
SRXAFS_StoreStatus(struct rx_call *call, AFSFid *fid, AFSStoreStatus *sst,
		   AFSFetchStatus *st, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_StoreStatus(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  return RXGEN_OPCODE;
}


/** Remove a non-directory file (not implemented) **/
afs_int32 
SRXAFS_RemoveFile(struct rx_call *call, AFSFid *dfid, char *name,
                       AFSFetchStatus *dst, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_RemoveFile(%u:%u:%u/%s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name);
  return RXGEN_OPCODE;
}


/** Create a regular file (not implemented) **/
afs_int32 
SRXAFS_CreateFile(struct rx_call *call, AFSFid *dfid,
                       char *name, AFSStoreStatus *sst,
                       AFSFid *newfid, AFSFetchStatus *st,
                       AFSFetchStatus *dst, AFSCallBack *cb, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_CreateFile(%u:%u:%u/%s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name);
  return RXGEN_OPCODE;
}


/** Change the name of a file (not implemented) **/
afs_int32 
SRXAFS_Rename(struct rx_call *call, AFSFid *old_dfid, char *old_name,
	      AFSFid *new_dfid, char *new_name,
	      AFSFetchStatus *ost, AFSFetchStatus *nst, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_Rename(%u:%u:%u/%s, %u:%u:%u/%s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        old_dfid->Volume, old_dfid->Vnode, old_dfid->Unique, old_name,
        new_dfid->Volume, new_dfid->Vnode, new_dfid->Unique, new_name);
  return RXGEN_OPCODE;
}


/** Create a symbolic link (not implemented) **/
afs_int32 
SRXAFS_Symlink(struct rx_call *call, AFSFid *dfid, char *name,
	       char *contents, AFSStoreStatus *sst,
	       AFSFid *newfid, AFSFetchStatus *st,
	       AFSFetchStatus *dst, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_Symlink(%u:%u:%u/%s, [%s])",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name, contents);
  return RXGEN_OPCODE;
}


/** DFS translator version of Symlink (not implemented) **/
afs_int32 
SRXAFS_DFSSymlink(struct rx_call *call, AFSFid *dfid, char *name,
		  char *contents, AFSStoreStatus *sst,
		  AFSFid *newfid, AFSFetchStatus *st,
		  AFSFetchStatus *dst, AFSCallBack *cb, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_Symlink(%u:%u:%u/%s, [%s])",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name, contents);
  return RXGEN_OPCODE;
}


/** Create a hard link to a file (not implemented) **/
afs_int32 
SRXAFS_Link(struct rx_call *call, AFSFid *dfid, char *name, AFSFid *fid,
	    AFSFetchStatus *st, AFSFetchStatus *dst, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_Link(%u:%u:%u/%s, %u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name,
        fid->Volume, fid->Vnode, fid->Unique);
  return RXGEN_OPCODE;
}


/** Create a directory (not implemented) **/
afs_int32 
SRXAFS_MakeDir(struct rx_call *call, AFSFid *dfid,
	       char *name, AFSStoreStatus *sst,
	       AFSFid *newfid, AFSFetchStatus *st,
	       AFSFetchStatus *dst, AFSCallBack *cb, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_MakeDir(%u:%u:%u/%s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name);
  return RXGEN_OPCODE;
}


/** Remove a directory (not implemented) **/
afs_int32 
SRXAFS_RemoveDir(struct rx_call *call, AFSFid *dfid, char *name,
		 AFSFetchStatus *dst, AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_RemoveDir(%u:%u:%u/%s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name);
  return RXGEN_OPCODE;
}


/** Notification that a client is giving up all callbacks from this server **/
afs_int32 
SRXAFS_GiveUpCallBacks(struct rx_call *call, AFSCBFids *fids, AFSCBs *cbs)
{
  debug(DEBUG_RPC, "%I => AFS_GiveUpCallBacks()",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))));
  return 0;
}


/** Get the current time (and see if the fileserver is up) **/
afs_int32 
SRXAFS_GetTime(struct rx_call *call, afs_uint32 *secs, afs_uint32 *usecs)
{
  debug(DEBUG_RPC, "%I => AFS_GetTime()",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))));
  *secs = FT_ApproxTime();
  *usecs = 0;
  return 0;
}


/** Get status for multiple FID's at once **/
afs_int32 
SRXAFS_BulkStatus(struct rx_call *call, AFSCBFids* fids,
                       AFSBulkStats *stats, AFSCBs *cbs, AFSVolSync *vsync)
{
  AFSFetchStatus myst;
  afs_uint32 code;
  int i, count;
  CI *client;
  CH *chost;
  FC *fc;

  count = fids->AFSCBFids_len;
  debug(DEBUG_RPC, "%I => AFS_BulkStatus(%d entries)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))), count);
  memset(vsync, 0, sizeof(*vsync));

  memset(stats, 0, sizeof(*stats));
  stats->AFSBulkStats_len = count;
  stats->AFSBulkStats_val = (AFSFetchStatus *)osi_Alloc(count * sizeof(AFSFetchStatus));
  memset(stats->AFSBulkStats_val, 0, count * sizeof(AFSFetchStatus));

  memset(cbs, 0, sizeof(*cbs));
  cbs->AFSCBs_len = count;
  cbs->AFSCBs_val = (AFSCallBack *)osi_Alloc(count * sizeof(AFSCallBack));
  memset(cbs->AFSCBs_val, 0, count * sizeof(AFSCallBack));

  client = ci_getinfo(call);
  chost = ci_gethost(call);

  for (i = 0; i < count; i++) {
    ci_demunge_vnode(chost, fids->AFSCBFids_val[i].Volume,
                     &fids->AFSCBFids_val[i].Vnode,
                     &fids->AFSCBFids_val[i].Unique);
    if ((code = getfinfo(chost, fids->AFSCBFids_val + i, &fc, &myst))
    ||  (code = access_rights(fc, &myst, client))
    ||  (code = access_ok(fc, &myst, client, AMODE_STAT)))
      return code;

    cbs->AFSCBs_val[i].CallBackVersion = CALLBACK_VERSION;
    cbs->AFSCBs_val[i].ExpirationTime  = FT_ApproxTime() + HAFS_CB_TIME;
    cbs->AFSCBs_val[i].CallBackType    = CB_SHARED;
    memcpy(stats->AFSBulkStats_val + i, &myst, sizeof(AFSFetchStatus));
  }
  return 0;
}


/** Request a new advisory lock (not implemented) **/
afs_int32 
SRXAFS_SetLock(struct rx_call *call, AFSFid *fid, ViceLockType locktype,
	       AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_SetLock(%u:%u:%u, %d)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique, locktype);
  return RXGEN_OPCODE;
}


/** Request an extension on an existing lock (not implemented) **/
afs_int32 
SRXAFS_ExtendLock(struct rx_call *call, AFSFid *fid,
		  AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_ExtendLock(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  return RXGEN_OPCODE;
}


/** Release an existing lock (not implemented) **/
afs_int32 
SRXAFS_ReleaseLock(struct rx_call *call, AFSFid *fid,
		   AFSVolSync *vsync)
{
  debug(DEBUG_RPC, "%I => AFS_ReleaseLock(%u:%u:%u)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        fid->Volume, fid->Vnode, fid->Unique);
  return RXGEN_OPCODE;
}


/** Look up a directory entry **/
afs_int32 
SRXAFS_Lookup(struct rx_call *call, AFSFid *dfid, char *name,
	      AFSFid *fid, AFSFetchStatus *st,
	      AFSFetchStatus *dst, AFSCallBack *cb, AFSVolSync *vsync)
{
#ifdef USE_DFS_SEMANTICS
  AFSFetchStatus mydst, myst;
  afs_uint32 code;
  FC *fc, *dfc;
  AFSFid myfid;
  CI *client;
  CH *chost;
  int dfd;

  debug(DEBUG_RPC, "%I => AFS_Lookup(%u:%u:%u, %s)",
        rx_HostOf(rx_PeerOf(rx_ConnectionOf(call))),
        dfid->Volume, dfid->Vnode, dfid->Unique, name);
  memset(fid,   0, sizeof(*fid));
  memset(st,    0, sizeof(*st));
  memset(dst,   0, sizeof(*dst));
  memset(cb,    0, sizeof(*cb));
  memset(vsync, 0, sizeof(*vsync));

  client = ci_getinfo(call);
  chost = ci_gethost(call);
  ci_demunge_vnode(chost, dfid->Volume, &dfid->Vnode, &dfid->Unique);

  if (dfid->Vnode) {
    if ((code = getfinfo(chost, dfid, &dfc, &mydst))
    ||  (code = access_rights(dfc, &mydst, client))
    ||  (code = access_ok(dfc, &mydst, client, AMODE_LOOKUP)))
    return code;

    if (code = safe_opendir(dfc, 0, &dfd)) return code;
    code = gen_fid(dfc, dfd, name, &myfid.Vnode, &myfid.Unique);
    close(dfd);
    if (code) return code;
  } else {
    myfid.Vnode = myfid.Unique = 1;
  }

  myfid.Volume = dfid->Volume;
  if ((code = getfinfo(chost, &myfid, &fc, &myst))
  ||  (code = access_rights(fc, &myst, client))
  ||  (code = access_ok(fc, &myst, client, AMODE_STAT)))
    return code;

  ci_munge_vnode(chost, &myfid.Vnode, &myfid.Unique);

  cb->CallBackVersion = CALLBACK_VERSION;
  cb->ExpirationTime  = FT_ApproxTime() + HAFS_CB_TIME;
  cb->CallBackType    = CB_SHARED;
  memcpy(fid, &myfid, sizeof(AFSFid));
  memcpy(st,  &myst,  sizeof(AFSFetchStatus));
  memcpy(dst, &mydst, sizeof(AFSFetchStatus));
  return 0;

#else /* USE_DFS_SEMANTICS */
  memset(fid,   0, sizeof(*fid));
  memset(st,    0, sizeof(*st));
  memset(dst,   0, sizeof(*dst));
  memset(cb,    0, sizeof(*cb));
  memset(vsync, 0, sizeof(*vsync));
  return RXGEN_OPCODE;
#endif
}

afs_int32 SRXAFS_OldSetLock(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ AFSFid * Fid,
			    /*IN */ ViceLockType Type,
			    /*IN */ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_OldExtendLock(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ AFSFid * Fid,
			       /*IN */ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_OldReleaseLock(
				/*IN */ struct rx_call *z_call,
				/*IN */ AFSFid * Fid,
				/*IN */ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetStatistics(
			       /*IN */ struct rx_call *z_call,
			       /*OUT*/ ViceStatistics * Statistics)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetVolumeInfo(
			       /*IN */ struct rx_call *z_call,
			       /*IN */ char * VolumeName,
			       /*OUT*/ VolumeInfo * Volumeinfo)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetVolumeStatus(
				 /*IN */ struct rx_call *z_call,
				 /*IN */ afs_int32 Volumeid,
				 /*OUT*/ AFSFetchVolumeStatus * Volumestatus,
				 /*OUT*/ char * *Name,
				 /*OUT*/ char * *OfflineMsg,
				 /*OUT*/ char * *Motd)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_SetVolumeStatus(
				 /*IN */ struct rx_call *z_call,
				 /*IN */ afs_int32 Volumeid,
				 /*IN */ AFSStoreVolumeStatus * Volumestatus,
				 /*IN */ char * Name,
				 /*IN */ char * OfflineMsg,
				 /*IN */ char * Motd)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetRootVolume(
			       /*IN */ struct rx_call *z_call,
			       /*OUT*/ char * *VolumeName)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_CheckToken(
			    /*IN */ struct rx_call *z_call,
			    /*IN */ afs_int32 ViceId,
			    /*IN */ AFSOpaque * token)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_NGetVolumeInfo(
				/*IN */ struct rx_call *z_call,
				/*IN */ char * VolumeName,
				/*OUT*/ AFSVolumeInfo * stuff)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_XStatsVersion(
			       /*IN */ struct rx_call *z_call,
			       /*OUT*/ afs_int32 * versionNumberP)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetXStats(
			   /*IN */ struct rx_call *z_call,
			   /*IN */ afs_int32 clientVersionNumber,
			   /*IN */ afs_int32 collectionNumber,
			   /*OUT*/ afs_int32 * srvVersionNumberP,
			   /*OUT*/ afs_int32 * timeP,
			   /*OUT*/ AFS_CollData * dataP)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_FlushCPS(
			  /*IN */ struct rx_call *z_call,
			  /*IN */ ViceIds * IdsArray,
			  /*IN */ IPAddrs * AddrsArray,
			  /*IN */ afs_int32 spare1,
			  /*OUT*/ afs_int32 * spare2,
			  /*OUT*/ afs_int32 * spare3)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_ResidencyCmd(
			      /*IN */ struct rx_call *z_call,
			      /*IN */ AFSFid * Fid,
			      /*IN */ struct ResidencyCmdInputs * Inputs,
			      /*OUT*/ struct ResidencyCmdOutputs * Outputs)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_InlineBulkStatus(
				  /*IN */ struct rx_call *z_call,
				  /*IN */ AFSCBFids * FidsArray,
				  /*OUT*/ AFSBulkStats * StatArray,
				  /*OUT*/ AFSCBs * CBArray,
				  /*OUT*/ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_FetchData64(
			     /*IN */ struct rx_call *z_call,
			     /*IN */ AFSFid * Fid,
			     /*IN */ afs_int64 Pos,
			     /*IN */ afs_int64 Length,
			     /*OUT*/ AFSFetchStatus * OutStatus,
			     /*OUT*/ AFSCallBack * CallBack,
			     /*OUT*/ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_StoreData64(
			     /*IN */ struct rx_call *z_call,
			     /*IN */ AFSFid * Fid,
			     /*IN */ AFSStoreStatus * InStatus,
			     /*IN */ afs_uint64 Pos,
			     /*IN */ afs_uint64 Length,
			     /*IN */ afs_uint64 FileLength,
			     /*OUT*/ AFSFetchStatus * OutStatus,
			     /*OUT*/ AFSVolSync * Sync)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GiveUpAllCallBacks(
				    /*IN */ struct rx_call *z_call)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_GetCapabilities(
				 /*IN */ struct rx_call *z_call,
				 /*OUT*/ Capabilities * capabilities)
{
  return RXGEN_OPCODE;
}

afs_int32 SRXAFS_CallBackRxConnAddr(
				    /*IN */ struct rx_call *z_call,
				    /*IN */ afs_int32 * addr)
{
  return RXGEN_OPCODE;
}

