/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/fidcache.c - FID cache management */

#include "includes.h"
#include "hostafsd.h"


/* Hash parameters and function */
#define FC_NHASH    1024
#define FC_VUMASK   ((2 << FC_VUBITS)  - 1)
#define FC_VOLMASK  ((2 << FC_VOLBITS) - 1)
#define FC_SIZE     (2 << (FC_VUBITS + FC_VOLBITS))
#define FC_HASH(vol,vn,u) (((vol & FC_VOLMASK) << FC_VUBITS) \
                        | ((vn^u) & FC_VUMASK))


static FC *FidCache[FC_SIZE];
static FC *FC_bypath[FC_NHASH];
static FC *FC_mounts;


/* Lookup a FID cache entry by FID */
FC *fc_byfid(AFSFid *fid)
{
  FC *ent;
  int i;

  debug(DEBUG_FC_LOOKUP, "fc_byfid(vol %u, vnode %u, uniq %u)",
        fid->Volume, fid->Vnode, fid->Unique);
  i = FC_HASH(fid->Volume, fid->Vnode, fid->Unique);
  for (ent = FidCache[i]; ent; ent = ent->next) {
    if (fid->Volume != ent->Volume) continue;
    if (fid->Vnode  != ent->Vnode ) continue;
    if (fid->Unique != ent->Unique) continue;
    ent->lastuse = FT_ApproxTime();
    return ent;
  }
  debug(DEBUG_FC_LOOKUP, "fc_byfid: not found!");
  return 0;
}


/* Lookup a FID cache entry by path */
FC *fc_bypath(char *path)
{
  FC *ent;
  int i;

  debug(DEBUG_FC_LOOKUP, "fc_bypath(%s)", path);
  i = namehash(path, FC_NHASH, 0);
  for (ent = FC_bypath[i]; ent; ent = ent->nextpath) {
    if (strcmp(ent->fullpath, path)) continue;
    ent->lastuse = FT_ApproxTime();
    return ent;
  }
  debug(DEBUG_FC_LOOKUP, "fc_bypath: not found!");
  return 0;
}


/* Lookup a FID cache entry for a mount point */
FC *fc_findmount(afs_uint32 parent, afs_uint32 volume)
{
  FC *ent;

  debug(DEBUG_FC_LOOKUP, "fc_findmount(parent = %u, vol = %u)",
        parent, volume);
  for (ent = FC_mounts; ent; ent = ent->nextpath) {
    if (ent->Volume != parent) continue;
    if (ent->mount != volume) continue;
    ent->lastuse = FT_ApproxTime();
    return ent;
  }
  debug(DEBUG_FC_LOOKUP, "fc_findmount: not found!");
  return 0;
}


/* Add a FID cache entry */
void fc_add(FC *ent)
{
  int i;

  debug(DEBUG_FC_UPDATE, "fc_add(vol %u, vnode %u, uniq %u)",
        ent->Volume, ent->Vnode, ent->Unique);
  ent->vmc = vmc_byvolid(ent->Volume);
  debug(DEBUG_FC_UPDATE, "fc_add: path = [%s]", ent->path);

  /* Add the new entry to the hash table */
  i = FC_HASH(ent->Volume, ent->Vnode, ent->Unique);
  if (FidCache[i]) FidCache[i]->prev = ent;
  ent->next = FidCache[i];
  ent->prev = 0;
  FidCache[i] = ent;

  /* Add the new entry to the by-name hash */
  if (ent->kind == FIDKIND_MOUNT) {
    ent->nextpath = FC_mounts;
    ent->prevpath = 0;
    if (FC_mounts) FC_mounts->prevpath = ent;
    FC_mounts = ent;
  } else {
    ent->fullpath = genpath(ent->vmc->path, ent->path);
    debug(DEBUG_FC_UPDATE, "fc_add: fullpath = [%s]", ent->fullpath);
    i = namehash(ent->fullpath, FC_NHASH, 0);
    ent->nextpath = FC_bypath[i];
    ent->prevpath = 0;
    if (FC_bypath[i]) FC_bypath[i]->prevpath = ent;
    FC_bypath[i] = ent;
  }

  /* If there is a parent, update its list of kids */
  ent->kids = 0;
  if (ent->parent) {
    if (ent->parent->kids) ent->parent->kids->prevsib = ent;
    ent->nextsib = ent->parent->kids;
    ent->prevsib = 0;
    ent->parent->kids = ent;
  } else {
    ent->nextsib = 0;
    ent->prevsib = 0;
  }

  /* Update last use */
  ent->lastuse = FT_ApproxTime();
}


/* Remove a FID cache entry - dangerous! */
void fc_remove(FC *ent)
{
  FC *kid;
  int i;

  debug(DEBUG_FC_UPDATE, "fc_remove(vol %u, vnode %u, uniq %u)",
        ent->Volume, ent->Vnode, ent->Unique);

  /* Remove from the hash table */
  i = FC_HASH(ent->Volume, ent->Vnode, ent->Unique);
  if (ent->next) ent->next->prev = ent->prev;
  if (ent->prev) ent->prev->next = ent->next;
  else FidCache[i] = ent->next;

  if (ent->nextpath) ent->nextpath->prevpath = ent->prevpath;
  if (ent->prevpath) ent->prevpath->nextpath = ent->nextpath;
  else if (ent->kind == FIDKIND_MOUNT) {
    FC_mounts = ent->nextpath;
  } else {
    i = namehash(ent->fullpath, FC_NHASH, 0);
    FC_bypath[i] = ent->nextpath;
  }

  /* Remove from the sibling list, if any */
  if (ent->nextsib) ent->nextsib->prevsib = ent->prevsib;
  if (ent->prevsib) ent->prevsib->nextsib = ent->nextsib;
  else if (ent->parent) ent->parent->kids = ent->nextsib;
  if (ent->parent) ent->parent->min_dv = FT_ApproxTime();

  /* Orphan children, if any */
  while (kid = ent->kids) {
    ent->kids = kid->nextsib;
    kid->parent = 0;
    kid->nextsib = 0;
    kid->prevsib = 0;
  }

  /* Free memory */
  osi_Free(ent->path, strlen(ent->path) + 1);
  osi_Free(ent->fullpath, strlen(ent->fullpath) + 1);
  osi_Free(ent, sizeof(FC));
}


/* Invalidate all FID cache entries for a specified volume.
 * If 'remove' is nonzero, remove the entries; otherwise
 * just invalidate any cached paths:
 #
 * FC entries may contain cached paths to objects.  These should
 * probably be updated to reflect a change in mount location.
 * Otherwise, clients attempting to access already-cached FID's
 * in this volume will lose, if the path must be used.  Ideally,
 * we could just invalidate cached paths, and update them lazily.
 */
void fc_inval(afs_uint32 volid, int remove)
{
  FC *ent, *nxtent;
  int i, h, count = 0;

  debug(DEBUG_FC_UPDATE, "fc_inval(vol %u %s)",
        volid, remove ? "remove" : "pathupdate");
  for (i = 0; i < FC_SIZE; i++) {
    for (ent = FidCache[i]; ent; ent = nxtent) {
      nxtent = ent->next;
      if (ent->Volume != volid) continue;
      count++;
      if (remove) fc_remove(ent);
      else if (ent->kind != FIDKIND_MOUNT) {
        debug(DEBUG_FC_UPDATE, "fc_inval: vol %u, vnode %u, uniq %u",
              ent->Volume, ent->Vnode, ent->Unique);
        debug(DEBUG_FC_UPDATE, "fc_inval: path = [%s]", ent->path);
        if (ent->nextpath) ent->nextpath->prevpath = ent->prevpath;
        if (ent->prevpath) ent->prevpath->nextpath = ent->nextpath;
        else {
          h = namehash(ent->fullpath, FC_NHASH, 0);
          FC_bypath[h] = ent->nextpath;
        }
        debug(DEBUG_FC_UPDATE, "fc_inval: old fullpath = [%s]", ent->fullpath);
        osi_Free(ent->fullpath, strlen(ent->fullpath) + 1);

        ent->fullpath = genpath(ent->vmc->path, ent->path);
        debug(DEBUG_FC_UPDATE, "fc_inval: new fullpath = [%s]", ent->fullpath);
        h = namehash(ent->fullpath, FC_NHASH, 0);
        ent->nextpath = FC_bypath[h];
        ent->prevpath = 0;
        if (FC_bypath[h]) FC_bypath[h]->prevpath = ent;
        FC_bypath[h] = ent;
      }
    }
  }
  debug(DEBUG_FC_UPDATE, "fc_inval: processed %d entries", count);
}



/* Test whether a FID cache entry is stale */
/* An entry is considered stale, and thus safe to remove, if
 * the entry, its parent (if present), and any children that
 * are directories have all not been used since the specified time
 * An entry for the root directory of a volume is never stale.
 */
int fc_isstale(FC *ent, afs_uint32 stale)
{
  FC *kid;

  if ((ent->lastuse > stale)
  ||  (ent->parent && ent->parent->lastuse > stale)
  ||  (ent->kind == FIDKIND_DIR && ent->Vnode == 1 && ent->Unique == 1))
    return 0;
  for (kid = ent->kids; kid; kid = kid->next)
    if (kid->kind == FIDKIND_DIR && kid->lastuse > stale) return 0;
  return 1;
}


/* FID cache initialization */
void fc_init(void)
{
  memset(FidCache, 0, sizeof(FidCache));
  memset(FC_bypath, 0, sizeof(FC_bypath));
  FC_mounts = 0;
}


/* FID cache shutdown */
void fc_shutdown(void)
{
  int i;
  FC *ent;

  for (i = 0; i < FC_SIZE; i++) {
    while (ent = FidCache[i]) {
      FidCache[i] = ent->next;
      if (ent->path) osi_Free(ent->path, strlen(ent->path) + 1);
      osi_Free(ent, sizeof(FC));
    }
  }
  memset(FidCache, 0, sizeof(FidCache));
}


/* Do periodic cleanup of the FID cache */
void fc_cleanup(void)
{
  afs_uint32 stale;
  int i, keep = 0, nuke = 0;
  FC *ent, *nxtent;

  stale = FT_ApproxTime() - FC_STALE_TIME;
  debug(DEBUG_FC_UPDATE, "fc_cleanup running (stale = %u)", stale);
  for (i = 0; i < FC_SIZE; i++) {
    for (ent = FidCache[i]; ent; ent = nxtent) {
      nxtent = ent->next;
      if (fc_isstale(ent, stale)) {
        nuke++;
        fc_remove(ent);
      } else keep++;
    }
  }
  debug(DEBUG_FC_UPDATE, "fc_cleanup complete (%d kept, %d removed)",
        keep, nuke);
}
