/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/ds_raw.c - Directory spooge from scratch */


#include "includes.h"
#include "hostafsd.h"
#include "afsdir.h"


struct dir_state {
  struct afsdir_dh dh;       /* Directory header */
  struct afsdir_page page;   /* Current page */
  CH *chost;                 /* Client (for vnode munging) */
  int pageno;                /* Current page # */
  int entno;                 /* Current (next avail) entry # */
  int used;                  /* # entries used in this page */
  int fd;                    /* file descriptor of new directory */
};


static void init_ph(struct afsdir_ph *ph)
{
  memset(ph, 0, sizeof(struct afsdir_ph));
  ph->magic = htons(AFSDIR_MAGIC);
  afsdir_alloc(ph->bitmap, 0);
}


static afs_uint32 readpage(DS *ds)
{
  afs_uint32 code;

  debug(DEBUG_DIRGEN, "[ds_raw]readpage(page %d)", ds->pageno);
  if (lseek(ds->fd, ds->pageno * AFSDIR_PAGESIZE, SEEK_SET) == -1) {
    code = errno;
    debug(DEBUG_DIRGEN, "[ds_raw]readpage: lseek failed (%d)", errno);
    return code;
  }
  if (read(ds->fd, &ds->page, AFSDIR_PAGESIZE) != AFSDIR_PAGESIZE) {
    code = errno;
    debug(DEBUG_DIRGEN, "[ds_raw]readpage: read failed (%d)", errno);
    return code;
  }
  return 0;
}


static afs_uint32 writepage(DS *ds)
{
  afs_uint32 code;

  debug(DEBUG_DIRGEN, "[ds_raw]writepage(page %d)", ds->pageno);
  if (lseek(ds->fd, ds->pageno * AFSDIR_PAGESIZE, SEEK_SET) == -1) {
    code = errno;
    debug(DEBUG_DIRGEN, "[ds_raw]writepage: lseek failed (%d)", errno);
    return code;
  }
  if (write(ds->fd, &ds->page, AFSDIR_PAGESIZE) != AFSDIR_PAGESIZE) {
    code = errno;
    debug(DEBUG_DIRGEN, "[ds_raw]writepage: write failed (%d)", errno);
    return code;
  }
  return 0;
}


afs_uint32 ds_init(DS **dsp, CH *chost, FC *ent, int dfd, int fd)
{
  afs_uint32 vnode, uniq, code;
  int i;

  *dsp = (DS *)osi_Alloc(sizeof(DS));
  memset(*dsp, 0, sizeof(DS));
  (*dsp)->page.ph.magic = htons(AFSDIR_MAGIC);
  for (i = 0; i < AFSDIR_DHE; i++)
    afsdir_alloc((*dsp)->page.ph.bitmap, i);
  (*dsp)->chost = chost;
  (*dsp)->fd = fd;
  (*dsp)->pageno = 0;
  (*dsp)->entno  = AFSDIR_DHE;
  (*dsp)->used   = AFSDIR_DHE;
  if ((code = ds_add(*dsp, ".", ent->Vnode, ent->Unique))) {
    ds_free(*dsp);
    return code;
  }
  if ((code = gen_direntry(ent, dfd, "..", *dsp))) {
    ds_free(*dsp);
    return code;
  }
  return 0;
}


afs_uint32 ds_add(DS *ds, char *name, afs_uint32 vnode, afs_uint32 unique)
{
  afs_uint32 code;
  int l = strlen(name) + 1;
  int ne = l > 16 ? 1 + ((l - 16) / 32) + !!((l - 16) % 32) : 1;
  int hash = namehash(name, AFSDIR_NHASH, 0);

  debug(DEBUG_DIRGEN, "[ds_raw]ds_add: %s => %u/%u", name, vnode, unique);
  ci_munge_vnode(ds->chost, &vnode, &unique);
  if (ne > AFSDIR_EPP - 1) return ENAMETOOLONG;
  if (ds->entno + ne > AFSDIR_EPP) {
    if ((code = writepage(ds))) return code;
    if (ds->pageno < 128) ds->dh.allomap[ds->pageno] = ds->used;

    memset(&ds->page, 0, sizeof(struct afsdir_page));
    ds->page.ph.magic = htons(AFSDIR_MAGIC);
    afsdir_alloc(ds->page.ph.bitmap, 0);
    ds->pageno++;
    ds->entno = 1;
    ds->used = 1;
  }
  ds->page.ent[ds->entno-1].flag   = AFSDIR_FFIRST;
  ds->page.ent[ds->entno-1].next   = ds->dh.hash[hash];
  ds->page.ent[ds->entno-1].vnode  = htonl(vnode);
  ds->page.ent[ds->entno-1].unique = htonl(unique);
  strcpy(ds->page.ent[ds->entno-1].name, name);
  ds->dh.hash[hash] = htons((ds->pageno * AFSDIR_EPP) + ds->entno);
  while (ne--) {
    afsdir_alloc(ds->page.ph.bitmap, ds->entno);
    ds->used++;
    ds->entno++;
  }
  return 0;
}


afs_uint32 ds_finalize(DS *ds)
{
  afs_uint32 code;
  int pages = ds->pageno + 1;
  
  debug(DEBUG_DIRGEN, "[ds_raw]ds_finalize()");
  if (ds->pageno < 128) ds->dh.allomap[ds->pageno] = ds->used;
  if (ds->pageno) {
    if (code = writepage(ds)) return code;
    ds->pageno = 0;
    if (code = readpage(ds)) return code;
  }
  memcpy(&ds->dh.ph, &ds->page.ph, sizeof(struct afsdir_ph));
  ds->dh.ph.pgcount = htons(pages);
  memcpy(&ds->page, &ds->dh, sizeof(struct afsdir_dh));
  if (code = writepage(ds)) return code;

  osi_Free(ds, sizeof(DS));
  return 0;
}


void ds_free(DS *ds)
{
  osi_Free(ds, sizeof(DS));
}
