/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* hostafsd/dirgen.c - Directory transformation */


#include "includes.h"
#include "hostafsd.h"


afs_uint32 gen_fid(FC *dent, int dfd, char *name, afs_uint32 *vnode, afs_uint32 *uniq)
{
  struct stat sbuf;
  char *path, *fullpath, *x;
  afs_uint32 code, is_dir, mount = 0;
  int fid[3], is_dot_dot = 0;
  FC *ent;

  debug(DEBUG_DIRGEN, "gen_fid(%d, %s)", dfd, name);
  debug(DEBUG_DIRGEN, "gen_fid: parent path is [%s]", dent->path);
  if (code = safe_dirstat(dfd, name, &sbuf)) return code;

  if (!strcmp(name, "..")) {
    debug(DEBUG_DIRGEN, "gen_fid: This is a .. entry");
    is_dot_dot = 1;
    x = strrchr(dent->path, '/');
    path = osi_Alloc(x - dent->path + 1);
    strncpy(path, dent->path, x - dent->path);
    path[x-dent->path] = 0;
  } else {
    path = osi_Alloc(strlen(dent->path) + strlen(name) + 2);
    sprintf(path, "%s/%s", dent->path, name);
  }

  if (is_dot_dot && dent->Vnode == 1) {
    debug(DEBUG_DIRGEN, "gen_fid: .. in a volume root");
    osi_Free(path, strlen(path) + 1);
    *vnode = *uniq = 1;
    return 0;
  }

  /* See if there is already such an entry */
  fullpath = genpath(dent->vmc->path, path);
  debug(DEBUG_DIRGEN, "gen_dirfid: new entry path is [%s]", path);
  debug(DEBUG_DIRGEN, "gen_dirfid: full path is [%s]", fullpath);
  ent = fc_bypath(fullpath);
  osi_Free(fullpath, strlen(fullpath) + 1);

  if (ent) {
    debug(DEBUG_DIRGEN, "gen_dirfid: found entry %u/%u/%u",
          ent->Volume, ent->Vnode, ent->Unique);
    debug(DEBUG_DIRGEN, "gen_dirfid: kind %d, dev 0x%08x, ino %d",
          ent->kind, ent->dev, ent->ino);
    if (ent->kind == FIDKIND_DIR && ent->Vnode == 1 && !is_dot_dot) {
      /* Volume root */
      debug(DEBUG_DIRGEN, "gen_dirfid: This is a volume root");
      mount = ent->Volume;
      if (ent = fc_findmount(dent->Volume, ent->Volume)) {
        debug(DEBUG_DIRGEN, "gen_dirfid: Found existing mount point %u/%u/%u",
              ent->Volume, ent->Vnode, ent->Unique);
        osi_Free(path, strlen(path) + 1);
        ent->lastuse = FT_ApproxTime();
        *vnode = ent->Vnode;
        *uniq  = ent->Unique;
        return 0;
      }

    } else if (ent->dev == sbuf.st_dev && ent->ino == sbuf.st_ino) {
      debug(DEBUG_DIRGEN, "gen_dirfid: This is the right entry");
      /* Correct cached entry */
      osi_Free(path, strlen(path) + 1);
      ent->lastuse = FT_ApproxTime();
      *vnode = ent->Vnode;
      *uniq  = ent->Unique;
      return 0;

    } else {
      /* Incorrect cached entry */
      debug(DEBUG_DIRGEN, "gen_dirfid: This is the wrong entry");
      fc_remove(ent);
    }
  }

  debug(DEBUG_DIRGEN, "gen_dirfid: creating new FC entry");
  ent = (FC *)osi_Alloc(sizeof(FC));
  memset(ent, 0, sizeof(FC));
  if (mount) {
    ent->kind  = FIDKIND_MOUNT;
    ent->mount = mount;
    is_dir = 0;
  } else {
    ent->path = path;
    ent->dev = sbuf.st_dev;
    ent->ino = sbuf.st_ino;

    if (sbuf.st_dev != dent->dev)   ent->kind = FIDKIND_SPECIAL;
    else if (S_ISDIR(sbuf.st_mode)) ent->kind = FIDKIND_DIR;
    else if (S_ISREG(sbuf.st_mode)) ent->kind = FIDKIND_FILE;
    else if (S_ISLNK(sbuf.st_mode)) ent->kind = FIDKIND_LINK;
    else                            ent->kind = FIDKIND_SPECIAL;
    is_dir = (ent->kind == FIDKIND_DIR);
  }

  
  ent->Volume = dent->Volume;
  if (is_dir) ent->Vnode = getrandomid() | 1;
  else        ent->Vnode = getrandomid() & ~1;
  ent->Unique = ++dent->vmc->max_uniq;
  fc_add(ent);
  *vnode = ent->Vnode;
  *uniq  = ent->Unique;
  return 0;
} 


afs_uint32 gen_direntry(FC *dent, int dfd, char *name, DS *ds)
{
  afs_uint32 code, vnode, uniq;

  code = gen_fid(dent, dfd, name, &vnode, &uniq);
  if (code) return code;
  return ds_add(ds, name, vnode, uniq);
}


/* Generate an AFS-style directory */
afs_uint32 dirgen(CH *chost, FC *ent, int *fd)
{
  afs_uint32 code;
  struct dirent *de;
  DIR *D = 0;
  DS *ds = 0;
  int dfd = -1;

  debug(DEBUG_DIRGEN, "dirgen(%u/%u/%u [%s])",
        ent->Volume, ent->Vnode, ent->Unique, ent->fullpath);
  *fd = -1;
  if (code = safe_opendir(ent, &D, &dfd)) goto fail;
  debug(DEBUG_DIRGEN, "dirgen: dfd = %d", dfd);
  if (code = safe_tmpfile(fd)) goto fail;

  if ((code = ds_init(&ds, chost, ent, dfd, *fd))) goto fail;
  while ((de = readdir(D))) {
    if (de->d_name[0] == '.'
    && (!de->d_name[1] || (de->d_name[1] == '.' && !de->d_name[2])))
      continue;
    if ((code = gen_direntry(ent, dfd, de->d_name, ds))) goto fail;
  }

  if ((code = ds_finalize(ds))) goto fail;

  /* Success! */
  debug(DEBUG_DIRGEN, "dirgen: dfd %d complete", dfd);
  closedir(D);
  close(dfd);
  return 0;

fail:
  if (ds) ds_free(ds);
  if (D) closedir(D);
  if (dfd >= 0) close(dfd);
  if (*fd >= 0) close(*fd);
  debug(DEBUG_DIRGEN, "dirgen: dfd %d failed", dfd);
  return code;
}
