/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* client.c - client management */

#include "includes.h"
#include "hostafsd.h"


#define CI_SIZE     32  /* Must be a power of two! */
#define CI_NGROUPS  32  /* Number of groups to load at a time */

CI *ClientInfo[CI_SIZE];
CH *ClientHostInfo[CI_SIZE];


static CI *ci_find(afs_uint32 ipaddr, char *princ)
{
  int hash;
  CI *ent;

  hash = namehash(princ ? princ : "", CI_SIZE, ipaddr);
  for (ent = ClientInfo[hash]; ent; ent = ent->next) {
    if (ent->hostaddr != ipaddr) continue;
    if ((ent->princ && princ && !strcmp(ent->princ, princ))
    ||  (!ent->princ && (!princ || !princ[0]))) {
      ent->lastuse = FT_ApproxTime();
      return ent;
    }
  }
  return 0;
}


static void ci_add(CI *ent)
{
  int hash;

  debug(DEBUG_CI, "ci_add(%I:%s)",
        ent->hostaddr, ent->princ ? ent->princ : "<null>");
  ent->lastuse = FT_ApproxTime();
  hash = namehash(ent->princ ? ent->princ : "", CI_SIZE, ent->hostaddr);
  if (ClientInfo[hash]) ClientInfo[hash]->prev = ent;
  ent->next = ClientInfo[hash];
  ClientInfo[hash] = ent;
}


static void ci_remove(CI *ent)
{
  int hash;

  debug(DEBUG_CI, "ci_remove(%I:%s)",
        ent->hostaddr, ent->princ ? ent->princ : "<null>");
  hash = namehash(ent->princ ? ent->princ : "", CI_SIZE, ent->hostaddr);
  if (ent->princ)   osi_Free(ent->princ, strlen(ent->princ) + 1);
  if (ent->lcluser) osi_Free(ent->lcluser, strlen(ent->lcluser) + 1);
  if (ent->groups)  osi_Free(ent->groups, ent->ngroups * sizeof(gid_t));
  if (ent->next) ent->next->prev = ent->prev;
  if (ent->prev) ent->prev->next = ent->next;
  else ClientInfo[hash] = ent->next;
}


CH *ci_gethost(struct rx_call *call)
{
  struct rx_connection *conn = rx_ConnectionOf(call);
  afs_uint32 ipaddr = rx_HostOf(rx_PeerOf(conn));
  short port = rx_PortOf(rx_PeerOf(conn));
  struct rx_securityClass *secobj;
  int hash;
  CH *ent;

  hash = (ipaddr ^ port) % CI_SIZE;
  for (ent = ClientHostInfo[hash]; ent; ent = ent->next) {
    if (ent->hostaddr != ipaddr) continue;
    if (ent->port != port) return ent;
    return ent;
  }

  ent = (CH *)osi_Alloc(sizeof(CH));
  memset(ent, 0, sizeof(*ent));
  ent->hostaddr = ipaddr;
  ent->port = port;
  ent->next = ClientHostInfo[hash];
  ent->prev = 0;
  ClientHostInfo[hash] = ent;
  des_new_random_key(ent->mungekey);
  des_key_sched(ent->mungekey, ent->mungeks);
  secobj = rxnull_NewClientSecurityObject();
  ent->conn = rx_NewConnection(ipaddr, port, AFSCB_SVCID, secobj, 0);
  RXAFSCB_InitCallBackState(ent->conn);
  return ent;
}


int ci_userok(CI *ent, char *user)
{
  AUTH_DAT adat;

  debug(DEBUG_AUTH, "ci_userok(%I:%s => %s)",
        ent->hostaddr, ent->princ ? ent->princ : "<null>", user);
  if (!ent->princ) {
    debug(DEBUG_AUTH, "ci_userok: no (not authed)");
    return 0;
  }
  memset(&adat, 0, sizeof(adat));
  if (kname_parse(adat.pname, adat.pinst, adat.prealm, ent->princ)) {
    debug(DEBUG_AUTH, "ci_userok: no (kname_parse failed)");
    return 0;
  }
  if (kuserok(&adat, user)) {
    debug(DEBUG_AUTH, "ci_userok: no (kuserok failed)");
    return 0;
  }
  debug(DEBUG_AUTH, "ci_userok: yes");
  return 1;
}


int ci_rootok(CI *ent)
{
  struct passwd *pwent;
  char *name;
  int result;

  debug(DEBUG_AUTH, "ci_rootok(%I:%s)",
        ent->hostaddr, ent->princ ? ent->princ : "<null>");
  if (!ent->princ) {
    debug(DEBUG_AUTH, "ci_rootok: no (not authed)");
    return 0;
  }
  pwent = getpwuid(0);
  if (!pwent) {
    debug(DEBUG_AUTH, "ci_rootok: no (getpwuid failed)");
    return 0;
  }
  name = osi_Alloc(strlen(pwent->pw_name) + 1);
  strcpy(name, pwent->pw_name);
  result = ci_userok(ent, pwent->pw_name);
  osi_Free(name, strlen(name) + 1);
  return result;
}


static void ci_updateauth(CI *ent, char *lcluser)
{
  gid_t gbuf[CI_NGROUPS], *groups = 0, *xgroups;
  struct passwd *pwent;
  struct group *grent;
  int gbn = 0, gn = 0;
  char **x;

  debug(DEBUG_AUTH, "ci_updateauth(%I:%s, %s)",
        ent->hostaddr, ent->princ ? ent->princ : "<null>", lcluser);
  if (lcluser && (pwent = getpwnam(lcluser))) {
    if (ent->lcluser) osi_Free(ent->lcluser, strlen(ent->lcluser) + 1);
    ent->lcluser = osi_Alloc(strlen(lcluser) + 1);
    strcpy(ent->lcluser, lcluser);
    ent->uid = pwent->pw_uid;
    ent->gid = pwent->pw_gid;
    debug(DEBUG_AUTH, "ci_updateauth: uid = %d, gid = %d", ent->uid, ent->gid);

    setgrent();
    while (grent = getgrent()) {
      for (x = grent->gr_mem; *x && strcmp(*x, lcluser); x++);
      if (!x) continue;
      gbuf[gbn++] = grent->gr_gid;
      if (gbn < CI_NGROUPS) continue;
      xgroups = (gid_t *)osi_Alloc((gn + gbn) * sizeof(gid_t));
      if (gn) memcpy(xgroups, groups, gn * sizeof(gid_t));
      memcpy(xgroups + gn, gbuf, gbn * sizeof(gid_t));
      osi_Free(groups, gn * sizeof(gid_t));
      groups = xgroups;
      gn += gbn;
      gbn = 0;
    }
    endgrent();
    if (ent->groups)  osi_Free(ent->groups, ent->ngroups * sizeof(gid_t));
    ent->groups = groups;
    ent->ngroups = gn;
    debug(DEBUG_AUTH, "ci_updateauth: %d groups", gn);
    return;
  } else if (lcluser) {
    debug(DEBUG_AUTH, "ci_updateauth: getpwnam failed!");
    ent->mode = CIMODE_UNAUTH;
  }

  debug(DEBUG_AUTH, "Clearing local user, uid/gid, and groups");
  if (ent->lcluser) osi_Free(ent->lcluser, strlen(ent->lcluser) + 1);
  ent->lcluser = 0;

  ent->uid = 65535;  /* Just in case there is a bug elsewhere, */
  ent->gid = 65535;  /* don't let them use it to become root */

  if (ent->groups) osi_Free(ent->groups, ent->ngroups * sizeof(gid_t));
  ent->groups = 0;
  ent->ngroups = 0;
}


afs_uint32 ci_setauth(CI *ent, int mode, char *lcluser)
{
  char luser[ANAME_SZ], lrealm[REALM_SZ];
  afs_uint32 code;
  AUTH_DAT adat;

  debug(DEBUG_AUTH, "ci_setauth(%I:%s, %d, %s",
        ent->hostaddr, ent->princ ? ent->princ : "<null>",
        mode, lcluser ? lcluser : "<null>");
  switch (mode) {
    case CIMODE_UNKNOWN:
      debug(DEBUG_AUTH, "ci_setauth: using default authorization");
      memset(&adat, 0, sizeof(adat));
      ent->flags &=~ CIFLAG_REQUEST;
      if (!ent->princ) {
        ent->mode = CIMODE_UNAUTH;
        ci_updateauth(ent, 0);

      } else if (!kname_parse(adat.pname, adat.pinst, adat.prealm, ent->princ)
#if 0
             &&  !krb_kntoln(&adat, luser)
#endif
             &&  ci_userok(ent, luser)) {
        ent->mode = CIMODE_AUTH;
        ci_updateauth(ent, luser);

      } else if (!kname_parse(adat.pname, adat.pinst, adat.prealm, ent->princ)
             &&  !krb_get_lrealm(lrealm, 1)
             &&  !strcmp(adat.prealm, lrealm)) {
        ent->mode = CIMODE_LOCAL;
        ci_updateauth(ent, 0);

      } else {
        ent->mode = CIMODE_FOREIGN;
        ci_updateauth(ent, 0);

      }
      return 0;

    case CIMODE_UNAUTH:
      debug(DEBUG_AUTH, "ci_setauth: Unauth mode requested");
      ent->flags |= CIFLAG_REQUEST;
      ent->mode = CIMODE_UNAUTH;
      ci_updateauth(ent, 0);
      return 0;

    case CIMODE_FOREIGN:
      debug(DEBUG_AUTH, "ci_setauth: Foreign mode requested");
      if (!ent->princ) {
        debug(DEBUG_AUTH, "ci_setauth: not authenticated!");
        return HAFS_NOTAUTH;
      }
      ent->flags |= CIFLAG_REQUEST;
      ent->mode = CIMODE_FOREIGN;
      ci_updateauth(ent, 0);
      return 0;

    case CIMODE_LOCAL:
      debug(DEBUG_AUTH, "ci_setauth: Local/Anon mode requested");
      if (!ent->princ) {
        debug(DEBUG_AUTH, "ci_setauth: not authenticated!");
        return HAFS_NOTAUTH;
      }
      if ((code = kname_parse(adat.pname, adat.pinst, adat.prealm, ent->princ))
      ||  (code = krb_get_lrealm(lrealm, 1))) {
        debug(DEBUG_AUTH, "ci_setauth: kname_parse or krb_get_lrealm failed!");
        return kerberize_error(code);
      }
      if (!strcmp(adat.prealm, lrealm)) {
        ent->flags |= CIFLAG_REQUEST;
        ent->mode = CIMODE_LOCAL;
        ci_updateauth(ent, 0);
        return 0;
      }
      if (!lcluser) {
         debug(DEBUG_AUTH, "ci_setauth: not local and no user specified");
         return HAFS_NOTAUTH;
      }
      /* fall through */

    case CIMODE_AUTH:
      if (mode == CIMODE_AUTH)
        debug(DEBUG_AUTH, "ci_setauth: Local/Auth mode requested\n");
      if (!lcluser) {
        debug(DEBUG_AUTH, "ci_setauth: no user specified");
        return HAFS_BADAUTH;
      }
      if (!ent->princ) {
        debug(DEBUG_AUTH, "ci_setauth: not authenticated!");
        return HAFS_NOTAUTH;
      }
      if (!ci_userok(ent, lcluser)) {
        debug(DEBUG_AUTH, "ci_setauth: not authorized!");
        return HAFS_NOTAUTH;
      }
      ent->flags |= CIFLAG_REQUEST;
      ent->mode = mode;
      ci_updateauth(ent, (mode == CIMODE_AUTH) ? lcluser : 0);
      return 0;

    default:
      debug(DEBUG_AUTH, "ci_setauth: unknown mode requested\n");
      return HAFS_BADAUTH;
  }
}


CI *ci_getinfo(struct rx_call *call)
{
  struct rx_connection *conn = rx_ConnectionOf(call);
  afs_uint32 ipaddr = rx_HostOf(rx_PeerOf(conn));
  char name[ANAME_SZ], inst[INST_SZ], realm[REALM_SZ];
  char principal[MAX_K_NAME_SZ], *x;
  int hash;
  CI *ent;
  CH *hent;

  if (rx_SecurityClassOf(conn) == 2
  &&  !rxkad_GetServerInfo(conn, 0, 0, name, inst, realm, 0)) {
    for (x = realm; *x; x++) if (islower(*x)) *x = toupper(*x);
    sprintf(principal, "%s%s%s@%s", name, inst[0] ? "." : "", inst, realm);
  } else {
    principal[0] = 0;
  }
  debug(DEBUG_CI, "ci_getinfo(principal=%s)", principal);
  if (ent = ci_find(ipaddr, principal)) return ent;

  debug(DEBUG_CI, "ci_getinfo: generating new entry");
  ent = (CI *)osi_Alloc(sizeof(CI));
  memset(ent, 0, sizeof(CI));
  ent->hostaddr = ipaddr;
  if (principal[0]) {
    ent->princ = osi_Alloc(strlen(principal) + 1);
    strcpy(ent->princ, principal);
  }
  ci_setauth(ent, 0, 0);
  ci_add(ent);
  return ent;
}


static void domunge(des_cblock block, des_key_schedule ks, int dir)
{
  des_cblock ivec;
  afs_uint32 v;

  memset(ivec, 0, 8);
  do {
    des_cbc_encrypt((des_cblock *)block, (des_cblock *)block, 8, ks, ivec, dir);
    memcpy(&v, block, 4);
  } while (!(v & 0xfffffffe));
}


void ci_munge_vnode(CH *ent, afs_uint32 *vnode, afs_uint32 *uniq)
{
  des_cblock block;
  afs_uint32 v;

  if (!(*vnode & 0xfffffffe)) return;
  memcpy(block, vnode, 4);
  memcpy(block + 4, uniq, 4);
  domunge(block, ent->mungeks, 1);
  memcpy(&v, block, 4);
  memcpy(uniq, block + 4, 4);
  *vnode = (v & 0xfffffffe) | (*vnode & 0x00000001);
}


void ci_demunge_vnode(CH *ent, afs_uint32 vol, afs_uint32 *vnode, afs_uint32 *uniq)
{
  des_cblock block, ivec;
  afs_uint32 v0, u0, v1, u1;
  AFSFid fid;
  int match;

  if (!(*vnode & 0xfffffffe)) return;

  memcpy(block, vnode, 4);
  memcpy(block + 4, uniq, 4);
  domunge(block, ent->mungeks, 0);
  memcpy(&v0, block, 4);
  memcpy(&u0, block + 4, 4);

  *vnode ^= 1;
  memcpy(block, vnode, 4);
  memcpy(block + 4, uniq, 4);
  domunge(block, ent->mungeks, 0);
  memcpy(&v1, block, 4);
  memcpy(&u1, block + 4, 4);

  /* 4 cases:
   * - v0 matches - return v0/u0
   * - v1 matches - return v1/u1
   * - both match - more work
   * - neither matches - return 0/0
  */

  match = (((v0 & 1) ^ (*vnode & 1)) ? 1 : 0)
        | (((v1 & 1) ^ (*vnode & 1)) ? 2 : 0);

  if (match == 3) {
    match = 0;
    fid.Volume = vol; fid.Vnode = v0; fid.Unique = u0;
    if (fc_byfid(&fid)) match |= 1;

    fid.Volume = vol; fid.Vnode = v1; fid.Unique = u1;
    if (fc_byfid(&fid)) match |= 2;
  }

  switch (match) {
    case 0: *vnode = 0;  *uniq = 0;  return;
    case 1: *vnode = v0; *uniq = u0; return;
    case 2: *vnode = v1; *uniq = u1; return;
    case 3: *vnode = 0;  *uniq = 0;  return; /* Oh, well */
  }
}


/* Client info initialization */
void ci_init(void)
{
  memset(ClientInfo, 0, sizeof(ClientInfo));
}


/* Client info shutdown */
void ci_shutdown(void)
{
  int i;
  CI *ent;

  for (i = 0; i < CI_SIZE; i++) {
    while (ent = ClientInfo[i]) {
      ClientInfo[i] = ent->next;
      osi_Free(ent, sizeof(CI));
    }
  }
  memset(ClientInfo, 0, sizeof(ClientInfo));
}


/* Do periodic cleanup of the client info cache */
void ci_clean(void)
{
  afs_uint32 stale;
  int i, keep = 0, nuke = 0;
  CI *ent, *nxtent;

  stale = FT_ApproxTime() - CI_STALE_TIME;
  debug(DEBUG_CI, "ci_clean running (stale = %u)", stale);
  for (i = 0; i < CI_SIZE; i++) {
    for (ent = ClientInfo[i]; ent; ent = nxtent) {
      nxtent = ent->next;
      if ((ent->lastuse > stale)
      ||  ((ent->flags & CIFLAG_REQUEST) && ent->expires > stale)) {
        keep++;
      } else {
        ci_remove(ent);
        nuke++;
      }
    }
  }
  debug(DEBUG_CI, "ci_clean complete (%d kept, %d removed)", keep, nuke);
}
