/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/*
 * hostafsd/afsdir.h - Description of the AFS directory format
 * Information in here was derived largely from documentation
 * available at http://www.coda.cs.cmu.edu/doc/htmp/codadir-1.html
 * and from previous investigation by the author.  Most of the
 * structure definitions here were taken almost directly from the
 * above-referenced documentation.
 */


#define AFSDIR_PAGESIZE    2048
#define AFSDIR_EPP         64
#define AFSDIR_DHE         13
#define AFSDIR_PH_PADSIZE  19
#define AFSDIR_MAXPAGES    128
#define AFSDIR_NHASH       128
#define AFSDIR_MAGIC       1234
#define AFSDIR_FFIRST      1


struct afsdir_ph {
  afs_uint16 pgcount;                   /* Total page count (dh only) */
  afs_uint16 magic;                     /* Must be AFSDIR_MAGIC */
  char freecount;                    /* unused */
  char bitmap[AFSDIR_EPP/8];         /* Map of allocated blobs */
  char padding[AFSDIR_PH_PADSIZE];
};


struct afsdir_dh {
  struct afsdir_ph ph;
  char allomap[AFSDIR_MAXPAGES];     /* Page allocation map */
  afs_uint16 hash[AFSDIR_NHASH];        /* Hash table of entries */
};


struct afsdir_ent {
  char flag;                         /* AFSDIR_FFIRST iff first entry */
  char length;                       /* unused; set to 0 */
  afs_uint16 next;                      /* blob # of next entry in bucket */
  afs_uint32 vnode;                     /* Vnode this refers to */
  afs_uint32 unique;                    /* Uniqifier this refers to */
  char name[16];                     /* Name of this entry */
  char pad[4];                       /* Pad to 32 bytes */
};


struct afsdir_page {
  struct afsdir_ph ph;
  struct afsdir_ent ent[AFSDIR_EPP-1];
};


#define afsdir_alloc(bm,i) (bm[(i)>>3] |= (1 << ((i) & 7)))


/* AFS directory format:
 * A directory consists of one or more 'pages', each of which is 2K
 * (AFS_PAGESIZE).  Each page contains EPP (currently 64) 'entries', each
 * of which is 32 bytes.  The first page begins with a DirHeader, which
 * is DHE entries long, and includes a PageHeader.  All other pages begin
 * with just a PageHeader, which is 1 entry long.  Every other entry is
 * a DirEntry, a DirXEntry (name extension), or unused.
 *
 * A Page Header contains the following elements:
 * - pgcount    contains a count of the number of pages in the directory,
 *              if the directory is new-style (>128 pages), or 0 if it is
 *              old-style.  This field is meaningful only in the Dir Header.
 * - tag        a magic number, which must be 1234
 * - freecount  apparently unused
 * - freebitmap A bitmap of free entries.  Each byte corresponds to 8
 *              entries, with the least significant bit referring to the
 *              first of those.  Each bit is set iff the corresponding
 *              entry is allocated.  Entries used by the page and dir
 *              headers are considered allocated.
 *
 * A Dir Header consists of a Page Header, followed by an allocation map
 * and hash table.  The allocation map contains one byte for each of the
 * first 128 pages; that byte contains the number of entries in that page
 * that are allocated.  Every page that actually exists has at peast one
 * entry allocated (the Page Header); if a byte in this map is 0, it means
 * that the page does not yet exist.
 *
 * Each bucket in the hash table is a linked list, using 'blob numbers'
 * as pointers.  A blob number is defined as (page# * EPP) + entry#.
 * The head of each chain is kept in the hash table, and the next pointers
 * are kept in the 'next' entry of each directory.
 *
 * Directory entries themselves contain the following elements:
 * - flag    Set to FFIRST iff this is the first blob in an entry
 *           (otherwise it will be a name continuation).  This is
 *           probably not reliable.
 * - length  Unused
 * - next    Pointer to the next element in this hash chain
 * - fid     FileID (vnode and uniquifier)
 * - name    Filename (null-terminated)
 */
