/*
 * hostafs - A lightweight AFS server
 *
 * Copyright (c) 1998-1999 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * Carnegie Mellon requests users of this software to return to
 *
 *  Software Distribution Coordinator  or  Software_Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* access.c - Access control */

#include "includes.h"
#include "hostafsd.h"

static afs_uint32 getbits(FC *fc, AFSFetchStatus *st, CI *client,
                       mode_t *bits, int *isowner)
{
  struct stat sbuf;
  afs_uint32 code;
  ACMODE *ac;
  int i;

  debug(DEBUG_ACCESS, "getbits: kind %d, bits 0%03o, owner %d, group %d",
        fc->kind, st->UnixModeBits, st->Owner, st->Group);
  if (client)
    debug(DEBUG_ACCESS, "getbits: mode %d, uid %d, gid %d, principal %s",
          client->mode, client->uid, client->gid, client->princ);
  else
    debug(DEBUG_ACCESS, "getbits: null client");

  /* select an access mode */
  if (isowner) *isowner = 0;
  if (!client) ac = &fc->vmc->access_unauth;
  else switch (client->mode) {
    case CIMODE_UNAUTH:  ac = &fc->vmc->access_unauth;  break;
    case CIMODE_FOREIGN: ac = &fc->vmc->access_foreign; break;
    case CIMODE_LOCAL:   ac = &fc->vmc->access_local;   break;
    case CIMODE_AUTH:    ac = &fc->vmc->access_auth;    break;
    case CIMODE_UNKNOWN:
    default:
      debug(DEBUG_ACCESS, "getbits: access denied");
      return EACCES;
  }
  debug(DEBUG_ACCESS, "getbits: acmode 0x%04x, uid %d, gid %d",
        ac->mode, ac->uid, ac->gid);

  /* Figure out what bits the user is entitled to */
  *bits = 0;
  if (ac->mode & ACMODE_WORLD) *bits |= S_IRWXO;
  if (ac->mode & ACMODE_GROUP) {
    if (client && client->mode == CIMODE_AUTH) {
      if (client->gid == st->Group) *bits |= S_IRWXG;
      if (client->ngroups) {
        for (i = 0; i < client->ngroups; i++)
          if (client->groups[i] == st->Group) *bits |= S_IRWXG;
      }
    } else {
      if (ac->gid == st->Group) *bits |= S_IRWXG;
    }
  }
  if (ac->mode & ACMODE_USER) {
    if (client && client->mode == CIMODE_AUTH) {
      if (client->uid == st->Owner) *bits |= S_IRWXU;
    } else {
      if (ac->uid == st->Owner) *bits |= S_IRWXU;
    }
  }
  if (!(ac->mode & ACMODE_RW)) *bits &= ~0222;
  if (ac->mode & ACMODE_SRCH)  *bits |= 0001;

  /* what permissions does the file grant? */
  switch (fc->kind) {
    case FIDKIND_FILE:
    case FIDKIND_DIR:
    case FIDKIND_LINK:
    case FIDKIND_MOUNT:
      *bits &= st->UnixModeBits;
      break;

    case FIDKIND_SPECIAL:
      *bits = 0;
      break;

    default:
      debug(DEBUG_ACCESS, "getbits: permission denied");
      return EPERM;
  }
  debug(DEBUG_ACCESS, "getbits: resulting bits are 0%03o", *bits);

  if (ac->mode & ACMODE_OWN) {
    if (client && client->mode == CIMODE_AUTH) {
      if (client->uid == st->Owner) *isowner = 1;
    } else {
      if (ac->uid == st->Owner) *isowner = 1;
    }
  }
  return 0;
}


static afs_uint32 compute_rights(int kind, afs_uint32 bits)
{
  afs_uint32 rights;

  if (kind == FIDKIND_DIR) {
#ifdef USE_DFS_SEMANTICS
    if ((bits & (S_IXUSR | S_IXGRP | S_IXOTH))
    ||  (bits & (S_IRUSR | S_IRGRP | S_IROTH)))
      rights |= PRSFS_LOOKUP | PRSFS_READ;
    if (bits & (S_IWUSR | S_IWGRP | S_IWOTH))
      rights |= PRSFS_WRITE | PRSFS_INSERT | PRSFS_DELETE;
#else
    rights = PRSFS_READ | PRSFS_WRITE;
    if (bits & (S_IXUSR | S_IXGRP | S_IXOTH)) rights |= PRSFS_LOOKUP;
    if (bits & (S_IRUSR | S_IRGRP | S_IROTH)) rights |= PRSFS_LOOKUP;
    if (bits & (S_IWUSR | S_IWGRP | S_IWOTH)) rights |= PRSFS_INSERT;
    if (bits & (S_IWUSR | S_IWGRP | S_IWOTH)) rights |= PRSFS_DELETE;
#endif
  } else {
    rights = 0;
    if (bits & (S_IRUSR | S_IRGRP | S_IROTH))
      rights |= PRSFS_READ | PRSFS_LOCK;
    if (bits & (S_IWUSR | S_IWGRP | S_IWOTH)) rights |= PRSFS_WRITE;
  }
  debug(DEBUG_ACCESS, "compute_rights(kind %d, bits 0%03o) => %R",
        kind, bits, rights);
  return rights;
}


afs_uint32 access_rights(FC *fc, AFSFetchStatus *st, CI *client)
{
  mode_t bits;
  afs_uint32 code;
  afs_uint32 rights;
  int isowner;

  if (code = getbits(fc, st, client, &bits, &isowner)) return code;
  st->CallerAccess = compute_rights(fc->kind, bits);
  if (isowner) st->CallerAccess |= PRSFS_ADMINISTER;
  if (code = getbits(fc, st, 0, &bits, 0)) {
    if (code == EACCES || code == EPERM) st->AnonymousAccess = 0;
    else return code;
  } else {
    st->AnonymousAccess = compute_rights(fc->kind, bits);
  }
  return 0;
}


afs_uint32 access_ok(FC *fc, AFSFetchStatus *st, CI *client, int mode)
{
  mode_t bits;
  afs_uint32 code;
  int isowner;

  debug(DEBUG_ACCESS, "access_ok(mode %d)", mode);
  if (code = getbits(fc, st, client, &bits, &isowner)) return code;
  switch (mode) {
    case AMODE_STAT:   return 0;
    case AMODE_READ:   bits &= (S_IRUSR | S_IRGRP | S_IROTH); break;
    case AMODE_WRITE:  bits &= (S_IWUSR | S_IWGRP | S_IWOTH); break;
    case AMODE_LOOKUP:
      bits &= (S_IRUSR | S_IRGRP | S_IROTH
            |  S_IXUSR | S_IXGRP | S_IXOTH);
      break;
    default:          return EPERM;
  }
  if (bits) return 0;
  return EACCES;
}


afs_uint32 access_acl(FC *fc, AFSFetchStatus *st, AFSOpaque *acl)
{
  afs_uint32 urights, grights, wrights;
  char buf[1024], ubuf[20], gbuf[20], *uname, *gname;
  struct passwd *pwent;
  struct group *grent;

  debug(DEBUG_ACCESS, "access_acl()");
  urights = compute_rights(fc->kind, st->UnixModeBits & S_IRWXU);
  grights = compute_rights(fc->kind, st->UnixModeBits & S_IRWXG);
  wrights = compute_rights(fc->kind, st->UnixModeBits & S_IRWXO);
  urights |= PRSFS_ADMINISTER;

  pwent = getpwuid(st->Owner);
  if (pwent) uname = pwent->pw_name;
  else sprintf(uname = ubuf, "uid:%d", st->Owner);

  grent = getgrgid(st->Group);
  if (grent) gname = grent->gr_name;
  else sprintf(gname = gbuf, "gid:%d", st->Group);

  sprintf(buf, "%d\n%d\n%s\t%d\n%s%s\t%d\n%s\t%d\n", 3, 0, uname, urights,
          grent ? "group:" : "", gname, grights, "system:world", wrights);
  acl->AFSOpaque_len = strlen(buf) + 1;
  acl->AFSOpaque_val = osi_Alloc(acl->AFSOpaque_len);
  strcpy(acl->AFSOpaque_val, buf);
  debug(DEBUG_ACCESS, "access_acl: %s", buf);
  return 0;
}
