;;;
;;; Copyright (c) 1992, 1993, 1994 Carnegie Mellon University
;;; All Rights Reserved.
;;;
;;; See COPYRIGHT file for further information.
;;;

(in-package :nesl-lisp)

(defun size-depth-type (symbol list definitions)
  (do* ((depth 0 (+ size depth))
	(rem-list list (cdr rem-list))
	(elt (car rem-list) (car rem-list))
	(size (l-from-type (second elt) definitions)
	      (l-from-type (second elt) definitions)))
       ((or (eql symbol (first elt)) (null (cdr rem-list)))
	(if (eql symbol (first elt))
	    (list size depth (second elt)) nil))))

(defun copy-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (if sdt
	(cons (third sdt) (if (plusp (first sdt))
			      (list (list 'COPY (first sdt) (second sdt)))
			    nil))
      (nesl-error "Internal error: There is no definition for ~s." symbol))))

(defun delete-type (symbol list definitions)
  (declare (ignore definitions))
  (do* ((rem-list list (cdr rem-list)))
       ((or (eql symbol (first (first rem-list))) (null (cdr rem-list)))
	(if (eql symbol (first (first rem-list)))
	    (rplacd (first rem-list) nil)))))

(defvar *cpop* t)

(defun copy-pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt
	(cons (third sdt) (if (and (plusp (first sdt)) (plusp (second sdt)))
			      (if *cpop*
				  (list (list 'CPOP (first sdt) (second sdt)))
				(list (list 'POP (first sdt) 
					    (+ (first sdt) (second sdt)))
				      (list 'COPY (first sdt) (second sdt))))
			    nil))
      (nesl-error "Internal error: There is no definition for ~s." symbol))))

(defun pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt (list 'POP (first sdt) (second sdt))
      (let ((binding (get-binding-definition symbol definitions)))
	(if binding nil
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun get-type-code (body arg-types fun-type definitions)
  (cond ((and (listp body) (eql (first body) 'base-typecase))
	 (let* ((type (foo-type fun-type (second body) arg-types definitions))
		(code (if (and (listp type) (eql (car type) 'function))
			  (or (second (assoc 'function (cddr body)))
			      `(:primitive (POP ,(l-from-type (car arg-types) 
							      definitions)
						0)))
			(second (assoc type (cddr body))))))
	   (when (not code) 
	     (nesl-error "Internal error: Bad type in Base-Typecase."))
	   code))
	((and (listp body) (eql (first body) 'poly-typecase))
	 (let ((type (foo-type fun-type (second body) arg-types definitions)))
	   (cond ((atom type) (third body)) ;; (cons :inline (third body))
		 ((and (listp type) (eql (first type) 'function))
		  (if (listp (second type))
		      (fifth body) 
		    ;;(nesl-error "Closures not fully implemented.")
		    (third body)))
		 ((and (listp type) (eql (first type) 'vector))
		  (fourth body))
		 ((and (listp type) (eql (first type) 'pair))
		  (fifth body))
		 (t (nesl-error "Internal error in poly-typecase")))))
	(t body)))

(defun generate-code (fundef arg-type recursivep definitions)
  (let ((ccode (get-cached-code fundef arg-type definitions) ))
    (or (and (or recursivep (not (eql ccode :stub))) ccode)
	(let ((type-specific-code
	       (get-type-code (code-compiled fundef)
			      (list arg-type)
			      (code-type fundef)
			      definitions)))
	  ;;(print (list (car names) arg-type))
	  ;; This is added so that recursive calls work
	  (when (eql type-specific-code :stub)
	    (nesl-error "No body for function ~a." (code-name fundef)))
	  (add-cached-code fundef arg-type :stub definitions)
	  (add-cached-code 
	   fundef arg-type
	   (cdr (trans-op (code-arguments fundef) 
			  (list arg-type)
			  type-specific-code definitions)) 
	   definitions)))))

(defun get-call-code (fundef arg-type definitions)
  (let ((code (generate-code fundef arg-type t definitions)))
    (if (and (listp code) (eql (car code) :inline))
	(reverse (cdr code))
      ;; BACK CALL FIX (car names)
      (list (list 'CALL (cons fundef arg-type))))))

(defun generate-constant-code (binding definitions)
  (let ((cached-code (get-cached-code binding nil definitions)))
    (or cached-code
	(let* ((compiled-code (code-compiled binding))
	       (fully-compiled-code 
		(cdr (trans-op nil nil compiled-code definitions))))
	  (add-cached-code binding nil fully-compiled-code definitions)))))

;;(defun check-type (type definitions)
;;  (format t "~%HERE: ~a" (print-type type definitions))
;;  type)

(defun trans-global-constant (code definitions)
  (let* ((serial-code (pscode-serial code))
	 (type (code-type serial-code)))
    (if (is-function-type? type)
	(cons (instantiate-fun-type type)
	      (if (listp (second (car type)))
		  (list (list 'CALL (cons serial-code nil)))
		nil))
      (progn
	(generate-constant-code serial-code definitions)
	(cons (car type)
	      (list (list 'CALL (cons serial-code nil))))))))

(defun fun-def (code arg-type parallelp definitions)
  (let ((fundef (if parallelp (pscode-parallel code) (pscode-serial code))))
    (cons (get-instantiated-function-type fundef arg-type definitions)
	  (get-call-code fundef arg-type definitions))))

(defun c-fun-def (ftype arg-type parallelp definitions)
  (when (not (and (listp ftype) (eql (car ftype) 'function)))
    (nesl-error "Internal error: Bad function type ~a" ftype))
  (let ((fcode (second ftype)))
    (if (listp fcode)
	(if parallelp
	    (fun-def (second fcode) 
		     (list 'pair (list 'pair (third fcode) 
					     (second arg-type))
			   (third arg-type))
		     parallelp definitions)
	  (fun-def (second fcode) (list 'pair (third fcode) arg-type)
		   parallelp definitions))
      (fun-def (binding-compiled-code fcode) arg-type parallelp definitions))))

(defun trans-func (list stack definitions)
  (let* ((parallelp (and (listp (car list)) (eql (caar list) 'parallel)))
	 (func-t-c (trans-exp (if parallelp (cdar list) (car list))
			      stack definitions)) 
	 (func-type (car func-t-c))
	 (func-code (cdr func-t-c))
	 (new-stack (cons (list 0 func-type) stack))
	 (arg-t-c (trans-exp (second list) new-stack definitions)) 
	 (arg-type (car arg-t-c))
	 (arg-code (cdr arg-t-c))
	 (apply-t-c (c-fun-def func-type arg-type parallelp definitions))
    	 (apply-type (car apply-t-c))
    	 (apply-code (cdr apply-t-c))
	 (result-code (nconc apply-code arg-code func-code)))
    (cons apply-type result-code)))

(defun trans-pair (exp stack definitions)
  (let* ((arg1-t-c (trans-exp (second exp) stack definitions))
	 (arg1-type (car arg1-t-c)) 
	 (arg1-code (cdr arg1-t-c))
	 (new-stack (cons (list 0 arg1-type) stack))
	 (arg2-t-c (trans-exp (third exp) new-stack definitions))
	 (arg2-type (car arg2-t-c)) 
	 (arg2-code (cdr arg2-t-c))
	 (result-type (list (first exp) arg1-type arg2-type))
	 (result-code (nconc arg2-code arg1-code)))
    (cons result-type result-code)))

(defun trans-closure (exp stack definitions)
  (let* ((func (second exp))
	 (arg-t-c  (trans-exp (third exp) stack definitions))
	 (arg-type (car arg-t-c)) 
	 (arg-code (cdr arg-t-c))
	 (func (if (listp func) (second (second func)) func))
	 (funtype (instantiate-fun-type (code-type (pscode-serial func))))
	 (stype (fourth funtype))
	 (result-type (list 'function (list 'closure func arg-type)
			    (third funtype) (third stype))))
    (when (unify (second stype) arg-type definitions) 
      (nesl-error "Yikes, internal error in trans-closure"))
    (cons result-type arg-code)))

;;(defun trans-closure (exp stack definitions)
;;  (let* ((func (second exp))
;;	 (arg-t-c  (trans-exp (third exp) stack definitions))
;;	 (arg-type (car arg-t-c)) 
;;	 (arg-code (cdr arg-t-c))
;;	 (result-type (list 'function (list 'closure func arg-type) nil nil)))
;;    (cons result-type arg-code)))

;;(defun trans-vector (exp stack definitions)
;;  (let* ((arg1-t-c (trans-exp (second exp) stack definitions))
;;	 (arg1-type (car arg1-t-c)) 
;;	 (arg1-code (cdr arg1-t-c))
;;	 (result-type (list 'vector (third arg1-type))))
;;    (cons result-type arg1-code)))

(defun fix-then-body (body else-type defs)
  (if (eql body 'then-body-hack)
      (make-empty-constant else-type defs)
    body))

(defun trans-if (exp stack definitions)
  (let* ((cond-l-c (trans-exp (second exp) stack definitions))
	 (bstack (copy-tree stack))
	 (else-pops (mapcar #'(lambda (var) (pop-stack var bstack definitions))
			    (set-difference (fifth exp) (sixth exp))))
	 (else-l-c (trans-exp (fourth exp) bstack definitions))
	 (then-body (fix-then-body (third exp) (car else-l-c) definitions))
	 (then-pops (mapcar #'(lambda (var) (pop-stack var stack definitions))
			    (set-difference (sixth exp) (fifth exp))))
	 (then-l-c (trans-exp then-body stack definitions)))
    (when (not (or (equal (car then-l-c) (car else-l-c))
		   (equal (clean-type (car then-l-c) definitions)
			  (clean-type (car else-l-c) definitions))))
      (nesl-error "Internal error: Then and else part must be of same type.
then-type: ~a~%else-type: ~a" (car then-l-c) (car else-l-c)))
    (when (not (equal (car cond-l-c) 'bool))
      (nesl-error "~a in ~s is a ~s instead of a BOOL."
		  (second exp) exp (car cond-l-c) ))
    (cons (car then-l-c)
	  (nconc
	   (list (list 'ENDIF))
	   (cdr else-l-c)
	   (reverse else-pops)
	   (list (list 'ELSE))
	   (cdr then-l-c)
	   (reverse then-pops)
	   (list (list 'IF))
	   (cdr cond-l-c)))))

(defun match-names-exp (names type)
  (cond ((atom names) 
	 (list (list names type)))
	((eql (car names) 'pair)
	 (if (eql (car type) 'pair)
	     (append (match-names-exp (second names) (second type))
		     (match-names-exp (third names) (third type)))
	   (append (match-names-exp (second names) 
				    (instantiate-fun-type
				     (code-type (pscode-serial 
						 (second (second type))))))
		   (match-names-exp (third names) (third (second type))))))
	((eql (car names) 'vector)
	 (match-names-exp (second names)
			  (list 'pair 'segdes (second type))))))

(defun trans-let (exp stack definitions)
  (let ((pattern-exp (first (second exp)))
	(binding-exp (second (second exp)))
	(unused-vars (third (second exp)))
	(body-exp (third exp)))
    (let* ((exp (trans-exp binding-exp stack definitions))
	   (exp-type (car exp))
	   (exp-code (cdr exp))
	   (matched-names (match-names-exp pattern-exp exp-type)))
      (when (eql matched-names :error)
	(nesl-error "Internal error: pattern match"))
      (let* ((nstack (append (reverse matched-names) 
			     (cons '(with-hold) stack)))
	    (unused-pops 
	     (mapcar #'(lambda (var) (pop-stack var nstack definitions))
		     unused-vars))
	    (body-l-c (trans-exp body-exp nstack definitions)))
	(when (plusp (second  (size-depth-type 'with-hold 
					       nstack definitions)))
	  (nesl-error "Internal error: unfreed variable"))
	(cons (first body-l-c)
	      (nconc (cdr body-l-c) (reverse unused-pops) exp-code))))))

(defun trans-constant (constant)
  (let ((type (nesl-constant-p constant)))
    (cons type (list (list 'VCONST type constant)))))

(defun trans-exp (expression stack definitions)
  (cond ((nesl-constant-p expression)
	 (trans-constant expression))
	((symbolp expression)
	 (copy-stack expression stack definitions))
	((pscode-p expression)
	 (trans-global-constant expression definitions))
	((listp expression)
	 (cond ((eql (car expression) 'let)
		(trans-let expression stack definitions))
	       ((eql (car expression) 'if)
		(trans-if expression stack definitions))
	       ((eql (car expression) 'pair)
		(trans-pair expression stack definitions))
	       ((eql (car expression) 'closure)
		(trans-closure expression stack definitions))
	       ;;((eql (car expression) 'vector)
	       ;;(trans-vector expression stack definitions))
	       ((eql (car expression) 'last)
		(copy-pop-stack (second expression) stack definitions))
	       (t (trans-func expression stack definitions))))
	(t (nesl-error "Invalid expression, ~s." expression))))

(defun trans-op (arguments source-type body definitions)
  (if (and (listp body) (eql (car body) :primitive))
      (cons nil (cons :inline (cdr body)))
    (flet ((match-names-check (name type)
	     (let ((matched-names (match-names-exp name type)))
	       (when (eql matched-names :error)
		 (nesl-error "Pattern ~a does not match specified type ~a."
			     name type))
	       matched-names)))
      (let* ((inlinep (and (listp body) (eql (car body) :inline)))
	     (body (if inlinep (cdr body) body))
	     (stack (reverse (cons '(op-hold) (mapcan #'match-names-check
						      arguments source-type))))
	     (l-c (trans-exp body stack definitions))
	     (result-type (car l-c))
	     (code (let ((depth (second (size-depth-type 
					 'op-hold stack definitions))))
		     (if (plusp depth)
			 (cons (list 'POP depth 
				     (l-from-type result-type 
						  definitions))
			       (cdr l-c))
		       (cdr l-c)))))
	(cons result-type 
	      (if inlinep 
		  (cons :inline (reverse code))
		(reverse (cons (list 'RET) code))))))))
