;;;
;;; Copyright (c) 1992, 1993, 1994 Carnegie Mellon University
;;; All Rights Reserved.
;;;
;;; See COPYRIGHT file for further information.
;;;

(in-package :nesl-lisp)

;; Store character strings for the first *max-push-pop* integers
(defparameter *max-push-pop* 400)
(defparameter *intlist* nil)

(defun make-int-array (size)
  (let ((result (make-array size)))
    (dotimes (i size)
      (setf (aref result i) (format nil " ~d" i)))
    result))

(eval-when (load eval)
  (setq *intlist* (make-int-array *max-push-pop*)))

(defparameter *fun-hash-table* 
  (make-hash-table :test #'equal :size 2000))

;; Returns Nil if already in the table, otherwise it adds it and returns T
(defun add-to-table (funname)
  (if (gethash funname *fun-hash-table*) nil
    (setf (gethash funname *fun-hash-table*) t)))

(defun clear-table () (clrhash *fun-hash-table*))

(defun table-to-list ()
  (let ((lst nil))
    (maphash #'(lambda (key val) (declare (ignore val))
		 (push key lst))
	     *fun-hash-table*)
    lst))

(defun time-stamp ()
  (multiple-value-bind 
   (second-ignore-me minute hour date month year) (get-decoded-time)
   (declare (ignore second-ignore-me))
   (format nil "~D/~D/~D ~2,'0D:~2,'0D" month date year hour minute)))

(defun get-call-recursive (func definitions)
  (let ((calls (car (generate-code (car func) (cdr func) nil definitions))))
    (get-calls-recursive calls definitions)))
  
(defun get-calls-recursive (func-list definitions)
  (if (null func-list) nil
    (progn 
      (when (add-to-table (car func-list))
	(get-call-recursive (car func-list) definitions))
      (get-calls-recursive (cdr func-list) definitions))))

(defun get-all-calls (name definitions)
  (clear-table)
  (get-call-recursive name definitions)
  (cons name (table-to-list)))

(defun write-code (name source-type ostream definitions)
  (declare (special *nesl-version* *verbose*))
  (format ostream "{ Generated by the NESL Compiler, Version ~a, on ~a. }~%"
	  *nesl-version* (time-stamp))
  (let ((calls (get-all-calls (cons name source-type) definitions)))
    (when *verbose* (format t "Writing..") (force-output))
    (dolist (func calls)
      (let ((code (cdr (get-cached-code (car func) (cdr func) definitions))))
	(dolist (line code) 
		(declare (type string line))
		(write-string line ostream))
	(terpri ostream)))))

(defvar vcode_copy "
COPY")

(defvar vcode_if "
IF")

(defvar vcode_pop "
POP")

(defvar vcode_cpop "
CPOP")

(defvar vcode_else "
ELSE")

(defvar vcode_call "
CALL ")

(defvar vcode_cr "
")

(defvar vcode_space " ")

(defun nums-rec (nums)
  (if (null nums) (list ")") 
    (cons (format nil " ~a" (car nums)) 
	  (nums-rec (cdr nums)))))

(defun nums (nums)
  (cond ((null nums) (list " ()"))
	((null (cdr nums)) (list (format nil " ~a" (car nums))))
	(t (cons " (" (nums-rec nums)))))

(defun convert-anything (thing)
  (cond ((typep thing 'integer) thing)
	((typep thing 'float) thing)
	((member thing '(t nil f)) (if (eql thing t) "T" "F"))
	((typep thing 'character) (char-code thing))
	(t (error "Don't know how to print out ~s." thing))))

(defun str-const-line (type value)
  (cond ((eql type 'segdes)
	 (append (cons vcode_cr (cons "CONST INT" (nums value)))
		 (list vcode_cr "MAKE_SEGDES")))
	((and (eql type 'char) 
	      (not (or (> (length value) 100)
		       (find #\  value) (find #\" value)
		       (find #\\ value))))
	 (list (format nil "~%CONST CHAR ~s" (coerce value 'string))))
	(t
	 (let ((const (cond ((eql type 'int) "CONST INT")
			    ((eql type 'char) "CONST INT")
			    ((eql type 'float) "CONST FLOAT")
			    ((eql type 'bool) "CONST BOOL"))))
	   (cons vcode_cr 
		 (cons const (nums (mapcar #'CONVERT-ANYTHING value))))))))

(defun func-to-string (code definitions)
  (if (null code) nil
    (let* ((pline (car code)))
      (append 
       (cond ((eql (car pline) 'VCONST)
	      (str-const-line (second pline) 
			      (vcode-vector-data (third pline))))
	     ((eql (car pline) 'COPY)
	      (list (concatenate 'string
				 vcode_copy
				 (aref *intlist* (second pline))
				 (aref *intlist* (third pline)))))
	     ((eql (car pline) 'POP)
	      (list (concatenate 'string 
				 vcode_pop
				 (aref *intlist* (second pline))
				 (aref *intlist* (third pline)))))
	     ((eql (car pline) 'CPOP)
	      (list (concatenate 'string 
				 vcode_cpop
				 (aref *intlist* (second pline))
				 (aref *intlist* (third pline)))))
	     ((eql (car pline) 'IF) (list vcode_if))
	     ((eql (car pline) 'ELSE) (list vcode_else))
	     ((eql (car pline) 'CALL)
	      (list vcode_call (string (get-full-name (car (second pline))
						      (cdr (second pline))
						      definitions))))
	     ((= (length pline) 1)
	      (list vcode_cr (string (first pline))))
	     ((= (length pline) 2)
	      (list vcode_cr (string (first pline)) 
		    vcode_space (string (second pline))))
	     ((= (length pline) 3)
	      (list vcode_cr (string (first pline)) 
		    vcode_space (string (second pline))
		    vcode_space (format nil "~a" (third pline))))
	     (t nil))
       (func-to-string (cdr code) definitions)))))

(defun get-calls (code)
  (if (null code) nil
    (let ((rest (get-calls (cdr code))))
      (if (eql (first (car code)) 'call)
	  (cons (second (car code)) rest)
	rest))))

(defparameter *print-io-types* nil)

(defun make-string-code (funspec stypes code definitions)
  (let ((full-name (get-full-name funspec stypes definitions)))
    (cons (get-calls code)
	  (cons (if *print-io-types*
		    (let ((otype (if (is-function-type? (code-type funspec))
				     (get-instantiated-function-type 
				      funspec stypes definitions)
				   (car (code-type funspec)))))
		      (format nil "~%FUNC ~a~%I~{ ~a~}~%O~{ ~a~}"
			      full-name (flatten-type stypes definitions)
			      (flatten-type otype definitions)))
		  (format nil "~%FUNC ~a" full-name))
		(func-to-string code definitions)))))

(defun write-func (name source-type definitions filename)
  (with-open-file (ofile (merge-pathnames filename) :direction :output
			 :if-exists :supersede)
    (let ((scode (pscode-serial (binding-compiled-code 
				 (get-binding-definition name definitions))))) 
      (write-code scode source-type ofile definitions))))
