;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

(defun size-depth-type (symbol list definitions)
  (do* ((depth 0 (+ size depth))
	(rem-list list (cdr rem-list))
	(elt (car rem-list) (car rem-list))
	(size (l-from-type (second elt) definitions)
	      (l-from-type (second elt) definitions)))
       ((or (eql symbol (first elt)) (null (cdr rem-list)))
	(if (eql symbol (first elt))
	    (list size depth (second elt)) nil))))

(defun copy-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (if sdt
	(cons (third sdt) (list (list 'COPY (first sdt) (second sdt))))
      (let ((const (get-variable symbol definitions)))
	(if const (trans-exp (car const) stack definitions)
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun delete-type (symbol list definitions)
  (declare (ignore definitions))
  (do* ((rem-list list (cdr rem-list)))
       ((or (eql symbol (first (first rem-list))) (null (cdr rem-list)))
	(if (eql symbol (first (first rem-list)))
	    (rplacd (first rem-list) nil)))))

(defun copy-pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt
	(cons (third sdt) (list (list 'POP (first sdt) 
				      (+ (first sdt) (second sdt)))
				(list 'COPY (first sdt) (second sdt))))
      (let ((const (get-variable symbol definitions)))
	(if const (trans-exp (car const) stack definitions)
	  (nesl-error "There is no variable definition for ~s." symbol))))))

(defun pop-stack (symbol stack definitions)
  (let ((sdt (size-depth-type symbol stack definitions)))
    (delete-type symbol stack definitions)
    (if sdt (list 'POP (first sdt) (second sdt))
      (let ((const (get-variable symbol definitions)))
	(if const nil
	  (nesl-error "There is no variable definition for ~s." symbol))))))

;; This is a real hack -- it needs to be fixed up
(defun get-type-code (names body arg-types fun-type definitions)
  (cond ((not (listp body)) body)
	((eql (first body) 'nesl::base-typecase)
	 (let* ((type (foo-type fun-type (second body) arg-types definitions))
		(code (second (assoc type (cddr body)))))
	   (when (not code) (nesl-error "Bad type in Base-Typecase."))
	   code))
	((eql (first body) 'nesl::poly-typecase)
	 (let ((type (foo-type fun-type (second body) arg-types definitions)))
	   (cond ((atom type) 
		  (third body))
		 ((and (listp type) (eql (first type) 'nesl::vector))
		  (fourth body))
		 ((and (listp type) (eql (first type) 'nesl::pair))
		  (fifth body))
		 (t (nesl-error "Internal error in poly-typecase")))))
	(t body)))

(defun generate-code (fundef arg-type recursivep definitions)
  (let* ((names (fundef-names fundef))
	 (ccode (get-cached-code fundef arg-type definitions) ))
    (or (and (or recursivep (not (eql ccode :stub))) ccode)
	(let ((type-specific-code
	       (get-type-code names
			      (fundef-code fundef) arg-type
			      (fundef-codetype fundef)
			      definitions)))
	  ;;(print (list (car names) arg-type))
	  ;; This is added so that recursive calls work
	  (add-cached-code fundef arg-type :stub definitions)
	  (add-cached-code 
	   fundef arg-type
	   (cdr (trans-op (cdr names) arg-type
			  type-specific-code definitions)) 
	   definitions)))))

(defun get-call-code (fundef arg-type definitions)
  (let ((code (generate-code fundef arg-type t definitions)))
    (if (and (listp code) (eql (car code) :inline))
	(reverse (cdr code))
      ;; BACK CALL FIX (car names)
      (list (list 'CALL (cons fundef arg-type))))))

(defun fun-def (fname arg-type definitions)
  ;; BACK CALL FIX
  (let ((fundef (if (fundef-p fname) 
		    fname
		  (get-fundef fname definitions))))
    (when (not fundef)
      (nesl-error "Function ~a is undefined (internal error)." fname))
    (let ((full-type (get-instantiated-function-type 
		      fundef arg-type definitions)))
      (cons (car full-type) 
	    (get-call-code fundef arg-type definitions)))))

(defun trans-func (list stack definitions)
  (let* ((arg-t-c (trans-list (cdr list) stack definitions)) 
	 (arg-type (car arg-t-c))
	 (arg-code (cdr arg-t-c))
	 (fun-t-c (fun-def (first list) arg-type definitions)))
    (nconc fun-t-c arg-code)))

(defun trans-if (exp stack definitions)
  (let* ((cond-l-c (trans-exp (second exp) stack definitions))
	 (bstack (copy-tree stack))
	 (if-pops (mapcar #'(lambda (var) (pop-stack var stack definitions))
			  (set-difference (sixth exp) (fifth exp))))
	 (if-l-c (trans-exp (third exp) stack definitions))
	 (else-pops (mapcar #'(lambda (var) (pop-stack var bstack definitions))
			  (set-difference (fifth exp) (sixth exp))))
	 (else-l-c (trans-exp (fourth exp) bstack definitions)))
    (when (not (equal (car if-l-c) (car else-l-c)))
	  (nesl-error "If and else part must be of same type for ~s." exp))
    (when (not (equal (car cond-l-c) 'bool))
	  (nesl-error "~a in ~s is a ~s instead of a BOOL."
		 (second exp) exp (car cond-l-c) ))
    (cons (car if-l-c)
	  (nconc
	    (list (list 'ENDIF))
	    (cdr else-l-c)
	    (reverse else-pops)
	    (list (list 'ELSE))
	    (cdr if-l-c)
	    (reverse if-pops)
	    (list (list 'IF))
	    (cdr cond-l-c)))))

(defun instantiate-rec-type (type definitions)
  (let ((record (get-typedef (car type) definitions)))
    (if (not record)
	(nesl-error "No such type ~a" (car type))
      (let* ((record-types (typedef-type record))
	     (assoc (mapcar #'(lambda (rec-type val) 
				(cons (first rec-type) val))
			    (cdr record-types) (cdr type))))
	(sublis assoc (cdr (car record-types)))))))

(defun match-names-list (names type definitions)
  (if names
      (append (match-names-exp (car names) (car type) definitions)
	      (match-names-list (cdr names) (cdr type) definitions))))

(defun match-names-exp (names type definitions)
  (if (atom names) 
      (list (list names type))
    (let ((instance (instantiate-rec-type type definitions)))
      (match-names-list (cdr names) instance definitions))))

(defun trans-with (exp stack definitions)
  (when (/= (length exp) 3)
	(nesl-error 
	 "A WITH form must contain one binding list and one expression"))
  (do* ((rem (second exp) (cdr rem))
	(code nil)
        (nstack (cons '(with-hold) stack)))
       ((not rem)
	(let ((body-l-c (trans-exp (third exp) nstack definitions)))
	  (cons (first body-l-c)
		(let ((depth (second (size-depth-type 'with-hold 
						      nstack definitions)))
		      (fullcode (nconc (cdr body-l-c) code))
		      (resultsize (l-from-type (first body-l-c) definitions)))
		  (if (plusp depth)
		      (cons (list 'POP depth resultsize) fullcode)
		    fullcode)))))
       (let* ((binding (car rem))
	      (exp (trans-exp (second binding) nstack definitions))
	      (exp-type (car exp))
	      (exp-code (cdr exp))
	      (name (first binding))
	      (matched-names (match-names-exp name exp-type definitions)))
	 (when (eql matched-names :error)
	       (nesl-error "Pattern ~a does not match assigned value ~a,~%~
                        which is of type ~a."
			   name (second binding) exp-type))
	 (setq nstack (append (reverse matched-names) nstack))
	 (let ((unused-pops 
		(mapcar #'(lambda (var) (pop-stack var nstack definitions))
			(cddr binding))))
	   (setq code (nconc (nconc (reverse unused-pops) exp-code) code))))))

(defun trans-constant (constant)
  (let ((type (nesl-constant-p constant)))
    (cons type (list (list 'VCONST type constant)))))

(defun trans-list (list stack definitions)
  (do* ((type nil)
	(rem list (cdr rem))
	(nstack stack)
	(code nil))
       ((not rem) (cons (reverse type) code))
       (let ((l-c (trans-exp (car rem) nstack definitions)))
	 (setq nstack (cons (list 0 (car l-c)) nstack))
	 (setq code (nconc (cdr l-c) code))
	 (setq type (cons (car l-c) type)))))

(defun trans-timer (expression stack definitions)
   (let ((subparse (trans-exp (second expression) stack definitions)))
     (cons (list 'nesl::pair (car subparse) 'nesl::float)
	   (append (list '(STOP_TIMER))
		   (cdr subparse)
		   (list '(START_TIMER))))))

(defun trans-exp (expression stack definitions)
  (cond ((nesl-constant-p expression)
	 (trans-constant expression))
	((symbolp expression)
	 (copy-stack expression stack definitions))
	((listp expression)
	 (cond ((eql (car expression) 'last)
		(copy-pop-stack (second expression) stack definitions))
	       ((eql (car expression) 'if)
		(trans-if expression stack definitions))
	       ((eql (car expression) 'with)
		(trans-with expression stack definitions))
	       ((eql (car expression) 'nesl::time)
		(trans-timer expression stack definitions))
	       (t (trans-func expression stack definitions))))
	(t (nesl-error "Invalid expression, ~s." expression))))

(defun trans-op (arguments source-type body definitions)
  (if (and (listp body) (eql (car body) :primitive))
      (cons nil (cons :inline (cdr body)))
    (flet ((match-names-check (name type)
	     (let ((matched-names (match-names-exp name type definitions)))
	       (when (eql matched-names :error)
		 (nesl-error "Pattern ~a does not match specified type ~a."
			     name type))
	       matched-names)))
    (let ((stack (reverse (cons '(op-hold) (mapcan #'match-names-check
						   arguments source-type)))))
      (let* ((l-c (trans-exp body stack definitions))
	     (result-type (car l-c)))
	(cons result-type
	      (reverse 
	       (cons (list 'RET)
		     (let ((depth (second (size-depth-type 'op-hold 
							   stack definitions))))
		       (if (plusp depth)
			   (cons (list 'POP depth 
				       (l-from-type result-type definitions))
				 (cdr l-c))
			 (cdr l-c)))))))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; BELOW HERE IS HACKED CODE TO MAKE POLYMORPHIC TYPES WORK
;;;;;;;;;;;;;;;;;;;

(defun gen-poly-code (args fun-types var struct options)
  (multiple-value-bind (binds args1 args2)
    (variable-bind-calls-pair (second args) (second fun-types) var)
    (let* ((calls (list (list (car args) args1)
			(list (car args) args2)))
	   (ret-code (return-poly-code 
		      calls (car fun-types) 
		      struct var (car options) (cdr options))))
      (free-body `(with ,binds ,ret-code)))))

(defun get-keyword (key lst default) 
  (if lst 
      (if (eql key (car lst)) 
	  (second lst) 
	(get-keyword key (cddr lst) default))
    default))

(defun return-poly-code (calls ret-type struct var pflag combine-funcs)
  (let* ((combine-func (get-keyword :combine combine-funcs nil))
	 (wrap-func (get-keyword :wrap combine-funcs nil))
	 (combine-code (combine-code-pair calls ret-type
					  var pflag combine-func)))
    (if wrap-func
	`(with ((wrap-val ,combine-code))
	   ,(conv-exp `(,wrap-func (nesl::pair "rec",(string (car struct)) 
					       wrap-val)) 
		      pflag nil))
      combine-code)))

(defun variable-bind-calls-pair (args argtypes typevar)
  (if (listp args)
      (multiple-value-bind (bindsleft vleft1 vleft2)
	  (variable-bind-calls-pair (second args) (second argtypes) typevar)
	(multiple-value-bind (bindsright vright1 vright2)
	  (variable-bind-calls-pair (third args) (third argtypes) typevar)
	  (values (append bindsleft bindsright)
		  `(nesl::pair ,vleft1 ,vright1)
		  `(nesl::pair ,vleft2 ,vright2))))
    (variable-bind-call-pair args argtypes typevar)))

(defun variable-bind-call-pair (arg argtype typevar)
  (cond ((eql argtype typevar)
	 (let ((v1 (gensym)) 
	       (v2 (gensym)))
	   (values `(((nesl::pair ,v1 ,v2) ,arg))
		   v1 v2)))
	((and (listp argtype) (eql (car argtype) 'nesl::vector)
	      (eql (second argtype) typevar))
	 (let ((v1 (gensym))
	       (v2 (gensym))
	       (seg (gensym)))
	   (values `(((nesl::vector 
		    (nesl::pair ,seg (nesl::pair ,v1 ,v2))) ,arg))
		 `(nesl::vector (nesl::pair ,seg ,v1))
		 `(nesl::vector (nesl::pair ,seg ,v2)))))
	(t 
	 (values nil arg arg))))

(defun combine-code-pair (calls return-type typevar pflag combine-func)
  (cond (combine-func
	 (let ((g1 (gensym)) (g2 (gensym)))
	   `(with ((,g1 ,(first calls)) 
		   (,g2 ,(second calls)))
	      ,(conv-exp `(,combine-func (nesl::pair ,g1 ,g2)) pflag nil))))
	((eql return-type typevar)
	 (cons 'nesl::pair calls))
	((and (listp return-type) (eql (car return-type) 'nesl::vector)
	      (eql (second return-type) typevar))
	 (let ((g1 (gensym)) (g2 (gensym)))
	   `(with (((nesl::vector (nesl::pair seg ,g1)) ,(first calls))
		   ((nesl::vector (nesl::pair seg ,g2)) ,(second calls)))
	      (nesl::vector (nesl::pair seg (nesl::pair ,g1 ,g2))))))
	(t (nesl-error "Poly-typecase can't return the type ~a." 
		       return-type))))
