;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package :nesl-lisp) 

(eval-when (compile)
  (proclaim '(special *max-print-length* *verbose* *obj-tailer* 
		      *nesl-bugs* *doclist* *current-typedef* 
		      *current-fundef*)))

;;;;;;;;;;;;;;;;;;
;;; THIS KEEPS ALL THE NESL STATE (functions, types, variables...)
;;;;;;;;;;;;;;;;;;

(defparameter *definitions* (make-defs-table))

;;;;;;;;;;;;;;;;;;
;;; THESE ARE VARIOUS TOPLEVEL PARAMETERS
;;;;;;;;;;;;;;;;;;

(defparameter *debug* nil)
(defparameter *redefine-default* nil)
(defparameter *cnesl-syntax* t)

;;;;;;;;;;;;;;;;;;
;;; KEEP TRACK OF WHAT ARE SPECIAL FORMS AND TOPLEVEL COMMANDS
;;;;;;;;;;;;;;;;;;

(defparameter *special-forms* 
  '(nesl::with nesl::over nesl::if nesl::defop nesl::defrec 
	       nesl::set))

(defparameter *top-level-commands* 
  '(progn nesl::use-machine nesl::cm-finger nesl::set_argcheck nesl::qstat
     nesl::use_machine nesl::cmfinger nesl::set_memory_size nesl::list_machines
     nesl::configuration nesl::set-memsize nesl::set-username
     nesl::list-machines))

(defparameter *top-level-funcs* '())

;;;;;;;;;;;;;;;;;;
;; Makes sure that users don't use t and f as variable names
;;;;;;;;;;;;;;;;;;

(add-nondefinable-constant 't *definitions*)
(add-nondefinable-constant 'f *definitions*)

;;;;;;;;;;;;;;
;;; ERROR HANDLER
;;;;;;;;;;;;;;

(defun nesl-error (format-string &rest args)
  (if *current-fundef*
      (if *cnesl-syntax*
	  (format t "~%Error in Function definition ~a.  " *current-fundef*)
	(format t "~%Error in (DEFOP (~a ..) ..).  " *current-fundef*))
    (if *current-typedef*
	(if *cnesl-syntax*
	    (format t "~%Error in Datatype declaration ~a.  " 
		    *current-typedef*)
	  (format t "~%Error in (DEFREC (~a ..) ..).  " *current-typedef*))
      (format t "~%Error at top level.  ")))
  (cond (*debug*
	 (format t "~%")
	 (apply 'error format-string args))
	(t
	 (format t "~%")
	 (apply 'format t format-string args)
	 (throw 'nesl-error :error))))

;;;;;;;;;;;;;
;;; HELP FACILITIES
;;;;;;;;;;;;;

(defun describe-nesl (funname definitions)
  (let ((fundef (get-fundef funname definitions)))
    (cond ((not funname)
	   (dolist (fun *doclist*)
	     (if (and (symbolp fun) (get-fundef fun definitions))
	       (let* ((fundef (get-fundef fun definitions))
		      (interface (pprint-nesl-string (fundef-names fundef))))
		 (if (fundef-shortdoc fundef)
		     (format t "~25a ~a" interface (fundef-shortdoc fundef))
		   (format t "~a" interface))))))
	  ((member funname *special-forms*)
	   (format t "~%~a is a special form; please see the NESL manual." 
		   funname))
	  ((member funname *top-level-commands*)
	   (format t "~%~a is a top level command; please use (help)." 
		   funname))
	  (fundef
	   (let ((interface (fundef-names fundef))
		 (type (fundef-type fundef))
		 (documentation (fundef-documentation fundef)))
	     (format t "~%INTERFACE:~% ~a~%~%"  
		     (pprint-nesl-string interface))
	     (when type (format t "TYPE:~% ~a~%~%" 
				(pretty-type-func type)))
	     (when documentation 
		   (format t "DOCUMENTATION:~%~% ~a~%" documentation))))
	  (t (format t "Function ~a not found." funname)))
    funname))

(defun nesl-help ()
  (format t 
"~%NESL toplevel forms:~%  ~
    (DEFOP (name arg*) [! typespec] exp)  -- Function Definition~%  ~
    (DEFREC (name typeexp*) typebind*)    -- Record Definition~%  ~
    (SET name exp)                        -- Toplevel Assignment~%  ~
    exp                                   -- Any NESL expression~%~%~
Toplevel Commands:~%  ~
    (DESCRIBE funname)    -- Describe a NESL function with funname.~%  ~
    (LOAD filename)       -- Load a file.~%  ~
    (VERBOSE)             -- Toggle the verbose switch.~%  ~
    (SET-PRINT-LENGTH n)  -- Set the maximum number of elements that are
                             printed in a sequence.~%  ~
    (LISP) or (EXIT)      -- Exit NESL and go to the Lisp interpreter.~%  ~
    (HELP)                -- Print this message.~%  ~
    (BUGS)                -- Lists the known bugs.~%~%~
Commands for running VCODE on remote machines:~%  ~
    (CONFIGURATION)       -- List the properties of the current configuration.~%  ~
    (USE-MACHINE config)  -- Use the machine with configuration CONFIG.~%  ~
    (LIST-MACHINES)       -- List the available configurations.~%  ~
    (SET-MEMSIZE n)       -- Set the memory size of the current configuration."
	  ))

(defun cnesl-help ()
  (format t 
"~%NESL toplevel forms:~%  ~
   function name(arg1,..,argn) [: typespec] = exp;  -- Function Definition~%  ~
   datatype name(t1,..tn) [:: typebind];            -- Record Definition~%  ~
   name = exp;                                      -- Toplevel Assignment~%  ~
   exp;                                             -- Any NESL expression~%~%~
Toplevel Commands:~%  ~
   DESCRIBE(funname);    -- Describe a NESL function with funname.~%  ~
   LOAD(filename);       -- Load a file.~%  ~
   VERBOSE();            -- Toggle the verbose switch.~%  ~
   SET_PRINT_LENGTH(n);  -- Set the maximum number of elements that are
                            printed in a sequence.~%  ~
   LISP(); or EXIT();    -- Exit NESL and go to the Lisp interpreter.~%  ~
   HELP();               -- Print this message.~%  ~
   BUGS();               -- Lists the known bugs.~%~%~
Commands for running VCODE on remote machines:~%  ~
   CONFIGURATION();      -- List the properties of the current configuration.~%  ~
   USE_MACHINE(config);  -- Use the machine with configuration CONFIG.~%  ~
   LIST_MACHINES();      -- List the available configurations.~%  ~
   SET_MEMORY_SIZE(n);   -- Set the memory size of the current configuration.
   name &= exp [,MEM := exp] [,MAXTIME := exp];
                         -- Executes exp in the background~%~%"
	  ))

(defun nesl-list-bugs ()
  (let ((i 0))
    (dolist (bug *nesl-bugs*)
      (format t "~%~a: ~a~%" (incf i) bug))))

;;;;;;;;;;;;
;;; STUFF FOR LOADING FILES
;;;;;;;;;;;;

(defun file-check (filename suffix type)
  (cond ((and (> (length filename) (length suffix))
	      (string-equal suffix
			    (subseq filename (- (length filename) 
						(length suffix))
				    (length filename)))
	      (probe-file filename))
	 (cons type filename))
	((probe-file (concatenate 'string filename suffix))
	 (cons type (concatenate 'string filename suffix)))))

(defun get-filename-and-type (filename)
  (let ((type-name
	 (cond ((file-check filename ".cnesl" :cnesl))
	       ((file-check filename ".nesl" :nesl))
	       ((file-check filename ".lisp" :lisp))
	       ((file-check filename *obj-tailer* :lisp))
	       ((probe-file filename) (cons :nesl filename))
	       (t (cons nil filename)))))
    (values (car type-name) (cdr type-name))))

(defun load-nesl (filename &key (verbose t) (print nil))
  (when (not (stringp filename))
    (nesl-error "The second argument to LOAD must be a character string."))
  (multiple-value-bind (type name) 
    (get-filename-and-type (namestring (merge-pathnames filename)))
    (cond ((not type)
	   (nesl-error "File ~a does not exist." name))
	  ((eql type :lisp)
	   (load name :verbose verbose :print print))
	  ((eql type :nesl)
	   (with-open-file (loadstr name :direction :input)
	     (when verbose 
	       (format t "~%; Nesl loading ~a." name))
	     (let ((*cnesl-syntax* nil))
	       (declare (special *cnesl-syntax*))
	       (nesl-loop loadstr :interactive nil :print print))))
	  ((eql type :cnesl)
	   (when (not (fboundp 'cgolread))
	     (nesl-error "CNESL is not loaded, use (load-cnesl-syntax)."))
	   (with-open-file (loadstr name :direction :input)
	     (when verbose 
	       (format t "~%; CNesl loading ~a." name))
	     (let ((*cnesl-syntax* t))
	       (declare (special *cnesl-syntax*))
	       (nesl-loop loadstr :interactive nil :print print)))))))

(defun trans-nesl-to-cnesl (filename)
  (let ((infile (concatenate 'string filename ".nesl"))
	(outfile (concatenate 'string filename ".cnesl"))
	(*cnesl-syntax* nil))
    (when (probe-file outfile) (delete-file outfile))
    (with-open-file (instr infile :direction :input)
      (with-open-file (outstr outfile :direction :output)
	(loop
	 (let ((readval (nesl-read instr nil)))
	   (if (nesl-exit? readval)
	       (return)
	     (progn
	       (pprint-nesl readval outstr)
	       (format outstr " $~%")))))))))

;;;;;;;;;;;;;;;;
;;; EVALUATING A NESL FORM
;;;;;;;;;;;;;;;;

;; This adds a line to the beginning and end of the code that prints
;; when interpretation is starting and ending.
(defun verbose-wrapper (form)
  (if *verbose*
      `(nesl::with ((cr (nesl::print_char #\newline))
		    (start (nesl::print_string "Start of interpretation..."))
		    (cr (nesl::print_char #\newline))
		    (result ,form)
		    (end (nesl::print_string "End of interpretation...")))
		   result)
    form))

(defun eval-nesl (form definitions &optional background)
  (cond ((nesl-constant-p form)
	 (values (coerce-nesl-constant form)
		 (nesl-constant-p form)))
	((listp form)
	 (let ((rtype (caar (typecheck-op '(main) nil form definitions)))
	       (*print-pretty* nil))
	   (declare (special *print-pretty*))
	   (when *verbose* (format t "~%Compiling VCODE...."))
	   (let* ((rtype-code (trans-op 
			       nil nil 
			       (trans-phase1 nil (verbose-wrapper form) nil)
			       definitions))
		  (code (cdr rtype-code))
		  (result-type (car rtype-code)))
	     (add-cached-code (get-fundef 'main definitions)
			      nil code definitions)
	     (values 
	      (if background
		  (run-vcode-background 'main definitions background)
		(group-data 
		 (run-vcode 'main (flatten-type result-type definitions)
			    definitions)
		 rtype definitions))
	      rtype))))
	((symbolp form)
	 (let ((val-type (get-variable form definitions)))
	   (cond ((and (listp (car val-type))
		       (eql `background (caar val-type)))
		  (let* ((out-err-files (second (car val-type)))
			 (type (cdr val-type))
			 (flattype (flatten-type type definitions))
			 (res (get-background-result out-err-files flattype))
			 (val (group-data res type definitions)))
		    (cond (res
			   (add-variable form val type definitions)
			   (values val type))
			  (t (nesl-error "Variable waiting for result.")))))
		 (val-type 
		  (values (car val-type) (cdr val-type)))
		 (t (nesl-error "Variable ~a is undefined." form)))))
	(t (nesl-error "Invalid form: ~a" form))))	 

;;;;;;;;;;;;;;;;;;;
;;; EVALUATING A TOPLEVEL FORM....yes, it is a giant case statement
;;;;;;;;;;;;;;;;;;;

(defun eval-toplevel (form definitions)
  (if (listp form)
      (cond ((member (car form) *top-level-commands*)
	     (eval form))
	    ((member (car form) *top-level-funcs*)
	     (apply (car form) (mapcar 
				#'(lambda (form) (eval-nesl form definitions))
				(cdr form))))
	    ((eql (car form) 'noop)
	     (values nil nil))
	    ((eql (car form) 'nesl::ptrans)
	     (pprint-nesl
	      (conv-body nil (strip-exp (second form) definitions) nil)))
	    ((eql (car form) 'nesl::ptransp)
	     (pprint-nesl
	      (conv-body nil (strip-exp (second form) definitions) t)))
	    ((eql (car form) 'nesl::pprint)
	     (pprint-nesl  (second form)))
	    ((eql (car form) 'nesl::cnesl_trans)
	     (trans-nesl-to-cnesl (second form)))
	    ((eql (car form) 'nesl::set)
	     (when (not (eql (length form) 3))
	       (nesl-error "The syntax for SET is: (SET Ident exp)."))
	     (check-valid-varname (second form) definitions)
	     (multiple-value-bind (val type)
	         (eval-nesl (third form) definitions)
	       (add-variable (second form) val type definitions)
	       (values val type)))
	    ((eql (car form) 'nesl::help)
	     (if *cnesl-syntax* (cnesl-help) (nesl-help)))
	    ((eql (car form) 'nesl::bugs)
	     (nesl-list-bugs))
	    ((eql (car form) 'nesl::defop)
	     (parse-defop (cdr form) definitions))
	    ((eql (car form) 'nesl::defrec)
	     (parse-defrec (cdr form) definitions))
	    ((eql (car form) 'nesl::defprimtype)
	     (add-prim-type (second form) definitions))
	    ((eql (car form) 'nesl::deftypeclass)
	     (add-type-class (second form) (third form) definitions))
	    ((or (eql (car form) 'nesl::cgol)
		 (eql (car form) 'nesl::cnesl))
	     (when (not (fboundp 'cgolread))
	       (nesl-error "CNESL is not loaded, use (load-cnesl-syntax)."))
	     (if (setq *cnesl-syntax* (not *cnesl-syntax*))
		 (format t "~%CNESL syntax")
	       (format t "~%Lisp syntax")))
	    ((or (eql (car form) 'nesl::load-cnesl-syntax)
		 (eql (car form) 'nesl::load-cgol-syntax)
		 (eql (car form) 'nesl::loadcneslsyntax))
	     (load-cnesl-syntax))
	    ((or (eql (car form) 'nesl::load)
		 (eql (car form) 'nesl::load-nesl))
	     (apply #'load-nesl (cdr form)))
	    ((eql (car form) 'nesl::describe)
	     (describe-nesl (second form) definitions))
	    ((or (eql (car form) 'nesl::set-print-length)
		 (eql (car form) 'nesl::set_print_length))
	     (when (not (integerp (second form)))
	       (nesl-error "The second argument to SET_PRINT_LENGTH must be ~
                            an integer."))
	     (setq *max-print-length* (second form)))
	    ((eql (car form) 'nesl::setq)
	     (nesl-error "SETQ is not valid, use SET instead."))
	    ((eql (car form) 'nesl::verbose)
	     (if (setq *verbose* (not *verbose*))
		 (format t "~%Verbose: On")
	       (format t "~%Verbose: Off")))
	    ((eql (car form) 'nesl::redefinep)
	     (if (setq *redefine-default* (not *redefine-default*))
		 (format t "~%Redefine: On")
	       (format t "~%Redefine: Off")))
	    ((eql (car form) 'nesl::debug)
	     (if (setq *debug* (not *debug*))
		 (format t "~%Debug: On")
	       (format t "~%Debug: Off")))
	    ((eql (car form) 'background)
	     (let* ((var (second form))
		    (exp (car-pair (third form)))
		    (mem-time-qname (flatten-pair (cdr-pair (third form)))))
	       (check-valid-varname var definitions)
	       (multiple-value-bind (out-err-files type)
		   (eval-nesl exp definitions mem-time-qname)
		 (add-variable var `(background ,out-err-files) 
			       type definitions)
		 (values (coerce-nesl-constant "stub") type))))
	    (t (eval-nesl form definitions)))
      (eval-nesl form definitions)))

;;;;;;;;;;;;;
;;; THE NESL READ/EVAL/PRINT LOOP
;;;;;;;;;;;;;

(defun nesl-read (in-stream interactive)
  (if *CNESL-SYNTAX*
      ;; this is a real hack since cgolread doesn't work
      (prog1 (or (cgolread in-stream nil :exit) :exit)
	(when interactive (read-line *standard-input* nil)))
    (let ((result (nesl-read-toplevel (read in-stream nil :exit))))
      result)))

(defun pretty-print-function (names)
  (if *cnesl-syntax*
      (if (= (length names) 1)
	  (format nil "~a()" (first names))
	(format nil "~a(~a~{,~a~})" 
		(first names) (second names) (cddr names)))
    names))

(defun nesl-print (value type)
  (if type
      (if (and (listp type) (eql (car type) 'function))
	  (format t "~%~a : ~a" 
		  (pretty-type value) (pretty-type-func (cdr type)))
	(if *cnesl-syntax*
	    (format t "~%~a : ~a" 
		    (cnesl-print-data (nest-constant value) t)
		    (pretty-type type))
	  (format t "~%~s : ~a" (nest-constant value) (pretty-type type))))
    nil))

(defun nesl-exit? (readval)
  (or (eql readval :exit) 
      (and (listp readval)
	   (member (car readval) '(nesl::exit nesl::lisp nesl::quit)))))

(defun nesl-loop (in-stream &key interactive print)
  (let ((*package* (find-package 'nesl))
	(*read-default-float-format* 'double-float)
	(definitions *definitions*))
    (declare (special *package* *read-default-float-format*))
    (add-function '(main) nil nil nil definitions)
    (loop
     (when interactive (format t "~%<Nesl> ") (force-output))
     (catch 'nesl-error
       (let ((readval (nesl-read in-stream interactive)))
	 (if (nesl-exit? readval)
	     (return)
	   (multiple-value-bind (value type) 
	     (eval-toplevel readval definitions)
	     (when print (nesl-print value type)))))))))

(defun nesl ()
  (format t "~%; NESL version ~a (~a)~%" *nesl-version* *nesl-date*)
  (nesl-loop *standard-input* :interactive t :print t))

(defun user::nesl () (nesl))
