;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

;; Store character strings for the first *max-push-pop* integers
(defparameter *max-push-pop* 400)
(defparameter *intlist* nil)

(defun make-int-array (size)
  (let ((result (make-array size)))
    (dotimes (i size)
      (setf (aref result i) (format nil " ~d" i)))
    result))

(eval-when (load eval)
  (setq *intlist* (make-int-array *max-push-pop*)))

(defun time-stamp ()
  (multiple-value-bind 
   (second-ignore-me minute hour date month year) (get-decoded-time)
   (declare (ignore second-ignore-me))
   (format nil "~D/~D/~D ~2,'0D:~2,'0D" month date year hour minute)))

(defun get-call-recursive (func current-list definitions)
  (let ((calls (car (generate-code (car func) (cdr func) 
				  nil definitions))))
    (get-calls-recursive calls (cons func current-list) definitions)))
  
(defun get-calls-recursive (func-list current-list definitions)
  (if (null func-list) current-list
    (let ((new-list (if (find (car func-list) current-list :test #'equal)
			 current-list
		       (get-call-recursive (car func-list) current-list
					   definitions))))
      (get-calls-recursive (cdr func-list) new-list definitions))))

(defun write-code (name source-type ostream definitions)
  (format ostream "{ Generated by the NESL Compiler, Version ~a, on ~a. }~%"
	  *nesl-version* (time-stamp))
  (let ((calls (get-call-recursive (cons name source-type) nil definitions)))
    (dolist (func (reverse calls))
      (let ((code (cdr (get-cached-code (car func) (cdr func) definitions))))
	(dolist (line code) 
		(declare (type string line))
		(write-string line ostream))
	(terpri ostream)))))

(defvar vcode_copy "
COPY")

(defvar vcode_if "
IF")

(defvar vcode_pop "
POP")

(defvar vcode_else "
ELSE")

(defvar vcode_call "
CALL ")

(defvar vcode_cr "
")

(defvar vcode_space " ")

(defun nums-rec (nums)
  (if (null nums) (list ")") 
    (cons (format nil " ~a" (car nums)) 
	  (nums-rec (cdr nums)))))

(defun nums (nums)
  (cond ((null nums) (list " ()"))
	((null (cdr nums)) (list (format nil " ~a" (car nums))))
	(t (cons " (" (nums-rec nums)))))

(defun str-const-line (type value)
  (cond ((eql type 'nesl::segdes)
	 (append (cons vcode_cr (cons "CONST INT" (nums value)))
		 (list vcode_cr "MAKE_SEGDES")))
	((and (eql type 'char) 
	      (not (or (> (length value) 100)
		       (find #\  value) (find #\" value) 
		       (find #\\ value))))
	 (list (format nil "~%CONST CHAR ~s" (coerce value 'string))))
	(t
	 (let ((const (cond ((eql type 'nesl::int) "CONST INT")
			    ((eql type 'nesl::char) "CONST INT")
			    ((eql type 'nesl::float) "CONST FLOAT")
			    ((eql type 'nesl::bool) "CONST BOOL"))))
	   (cons vcode_cr 
		 (cons const (nums (mapcar #'CONVERT-ANYTHING value))))))))

(defun func-to-string (code definitions)
  (if (null code) nil
    (let* ((pline (car code)))
      (append 
       (cond ((eql (car pline) 'VCONST)
	      (str-const-line (second pline) 
			      (vcode-vector-data (third pline))))
	     ((eql (car pline) 'COPY)
	      (list (concatenate 'string
				 vcode_copy
				 (aref *intlist* (second pline))
				 (aref *intlist* (third pline)))))
	     ((eql (car pline) 'POP)
	      (list (concatenate 'string 
				 vcode_pop
				 (aref *intlist* (second pline))
				 (aref *intlist* (third pline)))))
	     ((eql (car pline) 'IF) (list vcode_if))
	     ((eql (car pline) 'ELSE) (list vcode_else))
	     ((eql (car pline) 'CALL)
	      (list vcode_call (string (get-full-name (car (second pline))
						      (cdr (second pline))
						      definitions))))
	     ((= (length pline) 1)
	      (list vcode_cr (string (first pline))))
	     ((= (length pline) 2)
	      (list vcode_cr (string (first pline)) 
		    vcode_space (string (second pline))))
	     (t nil))
       (func-to-string (cdr code) definitions)))))

(defun get-calls (code)
  (if (null code) nil
    (let ((rest (get-calls (cdr code))))
      (if (eql (first (car code)) 'call)
	  (cons (second (car code)) rest)
	rest))))

(defun make-string-code (funspec stypes code definitions)
  (let ((full-name (get-full-name funspec stypes definitions)))
    (cons (get-calls code)
	  (cons (format nil "~%FUNC ~a" full-name)
		       (func-to-string code definitions)))))

(defun write-func (name source-type definitions filename)
  (with-open-file (ofile (merge-pathnames filename) :direction :output)
    (write-code (get-fundef name definitions)
		source-type ofile definitions)))

