(in-package 'nesl-lisp)

(defun nesl-read-binding (binding over?)
  (when (or (not (listp binding))
	    (not (= (length binding) 2)))
    (nesl-error 
     "In the binding ~a of a ~a form.~%~
            The syntax for a binding is:  (pattern exp)." 
     binding (if over? "V. or OVER" "WITH")))
  `(,(nesl-read-exp (first binding)) ,(nesl-read-exp (second binding))))

(defun nesl-read-bindings (a over?)
  (when (not (listp a))
    (nesl-error
     "In the bindings ~a of a ~a form.~%~
            The syntax for a binding is:  (pattern exp)*." 
     a (if over? "V. or OVER" "WITH")))
  (if (null a) a
    (cons (nesl-read-binding (car a) over?)
	  (nesl-read-bindings (cdr a) over?))))
	    
(defun nesl-read-with (a)
  (when (or (not (= (length a) 3))
	    (not (listp (second a))))
    (nesl-error 
     "In the WITH expression ~%  ~a.~%~
      The syntax for WITH is:  (WITH (expbind*) exp)."
     a))
  `(with ,(nesl-read-bindings (second a) nil) ,(nesl-read-exp (third a))))

(defun nesl-read-over (a)
  (when (or (not (= (length a) 3))
	    (not (listp (second a)))
	    (not (> (length (second a)) 0)))
    (nesl-error 
     "In the OVER expression ~%  ~a.~%~
      The syntax for OVER is:  (OVER (expbind+) exp)."
     a))
  `(nesl::over ,(nesl-read-bindings (second a) t) 
	       ,(nesl-read-exp (third a))))

(defun nesl-read-list (a)
  (if (= (length a) 1) (nesl-read-exp (car a))
    `(nesl::pair ,(nesl-read-exp (car a)) ,(nesl-read-list (cdr a)))))

(defun nesl-read-func (a)
  (when (not (> (length a) 1))
    (nesl-error 
     "Error in the expression ~%  ~a.~%~
        Function calls require at least one argument."
     a))
  (list (car a) (nesl-read-list (cdr a))))

(defun nesl-read-if (a)
  (when (not (= (length a) 4))
    (nesl-error 
     "Error in the IF expression ~%  ~a.~%~
        The syntax for IF is:  (IF exp exp exp)."
     a))
  `(if ,(nesl-read-exp (second a))
       ,(nesl-read-exp (third a))
     ,(nesl-read-exp (fourth a))))

(defun nesl-read-pair (a)
  (when (not (= (length a) 3))
    (nesl-error 
     "Error in the expression ~%  ~a.~%~
        The syntax for PAIR is:  (PAIR exp exp)."
     a))
  `(nesl::pair ,(nesl-read-exp (second a))
	       ,(nesl-read-exp (third a))))
  
(defun nesl-read-exp (exp)
  (cond ((nesl-constant-p exp) exp)
	((symbolp exp) exp)
	((listp exp)
	 (let ((exp (convert-vexp exp)))
	   (cond ((eql (car exp) 'if)
		  (nesl-read-if exp))
		 ((eql (car exp) 'with)
		  (nesl-read-with exp))
		 ((eql (car exp) 'nesl::over)
		  (nesl-read-over exp))
		 ((eql (car exp) 'nesl::pair)
		  (nesl-read-pair exp))
		 (t (nesl-read-func exp)))))
	(t (error "In NESL-READ, Invalid expression, ~s." exp))))

(defun nesl-read-next (a)
  (if (and (listp a) (eql (car a) :primitive)) 
      a 
    (nesl-read-exp a)))

(defun nesl-read-top (a)
  (if (listp a)
      (cond ((eql (car a) 'nesl::poly-typecase)
	     `(,(first a) ,(second a) 
	       ,(nesl-read-next (third a))
	       ,(nesl-read-next (fourth a))
	       ,@(if (fifth a) (list (nesl-read-next (fifth a))) nil)))
	    ((eql (car a) 'nesl::base-typecase)
	     `(,(first a) ,(second a) 
	       ,@(mapcar #'(lambda (a) (list (first a) 
					     (nesl-read-next (second a))))
			 (cddr a))))
	    (t (nesl-read-next a)))
    (nesl-read-exp a)))

(defun defop-syntax-error (a)
  (nesl-error "Bad function definition: (DEFOP ~a ...).~%~
                   The syntax for DEFOP is:~%~%  ~
         (DEFOP (Ident Ident+) [! typespec]~%    exp)~%"
	      a))

(defun nesl-read-type-list (a)
  (if (= (length a) 1) (nesl-read-type-exp (car a))
    `(nesl::pair ,(nesl-read-type-exp (car a)) 
		 ,(nesl-read-type-list (cdr a)))))

(defun nesl-read-type-exp (exp)
  (cond ((symbolp exp) exp)
	((listp exp)
	 (cond ((eql (car exp) 'nesl::pair)
		`(nesl::pair ,(nesl-read-type-exp (second exp))
			     ,(nesl-read-type-exp (third exp))))
	       ((= (length exp) 1) exp)
	       (t `(,(car exp) ,(nesl-read-type-list (cdr exp))))))
	(t (error "In NESL-READ, Invalid expression, ~s." exp))))

(defun nesl-read-defop (a)
  (let* ((names (second a)))
    (when (or (not (listp names))
	      (not (> (length names) 1))
	      (not (symbolp (first names)))
	      (< (length a) 3))
      (defop-syntax-error names))
    (cond ((eql (third a) 'nesl::! )
	   (when (< (length a) 5) (defop-syntax-error names))
	   (let* ((type (simplify-type (fourth a)))
		  (stype (nesl-read-type-list (cdar type)))
		  (dtype (nesl-read-type-exp (caar type))))
	     `(nesl::defop ,(nesl-read-exp (second a)) 
			   ,(cons (list dtype stype) (cdr type))
			   ,(nesl-read-top (fifth a))
			   ,@(cddr (cdddr a)))))
	  (t
	   `(nesl::defop ,(nesl-read-exp (second a)) 
			 nil
			 ,(nesl-read-top (third a))
			 ,@(cdddr a))))))

(defun nesl-read-defrec (a)
  (let ((names (second a)))	       
    (when (or (not (listp names))
	      (not (plusp (length names)))
	      (not (symbolp (first names)))
	      (< (length a) 2))
      (nesl-error "Bad type definition: (DEFREC ~a ...).~%~
                   The syntax for DEFREC is:  ~
         (DEFREC (Ident typeexp*) typebind*)."
		  names))
    `(nesl::defrec ,(nesl-read-type-exp (second a)) ,@(cddr a))))

(defun nesl-read-toplevel (exp)
  (cond ((listp exp)
	 (cond ((eql (car exp) 'nesl::defop)
		(nesl-read-defop exp))
	       ((eql (car exp) 'nesl::defrec)
		(nesl-read-defrec exp))
	       ((eql (car exp) 'nesl::deftypeclass)
		exp)
	       ((eql (car exp) 'nesl::set)
		`(nesl::set ,(second exp) ,(nesl-read-exp (third exp))))
	       ((member (car exp) 
			'(nesl::describe 
			  nesl::redefinep nesl::debug nesl::cnesl nesl::lisp 
			  nesl::exit nesl::help nesl::verbose nesl::load
			  nesl::set_print_length nesl::bugs nesl::configuration
			  nesl::use_machine nesl::list_machines
			  nesl::set_memory_size nesl::cm_finger nesl::progn))
		exp)
	       (t (nesl-read-exp exp))))
	(t exp)))

