;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package :nesl-lisp) 

(defparameter *current-fundef* nil)
(defparameter *current-typedef* nil)

(defun trans-phase1 (variables body pflag)
  (if (and (listp body) (eql (car body) :primitive))
      (if pflag 
	  (cons :primitive (cons (list 'POP 1 0) (cdr body)))
	body)
    (free-body (conv-body variables (strip-exp body *definitions*) pflag))))

(defun phase1-trans-body (names body type pflag)      
  (let ((variables (cdr names)))
    (cond ((and (listp body) (eql (car body) 'nesl::base-typecase))
	   `(,(first body) ,(second body)
	     ,@(mapcar #'(lambda (a) 
			   (list (first a) 
				 (trans-phase1 variables (second a) pflag)))
		       (cddr body))))
	  ((and (listp body) (eql (car body) 'nesl::poly-typecase))
	   `(,(first body) ,(second body)
	     ,(trans-phase1 variables (third body) pflag)
	     ,(trans-phase1 variables (fourth body) pflag)
	     ,(trans-phase1 variables (or (fifth body)
					  (generate-default-pair-code 
					   names
					   type (second body)))
			    pflag)))
	  (t (trans-phase1 variables body pflag)))))

(defun special-body? (body)
  (and (listp body)
       (or (eql (car body) 'nesl::base-typecase)
	   (eql (car body) 'nesl::poly-typecase)
	   (eql (car body) :primitive))))

(defun parse-defop (arguments definitions)
  (let* ((names (first arguments))
	 (args (cdr arguments))
	 (keys (cddr args)))
    (make-defop names (first args) (second args)
		(get-keyword :type-check keys t)
		(get-keyword :documentation keys nil)
		(get-keyword :example keys nil)
		(get-keyword :redefine keys *redefine-default*)
		(get-keyword :shortdoc keys nil)
		(get-keyword :infix keys nil)
		definitions)))

(defun add-function (names type body pflag definitions
			   &key documentation shortdoc example redefine infix)
  (let* ((ntype (cons (conv-type (car type) pflag) (cdr type)))
	 (stype (strip-type ntype definitions))
	 (nnames (conv-names names pflag))
	 (olddef (get-fundef (car nnames) definitions)))
    (declare (special *redefine-default*))
    (when (and (not *redefine-default*) olddef (fundef-redefine olddef))
      (nesl-error "~a is a built in function, it cannot be redefined."
		  (car names)))
    (make-function-def nnames definitions
		       :documentation documentation
		       :shortdoc shortdoc
		       :type ntype
		       :codetype stype
		       :example example
		       :redefine redefine
		       :infix infix)
    (finalize-function-def (phase1-trans-body names body (car type) pflag)
			   definitions)))

(defun make-defop (names type body 
			 type-check documentation example redefine shortdoc
			 infix definitions)
  (let* ((*current-fundef* (car names))
	 (simplified-type type)
	 (check-valid-user-type 
	  (check-type-list simplified-type definitions))
	 (type-check (and type-check (not (special-body? body))))
	 (flattype (if type-check
		       (typecheck-op names simplified-type body definitions)
		     simplified-type)))
    (declare (special *current-fundef*) (ignore check-valid-user-type))
    (when (vprefixp (car names))
      (nesl-error 
       "You cannot define a function with a name that starts with v."))
    ;; Add parallel version
    (add-function names flattype body t definitions)
    ;; Add serial version
    (add-function names flattype body nil definitions
		  :documentation documentation
		  :shortdoc shortdoc
		  :example example
		  :redefine redefine
		  :infix infix)
    (values names (cons 'function flattype))))

(defun get-defrec-typebindings (list)
  (if (or (null list) (keywordp (car list)))
      nil
    (cons (car list) (get-defrec-typebindings (cdr list)))))

(defun parse-defrec (arguments definitions)
  (let* ((names (first arguments))
	 (typebindings (get-defrec-typebindings (cdr arguments)))
	 (keys (nthcdr (length typebindings) (cdr arguments))))
    (declare (special *redefine-default*))
    (make-defrec names typebindings
		 (get-keyword :documentation keys nil)
		 (get-keyword :redefine keys *redefine-default*)
		 definitions)))

(defun add-type (names type pflag definitions
		       &key documentation redefine)
  (add-type-def (car (conv-names names pflag) )
		(cons (conv-type (car type) pflag) (cdr type))
		definitions)
  (add-function names type '(:primitive) pflag definitions
		:documentation documentation
		:redefine redefine))

(defun make-args (count)
  (do ((i count (- i 1))
       (l nil (cons (intern (format nil "A~a" i)) l)))
      ((zerop i) l)))
	  
(defun make-defrec (names typebindings
			  documentation redefine
			  definitions)
  (let* ((name (car names))
	 (*current-typedef* name)
	 (simp-types (intern-type-list (cdr names)))
	 (check-valid-type (check-type-list (cons simp-types typebindings)
					    definitions))
	 (full-names (cons name (make-args (length (cdr names)))))
	 (return-type (cons name (mapcar #'first typebindings)))
	 (type (cons (cons return-type simp-types) typebindings))
	 (doc (or documentation
		  (format nil "Constructor for the record type ~a." name))))
    (declare (special *current-typedef*) (ignore check-valid-type))
    (add-type full-names type nil definitions
	      :documentation doc
	      :redefine redefine)
    (add-type full-names type t definitions)
    (values full-names (cons 'function type))))


#| *********************
The following generates code for polymorphic functions on pairs.
|# 

(defun generate-default-pair-code (names funtype typevar)
  ;;(print (list 'here names funtype typevar))
  (multiple-value-bind (binds args1 args2)
    (variable-bind-calls (second names) (second funtype) typevar)
    (let ((calls (list (list (car names) args1)
		       (list (car names) args2))))
      `(with ,binds ,(combine-code calls (car funtype) typevar)))))

(defun variable-bind-calls (args argtypes typevar)
  (if (listp args)
      (multiple-value-bind (bindsleft vleft1 vleft2)
	  (variable-bind-calls (second args) (second argtypes) typevar)
	(multiple-value-bind (bindsright vright1 vright2)
	  (variable-bind-calls (third args) (third argtypes) typevar)
	  (values (append bindsleft bindsright)
		  `(nesl::pair ,vleft1 ,vright1)
		  `(nesl::pair ,vleft2 ,vright2))))
    (variable-bind-call args argtypes typevar)))

(defun variable-bind-call (arg argtype typevar)
  (cond ((eql argtype typevar)
	 (let ((v1 (gensym)) 
	       (v2 (gensym)))
	   (values `(((nesl::pair ,v1 ,v2) ,arg))
		   v1 v2)))
	((and (listp argtype) (eql (car argtype) 'nesl::vector)
	      (eql (second argtype) typevar))
	 (let ((v1 (gensym))
	       (v2 (gensym))
	       (seg (gensym)))
	   (values `(((nesl::vector 
		    (nesl::pair ,seg (nesl::pair ,v1 ,v2))) ,arg))
		 `(nesl::vector (nesl::pair ,seg ,v1))
		 `(nesl::vector (nesl::pair ,seg ,v2)))))
	(t 
	 (values nil arg arg))))

(defun combine-code (calls return-type typevar)
  (cond ((eql return-type typevar)
	 (cons 'nesl::pair calls))
	((and (listp return-type) (eql (car return-type) 'nesl::vector)
	      (eql (second return-type) typevar))
	 (let ((g1 (gensym)) (g2 (gensym)))
	   `(with (((nesl::vector (nesl::pair seg ,g1)) ,(first calls))
		   ((nesl::vector (nesl::pair seg ,g2)) ,(second calls)))
	      (nesl::vector (nesl::pair seg (nesl::pair ,g1 ,g2))))))
	(t (nesl-error "Default poly-typecase can't return the type ~a." 
		       return-type))))
