;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; CNESL mode for editing, interpreting.
;;;
;;; CNESL formatting by Tom Sheffler.
;;; Built on initial version by Guy Blelloch.
;;; Feb 1993
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; HISTORY
;;;
;;; V 1.1: Tom Sheffler, Feb 1993
;;; 	Automatic function type annotation of function definition.
;;; 	Pretty print in a popup window. 
;;;
;;; V 1.0: Tom Sheffler, Feb 1993
;;; 	Basic CNESL indentation code.
;;; 	Comment syntax.
;;;
;;; V 0.9: Guy Blelloch created, basic send-defun
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq auto-mode-alist (append '(("\\.cnesl$" . cnesl-mode)) auto-mode-alist))

;;; Mode Variables
(defconst cnesl-indent-level 4
  "*Indentation to be used inside of Cnesl blocks or arrays")

(defconst cnesl-tab-width 4
  "*Tab stop width for Cnesl mode")

(defun cnesl-make-tabs (stop)
  (and (< stop 132) (cons stop (cnesl-make-tabs (+ stop cnesl-tab-width)))))

(defconst cnesl-tab-stop-list (cnesl-make-tabs cnesl-tab-width)
  "*Tab stop list for cnesl mode")

(defvar cnesl-tab-always-indent t
  "TAB keys means always re-indent current line rather than insert TAB.")

(defvar nesl-process "nesl"
  "String name of process that is inferior Nesl")

(defvar cnesl-mode-hook nil
  "User-definable function hook called on entry to cnesl-mode")

(defvar cnesl-mode-map nil
  "Key map for CNESL mode.")

(defvar cnesl-mode-syntax-table nil
  "Syntax table in use in CNesl-mode buffers.")
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(if cnesl-mode-map			; set once upon loading
    ()
  (setq cnesl-mode-map (make-sparse-keymap))
  (define-key cnesl-mode-map "\e\C-x"	'cnesl-send-defun)
  (define-key cnesl-mode-map "\e\C-a"	'beginning-of-cnesl-function)
  (define-key cnesl-mode-map "\e\C-e"	'end-of-cnesl-function)
  (define-key cnesl-mode-map "\t"   	'cnesl-indent-command)
  (define-key cnesl-mode-map "\C-cp" 	'cnesl-pretty-print-defun)
  (define-key cnesl-mode-map "\C-ct" 	'cnesl-insert-function-type)
  )

(if cnesl-mode-syntax-table		; set once upon loading
    ()
  (setq cnesl-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?_ "w" cnesl-mode-syntax-table)
  (modify-syntax-entry ?\\ "\\" cnesl-mode-syntax-table)
  (modify-syntax-entry ?/ "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?* "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?+ "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?- "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?= "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?% "$" cnesl-mode-syntax-table)
  (modify-syntax-entry ?< "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?> "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?& "." cnesl-mode-syntax-table)
  (modify-syntax-entry ?| "." cnesl-mode-syntax-table))

(defun cnesl-mode-variables ()
  "Function that sets local CNESL mode variables."
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'comment-column)
  (make-local-variable 'comment-indent-hook)
  (make-local-variable 'indent-line-function)
  (make-local-variable 'tab-stop-list)
  (setq comment-start "% "
	comment-end " %"
	comment-start-skip "%<? *"	; Regexp to skip until comment body
	comment-column 40
    	comment-indent-hook 'cnesl-comment-indent
	indent-line-function 'cnesl-indent-line
	tab-stop-list cnesl-tab-stop-list
	)
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun cnesl-indent-command (&optional whole-exp)
  (interactive "P")
  "Indent current line as Cnesl code, or in some cases insert a tab character.
If cnesl-tab-always-indent is non-nil (the default), always indent current line.
Otherwise, indent the current line only if point is at the left margin
or in the line's indentation; otherwise insert a tab.

A numeric argument, regardless of its value,
means indent rigidly all the lines of the expression starting after point
so that this line becomes properly indented.
The relative indentation among the lines of the expression are preserved."
  (if whole-exp
      ;; If arg, always indent this line as CNESL
      ;; and shift remaining lines of expression the same amount.
      (let ((shift-amt (cnesl-indent-line))
	    beg end)
	(save-excursion
	  (if cnesl-tab-always-indent
	      (beginning-of-line))
	  (setq beg (point))
	  (forward-sexp 1)
	  (setq end (point))
	  (goto-char beg)
	  (forward-line 1)
	  (setq beg (point)))
	(if (> end beg)
	    (indent-code-rigidly beg end shift-amt "%<")))
    (if (and (not cnesl-tab-always-indent)
	     (save-excursion
	       (skip-chars-backward " \t")
	       (not (bolp))))
	(insert-tab)
      (cnesl-indent-line))))

(defun cnesl-indent-line ()
  "Indent current line as CNesl code.
Return the amount the indentation changed by."
  (let ((indent (calculate-cnesl-indent nil))
	beg shift-amt
	(case-fold-search nil)
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (cond ((eq indent nil)
	   (setq indent (current-indentation)))
	  ((eq indent t)
	   (setq indent (calculate-cnesl-indent-within-comment)))
	  (t
	   (skip-chars-forward " \t")
	   ;; Special cases for alignment.
	   (cond ((and (looking-at "else\\b")
		       (not (looking-at "else\\s_")))
		  (setq indent (save-excursion
				 (cnesl-backward-to-start-of-if)
				 (current-indentation))))
		 ((and (looking-at "then\\b")
		       (not (looking-at "then\\s_")))
		  (setq indent (save-excursion
				 (cnesl-backward-to-start-of-if)
				 (current-indentation))))
		 ((looking-at "in\\b")
		  (setq indent (save-excursion
				 (cnesl-backward-to-start-of-let)
				 (current-indentation)))))))
    (skip-chars-forward " \t")
    (setq shift-amt (- indent (current-column)))
    (if (zerop shift-amt)
	(if (> (- (point-max) pos) (point))
	    (goto-char (- (point-max) pos)))
      (delete-region beg (point))
      (indent-to indent)
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (if (> (- (point-max) pos) (point))
	  (goto-char (- (point-max) pos))))
    shift-amt))


(defun calculate-cnesl-indent (&optional parse-start)
  "Return appropriate indentation for current line as C code.
In usual case returns an integer: the column to indent to.
Returns nil if line starts inside a string, t if in a comment."
  (save-excursion
    (beginning-of-line)
    (let ((indent-point (point))
	  (case-fold-search nil)
	  state
	  containing-sexp)
      (if parse-start
	  (goto-char parse-start)
	(beginning-of-cnesl-function))
      (while (< (point) indent-point)
	;; TOM: move through at equal levels until we reach POINT
	(setq parse-start (point))
	(setq state (parse-partial-sexp (point) indent-point 0)))

      ;; TOM: innermost containing list: NIL if none
      (setq containing-sexp (car (cdr state)))

;;; TOM: for debugging/understanding!!
;;;      (save-excursion
;;;	(set-buffer "*scratch*")
;;;	(insert  (prin1-to-string state))
;;;	(insert (prin1-to-string containing-sexp))
;;;	(insert "\n")
;;;	)

      ;; TOM: if in comment, or in STRING just return nil.
      ;; TOM: comments aren't parsed right because % is given $ status
      (cond ((or (nth 3 state) (nth 4 state))
	     ;; return nil or t if should not change this line
	     ;; TOM: inside COMMENT or STRING ??
	     (nth 4 state))
	    
	    ;; TOM: we're inside an expression.
	    (containing-sexp 
	     ;; line is expression, not statement:
	     ;; indent to just after the surrounding open.
	     (goto-char (1+ containing-sexp))
	     (current-column))

	    ;; else, PUNT: just line it up with previous line
	    (t
	     (beginning-of-line)
	     (delete-horizontal-space)
	     ;; Check for LEFT-MARGIN constructs.
	     (if (or
		  (looking-at "%<<") ; Left margin comment?
		  (looking-at "function")
		  (looking-at "datatype")
		  (looking-at "\\$")
		  )
		 0			; no indentation
	       ;; ELSE, try to be more clever
	       (save-excursion
		 ;; Look at previous line.
		 (cnesl-backward-to-noncomment 1)
		 (let ((retval (cnesl-is-continuation-line)))
		   ;; if non-nil, then retval is either the keyword of
		   ;; previous line, or simply 't.
		   (if retval
		       (cond
			((equal retval '("let"))
			 (+ (current-indentation) cnesl-indent-level))
			((equal retval '("in"))
			 (+ (current-indentation) cnesl-indent-level))
			((equal retval '("if"))
			 (+ (current-indentation) cnesl-indent-level))
			((equal retval '("then"))
			 (+ (current-indentation) cnesl-indent-level))
			((equal retval '("else"))
			 (+ (current-indentation) cnesl-indent-level))
			(t
			 (progn
			   ;; This line is a continuation of a stmt.
			   ;; Find beginning of this stmt and align to it.
			   (cnesl-backward-to-start-of-continued-exp 1)
			   (current-indentation))))
		     ;; ELSE - prev line is NOT a continuation
		     ;; Align next line to this one
		     (current-indentation)
		     )))
	       ))))))
	       
(defun calculate-cnesl-indent-within-comment ()
  "Return the indentation amount for line, assuming that
the current line is to be regarded as part of a block comment."
  (let (end star-start)
    (save-excursion
      (save-excursion
	(set-buffer "*scratch*")
	(insert  "in INDENT-WITHIN-COMMENT")
	(insert "\n")
	)
      (beginning-of-line)
      (skip-chars-forward " \t")
      (skip-chars-backward " \t\n")	; previous line
      (setq end (point))		; stop here
      (beginning-of-line)
      (skip-chars-forward " \t")
      (current-column))))

(defun cnesl-backward-to-noncomment (lim)
  (let (opoint stop)
    (while (not stop)
      (skip-chars-backward " \t\n\f" lim)
      (setq opoint (point))
      (beginning-of-line)
      (if (and (search-forward "%" opoint 'move)
	       (< lim (point)))
	  (forward-char -1)
	(setq stop t)))))

(defun cnesl-backward-to-start-of-continued-exp (lim)
  (if (memq (preceding-char) '(?\) ?\] ?\}))
      (forward-sexp -1))
  (beginning-of-line)
  (if (<= (point) lim)
      (goto-char (1+ lim)))
  (skip-chars-forward " \t"))

(defun cnesl-backward-to-start-of-if (&optional limit)
  "Move to the start of the last ``unbalanced'' if."
  (or limit (setq limit (save-excursion (beginning-of-cnesl-function)(point))))
  (let ((if-level 1)
	(case-fold-search nil))
    (while (not (zerop if-level))
      (backward-sexp 1)
      (cond ((looking-at "else\\b")
	     (setq if-level (1+ if-level)))
	    ((looking-at "if\\b")
	     (setq if-level (1- if-level)))
	    ((< (point) limit)
	     (setq if-level 0)
	     (goto-char limit))))))

(defun cnesl-backward-to-start-of-let (&optional limit)
  "Move to the start of the last ``unbalanced'' let."
  (or limit (setq limit (save-excursion (beginning-of-cnesl-function) (point))))
  (let ((if-level 1)
	(case-fold-search nil))
    (while (not (zerop if-level))
      (backward-sexp 1)
      (cond ((looking-at "in\\b")
	     (setq if-level (1+ if-level)))
	    ((looking-at "let\\b")
	     (setq if-level (1- if-level)))
	    ((< (point) limit)
	     (setq if-level 0)
	     (goto-char limit))))))

(defun cnesl-mode ()
  "Major mode for editing CNESL code."
  (interactive)
  (kill-all-local-variables)
  (use-local-map cnesl-mode-map)
  (setq major-mode 'cnesl-mode)
  (setq mode-name "Cnesl")
  (set-syntax-table cnesl-mode-syntax-table)
  (cnesl-mode-variables)
  (run-hooks cnesl-mode-hook))

;;; comments of the form "%<" stay where they are,
;;; single "%" comments go to the right margin.
(defun cnesl-comment-indent ()
  (if (looking-at "%<")
      (current-column)
    (skip-chars-backward " \t")
    (max (if (bolp) 0 (1+ (current-column)))
	 comment-column)))

;;; Are we on a continuation line?
;;; Return 't or 'nil, or if keyword, return it
(defun cnesl-is-continuation-line ()
  (let* ((ch (preceding-char))
	 (ch-syntax (char-syntax ch)))
    (if (eq ch-syntax ?w)
	;; If prev char is a character, see if reserved word, return it
	(assoc (buffer-substring
		(progn (forward-word -1) (point))
		(progn (forward-word 1) (point)))
	       '(("let") ("in") ("if") ("then") ("else")))
      
      ;; We're ONLY on a continuation line if this line
      ;; is inside a paren expression.
      (save-excursion
	(beginning-of-line)
	(let ((indent-point (point))
	      (case-fold-search nil)
	      state
	      containing-sexp)
	  (beginning-of-cnesl-function)
	  (while (< (point) indent-point)
	    ;; TOM: move through at equal levels until we reach POINT
	    (setq parse-start (point))
	    (setq state (parse-partial-sexp (point) indent-point 0)))
	  (setq containing-sexp (car (cdr state)))
	  containing-sexp)))))
    
(defun cnesl-backward-to-start-of-continued-exp (lim)
;;  (if (memq (preceding-char) '( ?\) ?\] ))
;;      (forward-sexp -1))
  (while (cnesl-is-continuation-line)
    (end-of-line 0))
  (beginning-of-line)
  (if (<= (point) lim)
      (goto-char (1+ lim)))
  (skip-chars-forward " \t"))

(defun beginning-of-cnesl-function (&optional arg)
  "Move backward to next beginning-of-function.
With argument, do this that many times.
Returns t unless search stops due to end of buffer."
  (interactive "p")
  (and arg (< arg 0) (forward-char 1))
  (and (re-search-backward "^function" nil 'move (or arg 1))
       (progn (beginning-of-line) t)))

(defun end-of-cnesl-function (&optional arg)
  "Move backward to next beginning-of-defun.
With argument, do this that many times.
Returns t unless search stops due to end of buffer."
  (interactive "p")
  (and arg (< arg 0) (forward-char 1))
  (and (re-search-forward "\\$" nil 'move (or arg 1))
       (progn (end-of-line) t)))

(defun cnesl-send-defun (display-flag)
  "Send the current defun to the Nesl process made by M-x run-nesl.
With argument, force redisplay and scrolling of the *nesl* buffer.
Variable `inferior-cnesl-load-command' controls formatting of
the `load' form that is set to the Lisp process."
  (interactive "P")
  (or (get-process nesl-process)
      (error "No current lisp process"))
  (save-excursion
   (end-of-cnesl-function)
   (let ((end (point))
	 (filename (format "/tmp/emlisp%d.cnesl" 
			   (process-id (get-process nesl-process)))))
     (beginning-of-cnesl-function)
     (write-region (point) end filename nil 'nomessage)
     (process-send-string
      nesl-process (format inferior-cnesl-load-command filename)))
   (if display-flag
       (cnesl-redisplay-process-window))
   ))


;;; After sending an expression to the CNESL sub-process, move point to
;;; end of output, and display the process buffer window.
(defun cnesl-redisplay-process-window ()
  (let* ((process (get-process nesl-process))
	 (buffer (process-buffer process))
	 (w (or (get-buffer-window buffer) (display-buffer buffer)))
	 (height (window-height w))
	 (end))
    (save-excursion
      (set-buffer buffer)
      (setq end (point-max))
      (while (progn
	       (accept-process-output process)
	       (goto-char (point-max))
	       (beginning-of-line)
	       (or (= (point-max) end)
		   (not (looking-at inferior-lisp-prompt)))))
      (setq end (point-max))
      (vertical-motion (- 4 height))
      (set-window-start w (point)))
    (set-window-point w end)))

;;; After sending an expression to the CNESL sub-process, 
;;; return interpreter output as a string.
;;; Make sure it doesn't change the windows around.
(defun cnesl-get-process-output ()
  (save-window-excursion
    (let* ((process (get-process nesl-process))
	   (buffer (process-buffer process))
	   (w (or (get-buffer-window buffer) (display-buffer buffer)))
	   (begin)
	   (end))
      (set-buffer buffer)
      (setq begin (point-max))
      (setq end (point-max))
      (while (progn
	       (accept-process-output process)
	       (goto-char (point-max))
	       (beginning-of-line)
	       (or (= (point-max) end)
		   (not (looking-at inferior-lisp-prompt)))))
      (setq end (point))
      (set-window-point w (point-max))
      (buffer-substring begin end))))


;;; Display the string given in the CNESL pop-up window!
(defun cnesl-popup-window (s)
  (let* (
	 (cbuf (current-buffer))
	 (buffer (get-buffer-create "*cnesl-popup*"))
	 (w (or (get-buffer-window buffer) (display-buffer buffer)))
;;	 (height (window-height w))
	 )
    (set-buffer buffer)
    (erase-buffer)
    (insert s)
    (set-buffer cbuf)
    ))
    
;;; Send a CNESL function to the pretty printer.
;;; Display it in the POPUP-WINDOW, or bring up CNESL process window
;;; if a prefix arg is given.
(defun cnesl-pretty-print-defun (display-flag)
  "Send the current function to the CNESL process and pretty-print it
in a popup window.  Optional prefix arg means to bring up the CNESL
process window instead of a popup."
  (interactive "P")
  (save-excursion
    (let* (
	   (begin (save-excursion
		    (beginning-of-cnesl-function)
		    (point)))
	   (end (save-excursion
		  (end-of-cnesl-function)
		  (forward-word -1)	; skip back before $
		  (point)))
	   (fn (buffer-substring begin end)) ; extract function def
	   )
      (process-send-string nesl-process (format "pprint(%s) $\n" fn))))
  (if display-flag
      (cnesl-redisplay-process-window)
    (cnesl-popup-window
     (cnesl-get-process-output))))

  
;; Send a CNESL function to be evaluated and find the TYPE returned.
;; Modify the current function so that the type is inserted.
(defun cnesl-insert-function-type ()
  (interactive)
  (let* (
	 (bof)				; beginning of function
	 (eof				; end of function
	  (save-excursion
	    (end-of-cnesl-function)
	    ;; skip back before $
	    (forward-word -1)
	    (point)))
	 (eoa)				; end of arg list
	 (eot)				; end of type information
	 )
    (save-excursion			; find various points in function
	(beginning-of-cnesl-function)
	(setq bof (point))
	(forward-word 2)		; skip "function NAME"
	(forward-sexp 1)		; skip arg list
	(setq eoa (point))
	(skip-chars-forward " \t")
	(if (looking-at "=")
	    (setq eot (point))
	  (progn 
	    (search-forward "=" eof)
	    (forward-char -1)		; before the =
	    (setq eot (point))))
	)
    (process-send-string		; send the DEFUN to CNESL interpreter
     nesl-process
     (format "%s $\n" (buffer-substring bof eof)))
			 
    (let* (
	  (output (cnesl-get-process-output))
	  (type   (cnesl-parse-defun-output output))
	  )
      (if type				; if non-nil, replace type clause
	  (progn
	    (goto-char eoa)
	    (delete-region eoa eot)	; delete the old
	    (insert "\n" type " ")	; insert the new
	    )))
    ))

;; If there is no error, return the type portion of a defun or NIL
;; if there was an error
(defun cnesl-parse-defun-output (s)
  (save-window-excursion
    (set-buffer (get-buffer-create "*cnesl junk buffer*"))
    (set-syntax-table cnesl-mode-syntax-table) ; to parse words like CNESL
    (erase-buffer)
    (insert s)				; write it here
    (beginning-of-buffer)		; back to the top
    (if (search-forward "Error in Function" (point-max) t)
	nil				; error condition
      (progn
	(beginning-of-buffer)
	(forward-word 1)		; skip fn name
	(forward-sexp 1)		; skip arts
	(buffer-substring (point)
			  (save-excursion
			    (end-of-buffer)
			    (skip-chars-backward " \n")
			    (point)))
	))))
      
	    
  
