;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

;;; Interface code between vcode and lisp.
;;; (First pass written by Timothy Freeman)

(in-package :nesl-lisp) 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DEFINING F TO BE NIL
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun booleanp (x) (not (null (member x '(t nil f)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DEFINITION OF THE MACRO CHARACTERS #v AND #u
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (compile eval load)
  (defparameter vcode-vector-letter #\u
    "The letter used for low-level vcode-vectors.")

  (defparameter vcode-sequence-letter #\v
    "The letter used for higher-level nestable vcode-vectors.")

  (set-dispatch-macro-character #\# vcode-vector-letter 'read-vcode-vector)

  (set-dispatch-macro-character #\# vcode-sequence-letter 'read-vcode-sequence))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DEFINITION OF STRUCTURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; We use structures vcode-vector, vcode-tuple, and vcode-record
;;; instead of simple lists so we can figure out a unique vcode
;;; type for each lisp data structure.  
(defstruct (vcode-vector (:print-function print-vcode-vector))
  type
  data)

(defstruct (nested-sequence (:print-function print-nested-sequence)) data type)

(defstruct (nesl-struct (:print-function print-nesl-struct)) data type)

(defun print-vcode-vector (self stream depth)
  (declare (ignore depth))
  (if (and (= (length (vcode-vector-data self)) 1)
	   (not (eql (vcode-vector-type self) 'nesl::vector)))
      (format stream "~s" (elt (vcode-vector-data self) 0))
      (format stream "#~a.~s~:s"
	      vcode-vector-letter
	      (vcode-vector-type self)
	      (coerce (vcode-vector-data self) 'list))))

(defparameter *max-print-length* 100)

(defun print-nested-sequence (self stream depth)
  (declare (ignore depth))
  (let ((data (nested-sequence-data self))
	(type (nested-sequence-type self)))
    (cond ((eql type 'char)
	   (format stream "~s" (coerce data 'string)))
	  ((= (length data) 0)
	   (if *cnesl-syntax*
	       (format stream "[]" vcode-sequence-letter)
	     (format stream "#~a()" vcode-sequence-letter)))
	  ((<= (length data) *max-print-length*)
	   (if *cnesl-syntax*
	       (format stream "[~s~{,~s~}]" (car data) (cdr data))
	     (format stream "#~a~s" vcode-sequence-letter data)))
	  (t 
	   (let ((data (subseq data 0 *max-print-length*)))
	     (if *cnesl-syntax*
		 (format stream "[~s~{,~s~},...]" (car data) (cdr data))
	       (format stream "#~a(~{~s ~}...)" vcode-sequence-letter)))))))

(defun print-nesl-struct (self stream depth)
  (declare (ignore depth))
  (let ((data (nesl-struct-data self))
	(type (nesl-struct-type self)))
    (cond ((not *cnesl-syntax*)
	   (format stream "(~s~{ ~s~})" type data))
	  ((eql type 'nesl::pair)
	   (format stream "(~s,~s)" (first data) (second data)))
	  (t 
	   (format stream "~s(~s~{,~s~})" type (car data) (cdr data))))))

(defun nesl-floatp (val)
  (or (floatp val) (eql val 'nesl::infinity)))

(defparameter *typecheck-map*
  '((int . integerp) (bool . booleanp) (nesl::float . nesl-floatp) 
    (char . characterp) (nesl::segdes . integerp)
    (nesl::stream . integerp)))

(defun make-vcode-vector-safely (type data)
  (let* ((itype (or type 
		   (if (= (length data) 0)
		       (nesl-error "Can't determine the type of empty vector.")
		     (typeof-scalar (car data)))))
	(cfunc (cdr (assoc itype *typecheck-map*))))
    (when (not cfunc)
      (nesl-error "~a is an invalid type for a constant vector." itype))
    (when (not (every cfunc data))
      (nesl-error "Inhomogeneous values in a constant vector."))
    (make-vcode-vector :type itype :data data)))

(defun make-vcode-vector-semi-safely (type data)
  (let ((cfunc (cdr (assoc type *typecheck-map*))))
    (when (not cfunc)
      (nesl-error "~a is an invalid type for a constant vector." type))
    (when (not (or (null data) (funcall cfunc (car data))))
      (nesl-error "Bad value in a result vector."))
    (make-vcode-vector :type type :data data)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; STUFF FOR READING VECTORS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defparameter *sequence-reader-top-level* t)

(defun read-type (head stream)
  (cond ((eql head 'nesl::v.)
	 (list `nesl::vector (intern-type (read stream t nil t))))
	((and (symbolp head) (vprefixp head))
	 (list 'nesl::vector (read-type (vpostfix head) stream)))
	(t (intern-type head))))

(defun read-vcode-vector (stream subchar arg)
  (declare (ignore subchar arg))
  (if *read-suppress* nil
    (let ((type (cond ((char= (peek-char nil stream) #\.)
		       (read-char stream t nil t)
		       (read stream t nil t))
		      (t nil))))
      (make-vcode-vector-safely type (read stream t nil t)))))

(defun read-vcode-sequence (stream subchar arg)
  (declare (ignore subchar arg))
  (if *read-suppress* nil
    (if *sequence-reader-top-level*
	(let ((*sequence-reader-top-level* nil))
	  (declare (special *sequence-reader-top-level*))
	  (flatten-nesl-type (sequence-reader stream)))
      (sequence-reader stream))))

(defun sequence-reader (stream)
  (cond ((char= (peek-char nil stream) #\.)
	 (read-char stream t nil t)
	 (make-nested-sequence 
	  :type (expand-type (read-type (read stream t nil t) stream)
			     *definitions*)
	  :data nil))
	(t (make-nested-sequence :type nil :data (read stream t nil t)))))

(defun typeof-scalar (prim-val)
  (cond ((booleanp prim-val) 'bool)
	((integerp prim-val) 'int)
	((floatp prim-val) 'nesl::float)
	((characterp prim-val) 'char)
	(t (nesl-error "Invalid type ~a in a NESL constant." 
		       (type-of prim-val)))))

(defun typeof-vector (prim-vect)
  (cond ((nested-sequence-p prim-vect)
	 `(nesl::vector 
	   (nesl::pair
	    nesl::segdes
	    ,(or (nested-sequence-type prim-vect) 
	      (if (= (length (nested-sequence-data prim-vect)) 0)
		  (nesl-error "Can't figure out the type of an empty vector.")
		(typeof-vector (car (nested-sequence-data prim-vect))))))))
	((stringp prim-vect) '(nesl::vector (nesl::pair nesl::segdes char)))
	((listp prim-vect) 
	 (cons (car prim-vect) 
	       (mapcar 'typeof-vector (cdr prim-vect))))
	(t (typeof-scalar prim-vect))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; STUFF FOR FLATTENING A VECTOR
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun flatten-slot (list)
  (if (not list) nil
    (append (car list) (flatten-slot (cdr list)))))

(defun flatten-list (prim-vect type)
  (if (not type)
      (if (not (every #'null prim-vect))
	  (nesl-error "Inhomogeneous structure values in a constant vector.")
	nil)
    (cons (flatten-exp (mapcar #'car prim-vect) (car type))
	  (flatten-list (mapcar #'cdr prim-vect) (cdr type)))))

(defun flatten-exp (prim-vect type)
  (cond ((atom type)
	 (make-vcode-vector-safely type prim-vect))
	((eql (car type) 'nesl::vector)
	 (let ((subvects 
		(mapcar 
		 #'(lambda (a)
		     (cond ((stringp a) (coerce a 'list))
			   ((nested-sequence-p a)
			    (nested-sequence-data a))
			   (t (nesl-error 
			       "Inhomogeneous types in a constant vector."))))
		 prim-vect)))
	   (list 'nesl::vector 
		 (list 'nesl::pair
		       (make-vcode-vector-safely 
			'nesl::segdes (mapcar #'length subvects))
		       (flatten-exp (flatten-slot subvects) 
				    (third (second type)))))))
	((listp type)
	 (cons (car type)
	       (flatten-list (mapcar #'cdr prim-vect) (cdr type))))))

(defun flatten-nesl-type (val)
  (flatten-exp (list val) (typeof-vector val)))

;;; This returns the type if the object is a nesl constant
;;; Otherwise it returns nil
(defun nesl-constant-p (val)
   (cond ((vcode-vector-p val) (vcode-vector-type val))
	 ((booleanp val) 'bool)
	 ((integerp val) 'int)
	 ((floatp val) 'nesl::float)
	 ((characterp val) 'char)
	 ((stringp val) '(nesl::vector char))))

;;; This coerces a NESL constant into a vcode-vector
;;; It will fail if passed anything other than a nesl constant
(defun coerce-nesl-constant (val)
  (if (vcode-vector-p val) val
    (flatten-nesl-type val)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; PRINTING NESL VECTORS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defparameter *print-vcode-sequences* t
  "Whether to print things of type (vector foo) as #v(...).")

(defun partition-vector (lengths values)
  (if lengths
      (if (> (car lengths) (length values))
	  (nesl-error "Lengths don't match in a Nesl structure." nil)
	(cons (subseq values 0 (car lengths))
	      (partition-vector (cdr lengths) (subseq values (car lengths)))))
    (if values (nesl-error "Lengths don't match in a Nesl structure." nil))))

(defun zipcons (a b)
  (if (= (length a) 0)
      (if (= (length b) 0)
	  nil
	(nesl-error "Lengths don't match in a Nesl structure."))
    (if (= (length b) 0)
	(nesl-error "Lengths don't match in a Nesl structure.")
      (cons (cons (car a) (car b)) (zipcons (cdr a) (cdr b))))))

(defun nest-constant-list (list)
  (multiple-value-bind (car-result car-type) (nest-constant-exp (car list))
    (if (= (length list) 1)
	(values (mapcar #'list car-result) (list car-type))
      (multiple-value-bind (cdr-result cdr-type) 
          (nest-constant-list (cdr list))
        (values (zipcons car-result cdr-result) 
		(cons car-type cdr-type))))))


(defun nest-constant-exp (data)
  (cond ((vcode-vector-p data) 
	 (values (vcode-vector-data data) (vcode-vector-type data)))
	((and (listp data) (eql (car data) 'nesl::vector))
	 (let ((segdes (second (second data)))
	       (subdata (third (second data))))
	   (multiple-value-bind (subvals type) (nest-constant-exp subdata)
	      (values 
	       (mapcar #'(lambda (a) 
			   (make-nested-sequence :data a :type type))
		       (partition-vector (vcode-vector-data segdes) 
					 subvals))
	       `(nesl::vector (nesl::pair nesl::segdes ,type))))))
	((and (listp data) (> (length data) 1))
	 (multiple-value-bind (result type) (nest-constant-list (cdr data))
	   (values (mapcar #'(lambda (a) 
			       (make-nesl-struct :type (car data) :data a))
			   result)
		   (cons (car data) type))))
	((and (listp data) (= (length data) 1))
	 (nesl-error "NESL currently cannot print empty records."))
	(t
	 (nesl-error "Bad constant in nest-constant-exp."))))

(defun nest-constant (data)
  (if (vcode-vector-p data)
      data
    (car (nest-constant-exp data))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; GROUPING DATA
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun group-data-exp (type data)
  (cond ((atom type) 
	 (cons (make-vcode-vector-semi-safely
		type
		(if (eql type 'char)
		    (mapcar #'code-char (car data))
		  (car data)))
	       (cdr data)))
	(t
	 (let ((grouped-list (group-data-list (cdr type) data)))
	   (cons (cons (car type) (car grouped-list))
		 (cdr grouped-list))))))

(defun group-data-list (type data)
  (if type
      (let* ((head (group-data-exp (car type) data))
	     (tail (group-data-list (cdr type) (cdr head))))
	(cons (cons (car head) (car tail))
	      (cdr tail)))
    (cons nil data)))

(defun group-data (data type definitions)
  (car (group-data-exp (expand-type type definitions) data)))

