;;;
;;; Copyright (c) 1992 Carnegie Mellon University 
;;;                    SCAL project: Guy Blelloch, Siddhartha Chatterjee,
;;;                                  Jonathan Hardwick, Jay Sipelstein,
;;;                                  Marco Zagha
;;; All Rights Reserved.
;;;
;;; Permission to use, copy, modify and distribute this software and its
;;; documentation is hereby granted, provided that both the copyright
;;; notice and this permission notice appear in all copies of the
;;; software, derivative works or modified versions, and any portions
;;; thereof, and that both notices appear in supporting documentation.
;;;
;;; CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
;;; CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
;;; ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
;;;
;;; The SCAL project requests users of this software to return to 
;;;
;;;  Guy Blelloch				guy.blelloch@cs.cmu.edu
;;;  School of Computer Science
;;;  Carnegie Mellon University
;;;  5000 Forbes Ave.
;;;  Pittsburgh PA 15213-3890
;;;
;;; any improvements or extensions that they make and grant Carnegie Mellon
;;; the rights to redistribute these changes.
;;;

(in-package 'nesl-lisp)

(defun type-< (type1 type2 definitions)
  (if (eql type2 'nesl::any) 
      t
    (member type1 (subtypes type2 definitions))))

(defun variable-type? (type definitions)
  (or (eql type 'nesl::any)
      (> (length (subtypes type definitions)) 1)))

(defun variable-subtypestype? (type definitions)
  (or (eql type 'nesl::any)
      (subtypes type definitions)))

(defparameter *allow-type-vars* nil)

(defun parse-type-exp (type tstack sum-func prim-func list-func 
			    zero definitions)
  (cond ((null type) zero)
	((and (atom type) (primitive-type? type definitions))
	 (funcall prim-func type))
	((and (atom type) (assoc type tstack))
	 (cdr (assoc type tstack)))
	((listp type)
	 (let ((typedef (get-typedef (car type) definitions)))
	   (if (not typedef)
	       (nesl-error 
		"No record-type definition for ~a in type expression ~a."
		(car type) type)
	     (let* ((type-dec (typedef-type typedef))
		    (nstack
		     (mapcar #'(lambda (subtype var) 
				 (cons (car var) 
				       (parse-type-exp 
					subtype tstack sum-func prim-func
					list-func zero definitions)))
			     (cdr type)
			     (cdr type-dec))))
	       (funcall list-func 
		(car type)
		(parse-type-list (cdr (car type-dec)) nstack sum-func 
				 prim-func list-func zero definitions))))))
	(t (if *allow-type-vars*
	       (funcall prim-func type)
	     (nesl-error "No base-type definition for ~a." type)))))

(defun parse-type-list (type tstack sum-func prim-func list-func zero defs)
  (if type
      (funcall sum-func 
	       (parse-type-exp (car type) tstack sum-func 
			       prim-func list-func zero defs)
	       (parse-type-list (cdr type) tstack sum-func 
				prim-func list-func zero defs))
    zero))

(defun flatten-type (type definitions)
  (parse-type-exp type nil #'append #'(lambda (a) (list a))
		  #'(lambda (a b) (declare (ignore a)) b) nil definitions))

;;(defun l-from-type (type definitions)
;;  (parse-type-exp type nil #'+ #'(lambda (a) (declare (ignore a)) 1) 
;;		  #'(lambda (a b) (declare (ignore a)) b) 0 definitions))

;;(defun l-from-type-list (type definitions)
;;  (parse-type-list type nil #'+ #'(lambda (a) (declare (ignore a)) 1)
;;		   #'(lambda (a b) (declare (ignore a)) b) 0 definitions))

(defun expand-type (type definitions)
  (parse-type-exp type nil #'cons #'(lambda (a) a) 
		  #'cons nil definitions))

(defun expand-type-allow (type definitions)
  (let ((*allow-type-vars* t))
    (declare (special *allow-type-vars*))
    (parse-type-exp type nil #'cons #'(lambda (a) a) 
		    #'cons nil definitions)))

(defun andfunc (a b) (or a b))

(defun check-type-list (full-type definitions)
  (flet ((check-var-type (type) 
 	   (if (and (listp type) (= (length type) 2))
	       (if (variable-type? (second type) definitions)
		   (cons (first type) t)
		 (nesl-error "The type ~a is not a valid type-class."
			      (second type)))
	     (nesl-error "The typebinding ~a should be of the form: ~
                          (type-var type-class)."
			  type))))
	(parse-type-list 
	 (car full-type) (mapcar #'check-var-type (cdr full-type))
	 #'andfunc #'(lambda (a) (declare (ignore a)) t)
	 #'(lambda (a b) (declare (ignore a)) b) nil definitions)))

(defun intern-type (type)
  (cond ((symbolp type)
	 (if (vprefixp type)
	     (list 'nesl::vector (intern-type (vpostfix type)))
	   type))
	((listp type)
	 (cons (car type) (intern-type-list (cdr type))))
	(t
	 (nesl-error "The value ~a is an invalid type." type))))

(defun intern-type-list (type-list)
  (cond ((not type-list) nil)
	((eql (car type-list) 'v.)
	 (cons (list 'nesl::vector (intern-type (second type-list)))
	       (intern-type-list (cddr type-list))))
	((and (symbolp (car type-list)) (vprefixp (car type-list)))
	 (let ((foo (intern-type-list (cons (vpostfix (car type-list))
					    (cdr type-list)))))
	   (cons (list 'nesl::vector (car foo)) (cdr foo))))
	(t
	 (cons (intern-type (car type-list))
	       (intern-type-list (cdr type-list))))))

(defun simplify-types-internal (types)
  (if (listp types)
      (let* ((par-types (if (member '<- types) (list types) types)))
	(if (listp (car par-types))
	    (let ((long-types (intern-type-list (car par-types))))
	      (if  (eql (second long-types) '<-)
		  (cons (cons (car long-types) (cddr long-types)) 
			(cdr par-types))
		nil))
	  nil))
    nil))

(defun simplify-type (types)
  (or (simplify-types-internal types)
      (nesl-error "Bad typespec: ~a.~%The syntax of a typespec is:~%~%  ~
                   (typeexp <- typeexp*) |~%  ~
                   ((typeexp <- typeexp*) (Ident typeclass)*).~%"
		  types)))

;;; This function pretty prints the type.
;;; It returns a character string.
(defun pretty-type-nesl (type)
  (cond ((atom type)
	 (string type))
	((and (listp type) (eql (first type) 'nesl::vector))
	 (concatenate 'string "V." (pretty-type-nesl (second type))))
	((and (listp type) (eql (first type) 'function))
	 (format nil "~a <-~{ ~a~}"
		 (pretty-type-nesl (second type))
		 (mapcar #'pretty-type-nesl (cddr type))))
	((listp type) 
	 (format nil "(~a~{ ~a~})" 
		 (car type) 
		 (mapcar #'pretty-type-nesl (cdr type))))
	(t (nesl-error "Invalid type ~a" type))))

(defun pretty-type-cnesl (type &optional parenth?)
  (cond ((atom type)
	 (string type))
	((listp type)
	 (cond ((eql (first type) 'nesl::vector)
		(format nil "[~a]" (pretty-type-cnesl (second type) t)))
	       ((eql (first type) 'function)
		(format nil "~a -> ~a" 
			(pretty-type-cnesl (third type))
			(pretty-type-cnesl (second type))))
	       ((eql (first type) 'nesl::pair)
		(format nil (if parenth? "(~a,~a)" "~a,~a")
			(pretty-type-cnesl (second type) t)
			(pretty-type-cnesl (third type))))
	       ((eql (length type) 1)
		(format nil "~a" (car type)))
	       (t (format nil "~a(~a)" 
			  (car type) 
			  (pretty-type-cnesl (second type))))))
	(t (nesl-error "Invalid type ~a" type))))

(defun pretty-type (type)
  (if *cnesl-syntax* (pretty-type-cnesl type) (pretty-type-nesl type)))

(defun pretty-type-list (type-list)
  (let ((subtypes (mapcar #'pretty-type type-list)))
    (if *cnesl-syntax*
	(format nil "~a~{,~a~}" (car subtypes) (cdr subtypes))
      subtypes)))

(defun pretty-type-type-bind (type-bind)
  (format nil "~a IN ~a" (first type-bind) (second type-bind)))

(defun pretty-type-type-binds (type-binds)
  (if type-binds
      (if *cnesl-syntax*
	  (let ((binds (mapcar #'pretty-type-type-bind type-binds)))
	    (format nil ":: ~a~{; ~a~}" (car binds) (cdr binds)))
	(format nil "~{~a ~}" type-binds))
    ""))

(defun pretty-type-full (type)
  (format nil "~a ~a" 
	  (pretty-type (car type)) 
	  (pretty-type-type-binds (cdr type))))

(defun pretty-type-list-full (type)
  (format nil "~a ~a" 
	  (pretty-type-list (car type)) 
	  (pretty-type-type-binds (cdr type))))

(defun pretty-type-func (type)
  (format nil "~a ~a" 
	  (pretty-type (cons 'function (car type)))
	  (pretty-type-type-binds (cdr type))))


;;; ******************
;;; THe following is a special case of code given above.
;;; It was inserted for efficiency reasons
;;; ****************

(defun l-from-type (type definitions)
  (cond ((null type) 0)
	((and (atom type) (primitive-type? type definitions)) 1)
	((listp type)
	 (cond ((eql (car type) 'nesl::pair)
		(+ (l-from-type (second type) definitions)
		   (l-from-type (third type) definitions)))
	       ((eql (car type) 'nesl::vector)
		(+ 1 (l-from-type (second type) definitions)))
	       (t (nesl-error "Bad type ~a." type))))
	(t (nesl-error "No base-type definition for ~a." type))))

